/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.helper;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.userpref.UserPreferencesException;
import org.ametys.core.userpref.UserPreferencesManager;
import org.ametys.core.util.JSONUtils;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * The pilotage MCC sessions helper for user personal templates.
 */
public class PilotageMCCSessionsTemplateHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** The avalon role */
    public static final String ROLE = PilotageMCCSessionsTemplateHelper.class.getName();
    
    /** The MCC sessions templates context for user preference */
    private static final String __MCC_SESSIONS_TEMPLATES_USER_PREF_CONTEXT = "/mcc-sessions-templates";
    
    /** The MCC sessions templates id for user preference */
    private static final String __MCC_SESSIONS_TEMPLATES_USER_PREF_ID = "mcc-sessions";
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The user preferences manager */
    protected UserPreferencesManager _userPrefManager;
    
    /** The JSON utils */
    protected JSONUtils _jsonUtils;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _userPrefManager = (UserPreferencesManager) manager.lookup(UserPreferencesManager.ROLE);
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
    }
    
    /**
     * Create a template for the given MCC sessions
     * @param templateName the template name
     * @param mccSessions the MCC sessions
     * @return the map of results
     */
    @Callable(rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> createMCCSessionsTemplate(String templateName, List<Object> mccSessions)
    {
        Map<String, Object> results = new HashMap<>();
        UserIdentity user = _currentUserProvider.getUser();
        
        try
        {
            String sessionsAsJSON = _userPrefManager.getUserPreferenceAsString(user, __MCC_SESSIONS_TEMPLATES_USER_PREF_CONTEXT, Map.of(), __MCC_SESSIONS_TEMPLATES_USER_PREF_ID);
            _userPrefManager.addUserPreference(user, __MCC_SESSIONS_TEMPLATES_USER_PREF_CONTEXT, Map.of(), __MCC_SESSIONS_TEMPLATES_USER_PREF_ID, _addSessions(sessionsAsJSON, templateName, mccSessions));
        }
        catch (UserPreferencesException e)
        {
            getLogger().error("An error occurred setting user preferences for MCC sessions template name '{}'", templateName, e);
            results.put("error", true);
        }
        
        return results;
    }
    
    /**
     * Get all MCC sessions template of the current user
     * @param parameters the parameters
     * @return the map of results with templates title
     */
    @Callable(rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> getUserMCCSessionsTemplates(Map<String, Object> parameters)
    {
        Map<String, Object> results = new HashMap<>();
        UserIdentity user = _currentUserProvider.getUser();
        
        try
        {
            String templateUserPrefAsString = _userPrefManager.getUserPreferenceAsString(user, __MCC_SESSIONS_TEMPLATES_USER_PREF_CONTEXT, Map.of(), __MCC_SESSIONS_TEMPLATES_USER_PREF_ID);
            Map<String, Object> templateUserPrefs = _jsonUtils.convertJsonToMap(templateUserPrefAsString);
            List<Map<String, String>> templates = templateUserPrefs.keySet()
                .stream()
                .map(t -> Map.of("title", t))
                .toList();
            results.put("templates", templates);
        }
        catch (UserPreferencesException e)
        {
            getLogger().error("An error occurred getting user preferences for MCC sessions", e);
            results.put("error", true);
        }
        
        return results;
    }
    
    /**
     * Delete the MCC sessions template
     * @param templateName the template name
     * @return the map of results
     */
    @Callable(rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> deleteMCCSessionsTemplate(String templateName)
    {
        Map<String, Object> results = new HashMap<>();
        UserIdentity user = _currentUserProvider.getUser();
        
        try
        {
            String sessionsAsJSON = _userPrefManager.getUserPreferenceAsString(user, __MCC_SESSIONS_TEMPLATES_USER_PREF_CONTEXT, Map.of(), __MCC_SESSIONS_TEMPLATES_USER_PREF_ID);
            _userPrefManager.addUserPreference(user, __MCC_SESSIONS_TEMPLATES_USER_PREF_CONTEXT, Map.of(), __MCC_SESSIONS_TEMPLATES_USER_PREF_ID, _deleteSessions(sessionsAsJSON, templateName));
        }
        catch (UserPreferencesException e)
        {
            getLogger().error("An error occurred deleting user preferences for MCC sessions template name '{}'", templateName, e);
            results.put("error", true);
        }
        
        return results;
    }
    
    /**
     * Rename the MCC sessions template
     * @param oldTemplateName the old template name
     * @param newTemplateName the new template name
     * @return the map of results
     */
    @Callable(rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> renameMCCSessionsTemplate(String oldTemplateName, String newTemplateName)
    {
        Map<String, Object> results = new HashMap<>();
        UserIdentity user = _currentUserProvider.getUser();
        
        try
        {
            String sessionsAsJSON = _userPrefManager.getUserPreferenceAsString(user, __MCC_SESSIONS_TEMPLATES_USER_PREF_CONTEXT, Map.of(), __MCC_SESSIONS_TEMPLATES_USER_PREF_ID);
            _userPrefManager.addUserPreference(user, __MCC_SESSIONS_TEMPLATES_USER_PREF_CONTEXT, Map.of(), __MCC_SESSIONS_TEMPLATES_USER_PREF_ID, _renameSessions(sessionsAsJSON, oldTemplateName, newTemplateName));
        }
        catch (UserPreferencesException e)
        {
            getLogger().error("An error occurred renaming user preferences for MCC sessions template name '{}'", oldTemplateName, e);
            results.put("error", true);
        }
        
        return results;
    }
    
    /**
     * Rename the MCC sessions for a given template
     * @param templateName the template name
     * @return the map of results
     */
    @Callable(rights = Callable.NO_CHECK_REQUIRED)
    public Map<String, Object> getMCCSessionsTemplate(String templateName)
    {
        Map<String, Object> results = new HashMap<>();
        UserIdentity user = _currentUserProvider.getUser();
        
        try
        {
            String sessionsAsJSON = _userPrefManager.getUserPreferenceAsString(user, __MCC_SESSIONS_TEMPLATES_USER_PREF_CONTEXT, Map.of(), __MCC_SESSIONS_TEMPLATES_USER_PREF_ID);
            Map<String, Object> allSessions = _jsonUtils.convertJsonToMap(sessionsAsJSON);
            results.put("sessions", allSessions.get(templateName));
        }
        catch (UserPreferencesException e)
        {
            getLogger().error("An error occurred getting user preferences for MCC sessions template name '{}'", templateName, e);
            results.put("error", true);
        }
        
        return results;
    }
    
    private String _addSessions(String allSessionsAsJSON, String templateName, List<Object> mccSessions)
    {
        Map<String, Object> allSessions = _jsonUtils.convertJsonToMap(allSessionsAsJSON);
        allSessions.put(templateName, mccSessions);
        return _jsonUtils.convertObjectToJson(allSessions);
    }
    
    private String _renameSessions(String allSessionsAsJSON, String oldTemplateName, String newTemplateName)
    {
        Map<String, Object> allSessions = _jsonUtils.convertJsonToMap(allSessionsAsJSON);
        allSessions.put(newTemplateName, allSessions.get(oldTemplateName));
        allSessions.remove(oldTemplateName);
        return _jsonUtils.convertObjectToJson(allSessions);
    }
    
    private String _deleteSessions(String allSessionsAsJSON, String templateName)
    {
        Map<String, Object> allSessions = _jsonUtils.convertJsonToMap(allSessionsAsJSON);
        allSessions.remove(templateName);
        return _jsonUtils.convertObjectToJson(allSessions);
    }
}
