/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.consistency.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.courselist.CourseList.ChoiceType;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.plugins.odfpilotage.report.consistency.AbstractConsistencyAnalysis;
import org.ametys.plugins.odfpilotage.report.consistency.ConsistencyAnalysisResult;
import org.ametys.plugins.odfpilotage.report.consistency.ConsistencyAnalysisStatus;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Analysis on ECTS sums at each level of the {@link Program}.
 */
public class ECTSSumAnalysis extends AbstractConsistencyAnalysis
{
    private static final Pattern __ECTS_PATTERN = Pattern.compile("^.*?([0-9]+(?:[\\.,][0-9]+)?).*$");
    
    @Override
    public ConsistencyAnalysisResult analyze(ProgramItem programItem)
    {
        ConsistencyAnalysisResult result = new ConsistencyAnalysisResult();
        
        // Introduction
        result.setIntroText(new I18nizableText("plugin." + _pluginName, "PLUGINS_ODF_PILOTAGE_CONSISTENCY_ANALYSIS_ECTSSUM_INTRO"));
        
        // Columns
        result.addColumn("parentName", new I18nizableText("plugin." + _pluginName, "PLUGINS_ODF_PILOTAGE_CONSISTENCY_ANALYSIS_ECTSSUM_COLUMN_PARENT_NAME"), true);
        result.addColumn("parentCode", new I18nizableText("plugin." + _pluginName, "PLUGINS_ODF_PILOTAGE_CONSISTENCY_ANALYSIS_ECTSSUM_COLUMN_PARENT_CODE"));
        result.addColumn("parentECTS", new I18nizableText("plugin." + _pluginName, "PLUGINS_ODF_PILOTAGE_CONSISTENCY_ANALYSIS_ECTSSUM_COLUMN_PARENT_ECTS"));
        result.addColumn("childrenECTS", new I18nizableText("plugin." + _pluginName, "PLUGINS_ODF_PILOTAGE_CONSISTENCY_ANALYSIS_ECTSSUM_COLUMN_CHILDREN_ECTS"));
        
        // Errors
        List<Map<String, Object>> errors = _controlECTS(programItem, 0);
        result.addLines(errors);
        
        // Status
        result.setStatus(errors.isEmpty() ? ConsistencyAnalysisStatus.OK : ConsistencyAnalysisStatus.KO);
        
        return result;
    }
    
    /**
     * Get the {@link I18nizableText} representing the error message to display (with parameters).
     * @param programItem The program element which is inconsistency
     * @param parentECTS Number of ECTS on the parent
     * @param childrenECTS Number of ECTS on the children
     * @param level the level of the line to retrieve (hierarchical level begins at 0 for the program)
     * @return The error message
     */
    protected Map<String, Object> _getErrorMessage(ProgramItem programItem, Double parentECTS, Double childrenECTS, int level)
    {
        Map<String, Object> error = new HashMap<>();
        error.put("parentName", ((Content) programItem).getTitle());
        error.put("parentCode", programItem.getDisplayCode());
        error.put("parentECTS", parentECTS);
        error.put("childrenECTS", childrenECTS);
        error.put(ConsistencyAnalysisResult.INDENTATION_LEVEL_DATA_NAME, level);
        return error;
    }
    
    /**
     * Control the ECTS consistency between the current content and its children.
     * @param programItem The {@link ProgramItem} to check ECTS consistency
     * @param level the level of the program item to control (hierarchical level begins at 0 for the program)
     * @return The {@link List} of error messages
     */
    protected List<Map<String, Object>> _controlECTS(ProgramItem programItem, int level)
    {
        List<Map<String, Object>> errors = new ArrayList<>();
        
        // Children ECTS control (if there are children)
        List<ProgramItem> children = _odfHelper.getChildProgramItems(programItem);
        if (!children.isEmpty())
        {
            // Current content ECTS computation and control
            Double parentECTS = _getECTS((Content) programItem);
            // If there is no ECTS on the current content, there is no reason to control it.
            if (parentECTS != null)
            {
                Double childrenECTS = _getChildrenECTSSum(programItem);
                
                if (!parentECTS.equals(childrenECTS))
                {
                    errors.add(_getErrorMessage(programItem, parentECTS, childrenECTS, level));
                }
            }
            
            for (ProgramItem child : children)
            {
                errors.addAll(_controlECTS(child, level + 1));
            }
        }
        
        return errors;
    }
    
    /**
     * Get the sum of the children of the current item. It calculates the coef if it's a {@link CourseList}.
     * @param programItem The parent {@link ProgramItem}.
     * @return The sum of children ECTS
     */
    protected Double _getChildrenECTSSum(ProgramItem programItem)
    {
        // Ignore optional course list and avoid useless expensive calls
        if (programItem instanceof CourseList courseList && ChoiceType.OPTIONAL.equals(courseList.getType()))
        {
            return 0.0;
        }
        
        List<ProgramItem> children = _odfHelper.getChildProgramItems(programItem);

        Double coef = 1.0;
        Double childrenECTS = 0.0;
        if (!children.isEmpty())
        {
            if (programItem instanceof CourseList courseList)
            {
                switch (courseList.getType())
                {
                    case CHOICE:
                        // Apply the average of ECTS from children multiply by the minimum ELP to select
                        coef = ((double) courseList.getMinNumberOfCourses()) / children.size();
                        break;
                    case MANDATORY:
                    default:
                        // Add all ECTS from children
                        break;
                }
            }
            
            for (ProgramItem child : children)
            {
                Double childECTS = _getECTS((Content) child);
                if (childECTS == null)
                {
                    childECTS = _getChildrenECTSSum(child);
                }
                childrenECTS += childECTS;
            }
        }
        
        return coef * childrenECTS;
    }
    
    /**
     * Get the ECTS of the current {@link Content}. The method is different if it's a {@link Program} or a {@link Course} for example.
     * @param content The content
     * @return The value in the ECTS field
     */
    protected Double _getECTS(Content content)
    {
        // ECTS : program, subProgram (enumeration)
        // ECTS : container, course (enumeration)
        // Nothing on courseList
        
        if (!content.hasValue("ects"))
        {
            return null;
        }
        
        Object ects = content.getValue("ects");
        
        if (content instanceof Course || content instanceof Container)
        {
            return (Double) ects;
        }
        
        if (content instanceof AbstractProgram)
        {
            String code = new OdfReferenceTableEntry(((ContentValue) ects).getContent()).getCode();
            Matcher matcher = __ECTS_PATTERN.matcher(code);
            if (matcher.matches())
            {
                return Double.parseDouble(matcher.group(1).replace(",", "."));
            }
            return null;
        }

        return null;
    }
}
