/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.report.impl;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.CourseFactory;
import org.ametys.odf.courselist.CourseListFactory;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.ContainerFactory;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.SubProgramFactory;
import org.ametys.odf.skill.ODFSkillsHelper;

/**
 * Pilotage report for skills
 */
public class ElpCompetencesExtract extends AbstractMaquetteExtract
{
    @Override
    public void initialize() throws Exception
    {
        // Calling super is not necessary because we override the views anyway
        _courseView = getView(CourseFactory.COURSE_CONTENT_TYPE, "report-elpcompetences");
        _courseListView = getView(CourseListFactory.COURSE_LIST_CONTENT_TYPE, "report-elpcompetences");
        _containerView = getView(ContainerFactory.CONTAINER_CONTENT_TYPE, "report-elpcompetences");
        _subProgramView = getView(SubProgramFactory.SUBPROGRAM_CONTENT_TYPE, "report-elpcompetences");
        _programView = getView(ProgramFactory.PROGRAM_CONTENT_TYPE, "report-elpcompetences");
    }
    
    public String getType(Map<String, String> reportParameters, boolean shortName)
    {
        return "elpcompetences";
    }
    
    @Override
    public String getDefaultOutputFormat()
    {
        return OUTPUT_FORMAT_XLS;
    }
    
    @Override
    public Set<String> getSupportedOutputFormats()
    {
        return Set.of(OUTPUT_FORMAT_XLS);
    }
    
    @Override
    protected PilotageReportContent getReportContentForOrgUnit(String outputFormat, Map<String, String> reportParameters)
    {
        throw new UnsupportedOperationException("Impossible to launch the report '" + getType() + "' on an org unit.");
    }
    
    @Override
    public void saxProgramItem(ContentHandler handler, String programItemId, Map<String, String> reportParameters)
    {
        if (!ODFSkillsHelper.isSkillsEnabled())
        {
            throw new UnsupportedOperationException("Skills cannot be exported because the skills feature is disabled.");
        }
        
        super.saxProgramItem(handler, programItemId, reportParameters);
    }
    
    @Override
    protected void _saxOrgUnit(ContentHandler handler, String catalog, String lang, String orgUnitId, Map<String, String> reportParameters)
    {
        throw new UnsupportedOperationException("Impossible to launch the report '" + getType() + "' on an org unit.");
    }
    
    @Override
    protected ProgramItem getProgramItemIfValid(String programItemId)
    {
        ProgramItem programItem = _resolver.resolveById(programItemId);
        if (!(programItem instanceof AbstractProgram) && !(programItem instanceof Container))
        {
            throw new UnsupportedOperationException("The report '" + getType() + "' can be launch only on programs and containers.");
        }
        
        return programItem;
    }
    
    @Override
    protected Stream<? extends ProgramItem> _getProgramItemsFromProgram(Program program, Map<String, String> reportParameters)
    {
        // Create one file per year under the program
        return _odfHelper.getYears(program).stream();
    }
    
    @Override
    protected StringBuilder buildReportFileNamePrefix(ProgramItem programItem, Map<String, String> reportParameters)
    {
        StringBuilder filenamePrefix = super.buildReportFileNamePrefix(programItem, reportParameters);
        if (!(programItem instanceof Program) && reportParameters.containsKey(PARAMETER_PROGRAM))
        {
            Program parentProgram = _resolver.<Program>resolveById(reportParameters.get(PARAMETER_PROGRAM));
            
            filenamePrefix.append(parentProgram.getDisplayCode());
            filenamePrefix.append("-");
        }
        return filenamePrefix;
    }
    
    @Override
    protected void saxGlobalInformations(ContentHandler handler, ProgramItem programItem, Map<String, String> reportParameters) throws SAXException
    {
        super.saxGlobalInformations(handler, programItem, reportParameters);
        
        String programId = reportParameters.get(PARAMETER_PROGRAM);
        Program program = _resolver.<Program>resolveById(programId);
        
        XMLUtils.createElement(handler, "programId", program.getId());
        
        // Retrieve all paths of the program item in the parent program
        List<EducationalPath> pathsInProgram = _odfHelper.getEducationalPaths(programItem, false, true)
                .stream()
                .filter(path -> path.getProgramItemIds().get(0).equals(programId))
                .toList();
        for (EducationalPath path : pathsInProgram)
        {
            XMLUtils.createElement(handler, "path", _odfHelper.getEducationalPathAsString(path));
        }
        
        _reportHelper.saxAcquiredMicroSkills(handler, programItem, program);
    }
}
