/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.odfpilotage.cost.entity;

import java.util.HashMap;
import java.util.Map;

/**
 * Class representing a number of hours for each nature and can contain an original value for overridden course parts.
 */
public class VolumesOfHours
{
    private Double _originalVolumeOfHours;
    
    /** Volume of hours by nature */
    private Map<String, Double> _volumeOfHoursByNature;
    
    /** The constructor 
     */
    public VolumesOfHours()
    {
        _volumeOfHoursByNature = new HashMap<>();
    }
    
    /** Get the volume of hours by nature 
     * @return a {@link Map} the volumes of hours by nature
     */
    public Map<String, Double> getVolumes()
    {
        return _volumeOfHoursByNature;
    }
    
    /**
     * Get the volume of hours for the given nature
     * @param natureId the nature identifier
     * @return the volume of hours for the given nature, <code>null</code> if not defined
     */
    public Double getVolume(String natureId)
    {
        return _volumeOfHoursByNature.get(natureId);
    }
    
    /**
     * Add an volume of hours for the given nature
     * @param natureId the nature of the volume
     * @param value the number of hours
     */
    public void addVolume(String natureId, Double value)
    {
        _volumeOfHoursByNature.put(natureId, value);
    }
    
    /**
     * Add an overridden volume of hours for the given nature and store the old value into the original volume of hours
     * @param natureId the nature of the volume
     * @param value the number of hours
     */
    public void addOverriddenVolume(String natureId, Double value)
    {
        _originalVolumeOfHours = _volumeOfHoursByNature.get(natureId);
        _volumeOfHoursByNature.put(natureId, value);
    }
    
    /**
     * Get the original volume of hours for this item (should be a course part only)
     * @return the original value if it has been overridden, otherwise <code>null</code>
     */
    public Double getOriginalVolumeOfHours()
    {
        return _originalVolumeOfHours;
    }
    
    /**
     * Sum the volume of hours with its weight on the current object
     * @param volumesOfHours The volumes to sum with the original object
     * @param weight the weight to apply on sub-elements
     */
    public void sum(VolumesOfHours volumesOfHours, Double weight)
    {
        Map<String, Double> volumeOfHoursByNature = volumesOfHours.getVolumes();
        for (String natureId : volumeOfHoursByNature.keySet())
        {
            _volumeOfHoursByNature.put(natureId, _volumeOfHoursByNature.getOrDefault(natureId,  0D) + volumeOfHoursByNature.get(natureId) * weight);
        }
    }
    
    /**
     * Get the total volume of hours of all natures
     * @return the total volume of hours
     */
    public Double getTotal()
    {
        return _volumeOfHoursByNature.values()
            .stream()
            .reduce(0D, Double::sum);
    }
}
