/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.manager;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.Constants;
import org.apache.cocoon.components.LifecycleHelper;
import org.apache.cocoon.environment.Context;
import org.apache.cocoon.util.log.SLF4JLoggerAdapter;
import org.apache.commons.lang.StringUtils;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.model.restrictions.RestrictedModelItem;
import org.ametys.cms.repository.Content;
import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.core.right.RightManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.contenttype.ODFContentTypeAttributeManager;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.plugins.core.impl.cache.AbstractCacheKey;
import org.ametys.plugins.odfpilotage.helper.MCCWorkflowHelper;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper;
import org.ametys.plugins.odfpilotage.helper.PilotageStatusHelper;
import org.ametys.plugins.odfpilotage.helper.PilotageStatusHelper.PilotageStatus;
import org.ametys.plugins.odfpilotage.rule.RulesManager;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ItemParserHelper.ConfigurationAndPluginName;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.ViewItem;
import org.ametys.runtime.model.ViewItemContainer;
import org.ametys.runtime.model.ViewItemGroup;
import org.ametys.runtime.model.ViewParser;

/**
 * The odf content type attribute manager for pilotage
 */
public class ODFPilotageContentTypeAttributeManager extends ODFContentTypeAttributeManager implements Configurable, Contextualizable, Serviceable, Initializable
{
    private static String _ATTRIBUTE_FILE_PATH = "/org/ametys/plugins/odfpilotage/manager/restrictions.xml";
    private static String _ATTRIBUTE_FILE_PATH_TO_OVERRIDE = "/WEB-INF/param/odf/pilotage/restrictions.xml";

    private static final String __PILOTAGE_STATUS_UNMODIFIABLE_ATTRIBUTES_CACHE_ID = ODFPilotageContentTypeAttributeManager.class.getName() + "$unmodifiableAttributes";
    
    private static final String __MCC_STATUS_UNMODIFIABLE_ATTRIBUTES_CACHE_ID = ODFPilotageContentTypeAttributeManager.class.getName() + "$parentProgramsForRules";
    
    /** The Cocoon context */
    protected Context _cocoonContext;
    
    /** The pilotage status helper */
    protected PilotageStatusHelper _pilotageStatusHelper;
    /** The MCC workflow helper */
    protected MCCWorkflowHelper _mccWorkflowHelper;
    
    /** The ODF helper */
    protected PilotageHelper _pilotageHelper;
    
    /** The right manager */
    protected RightManager _rightManager;
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The map of disabled attributes for each content type and each pilotage status */
    protected Map<String, Map<StatusRestriction, List<String>>> _disabledAttributes;
    
    /** The cache manager */
    protected AbstractCacheManager _cacheManager;
    /** Content type extension point */
    protected ContentTypeExtensionPoint _contentTypeEP;
    private org.apache.avalon.framework.context.Context _context;
    private ServiceManager _manager;
    
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _manager = manager;
        _pilotageStatusHelper = (PilotageStatusHelper) manager.lookup(PilotageStatusHelper.ROLE);
        _mccWorkflowHelper = (MCCWorkflowHelper) manager.lookup(MCCWorkflowHelper.ROLE);
        _pilotageHelper = (PilotageHelper) manager.lookup(PilotageHelper.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _cacheManager = (AbstractCacheManager) manager.lookup(AbstractCacheManager.ROLE);
        _contentTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
    }
    
    @Override
    public void contextualize(org.apache.avalon.framework.context.Context context) throws ContextException
    {
        _context = context;
        _cocoonContext = (Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }
    
    public void initialize() throws Exception
    {
        _cacheManager.createRequestCache(__PILOTAGE_STATUS_UNMODIFIABLE_ATTRIBUTES_CACHE_ID,
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_PILOTAGE_STATUS_UNMODIFIABLE_ATTRIBUTES_LABEL"),
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_PILOTAGE_STATUS_UNMODIFIABLE_ATTRIBUTES_DESCRIPTION"),
                true);
        
        _cacheManager.createRequestCache(__MCC_STATUS_UNMODIFIABLE_ATTRIBUTES_CACHE_ID,
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_MCC_STATUS_UNMODIFIABLE_ATTRIBUTES_LABEL"),
                new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_PILOTAGE_CACHE_MCC_STATUS_UNMODIFIABLE_ATTRIBUTES_DESCRIPTION"),
                true);
    }
    
    /**
     * Enumeration for available restrictions on attributs
     *
     */
    public enum StatusRestriction
    {
        /** Restriction on mention validated (program) */
        MENTION_VALIDATED,
        /** Restriction on orgunit validated (program) */
        ORGUNIT_VALIDATED,
        /** Restriction on CFVU validated (program) */
        CFVU_VALIDATED,
        /** Restriction on MCC rules validated */
        RULES_VALIDATED,
        /** Restriction on MCC mention validated */
        MCC_VALIDATED,
        /** Restriction on MCC orgunit controlled */
        MCC_ORGUNIT_CONTROLLED,
        /** Restriction on MCC orgunit validated */
        MCC_ORGUNIT_VALIDATED,
        /** Restriction on MCC controlled */
        MCC_CFVU_CONTROLLED,
        /** Restriction on MCC validated */
        MCC_CFVU_VALIDATED,
        /** No restriction */
        NONE
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _disabledAttributes = new HashMap<>();
        
        try
        {
            File xml = new File(_cocoonContext.getRealPath(_ATTRIBUTE_FILE_PATH_TO_OVERRIDE));
            if (!xml.isFile())
            {
                try (InputStream is = getClass().getResourceAsStream(_ATTRIBUTE_FILE_PATH))
                {
                    Configuration cfg = new DefaultConfigurationBuilder().build(is);
                    _fillDisabledAttributesMap(cfg);
                }
            }
            else
            {
                try (InputStream is = new FileInputStream(xml))
                {
                    Configuration cfg = new DefaultConfigurationBuilder().build(is);
                    _fillDisabledAttributesMap(cfg);
                }
            }
        }
        catch (Exception e)
        {
            throw new ConfigurationException("Error while parsing restrictions.xml", e);
        }
    }
    
    /**
     * Fill the disabled metadata from the configuration
     * @param cfg the configuration
     * @throws ConfigurationException if an error occurred
     */
    protected void _fillDisabledAttributesMap(Configuration cfg) throws ConfigurationException
    {
        for (Configuration contentTypeConf : cfg.getChildren("content-type"))
        {
            String contentTypeId = contentTypeConf.getAttribute("id");
            ContentType contentType = _contentTypeEP.getExtension(contentTypeId);
            
            Map<StatusRestriction, List<String>> attributesByState = new HashMap<>();
            for (Configuration statusConf : contentTypeConf.getChildren("status"))
            {
                ViewParser parser = new ODFStatusRestrictionsViewParser(contentType);
                try
                {
                    LifecycleHelper.setupComponent(parser, new SLF4JLoggerAdapter(getLogger()), _context, _manager, null);
                    View view = parser.parseView(new ConfigurationAndPluginName(statusConf, StringUtils.EMPTY));
                    
                    List<String> attributes = _getAttributePathsFromView(view, StringUtils.EMPTY);
                    
                    String statusName = statusConf.getAttribute("name");
                    StatusRestriction statusRestriction = StringUtils.isNotBlank(statusName) ? StatusRestriction.valueOf(statusName) : StatusRestriction.NONE;

                    attributesByState.put(statusRestriction, attributes);
                }
                catch (Exception e)
                {
                    throw new ConfigurationException("Unable to parse ODF status restrictions for content type ", statusConf, e);
                }
                finally
                {
                    LifecycleHelper.dispose(parser);
                }
            }
            
            _disabledAttributes.put(contentTypeConf.getAttribute("id"), attributesByState);
        }
    }
    
    private List<String> _getAttributePathsFromView(ViewItemContainer view, String prefix)
    {
        List<String> attributePaths = new ArrayList<>();
        
        List<ViewItem> viewItems = view.getViewItems();
        for (ViewItem viewItem : viewItems)
        {
            attributePaths.add(prefix + viewItem.getName());
            if (viewItem instanceof ViewItemGroup group)
            {
                attributePaths.addAll(_getAttributePathsFromView(group, viewItem.getName() + ModelItem.ITEM_PATH_SEPARATOR));
            }
        }
        
        return attributePaths;
    }
    
    private Cache<CacheKey, List<String>> _getPilotageStatusUnmodifiableAttributesCache()
    {
        return _cacheManager.get(__PILOTAGE_STATUS_UNMODIFIABLE_ATTRIBUTES_CACHE_ID);
    }
    
    private Cache<CacheKey, List<String>> _getMCCStatusUnmodifiableAttributesCache()
    {
        return _cacheManager.get(__MCC_STATUS_UNMODIFIABLE_ATTRIBUTES_CACHE_ID);
    }
    
    @Override
    public boolean canWrite(Content content, RestrictedModelItem<Content> definition) throws AmetysRepositoryException
    {
        boolean canWrite = super.canWrite(content, definition);
        canWrite &= _canWriteStatusRestrictions(content, definition.getPath());
        canWrite &= _pilotageHelper.canWriteMccRestrictions(content, definition.getPath());

        return canWrite;
    }
    
    // Write restriction based on restrictions.xml file
    private boolean _canWriteStatusRestrictions(Content content, String attributePath)
    {
        if (content == null || !(content instanceof ProgramItem || content instanceof CoursePart))
        {
            // not concern by pilotage nor mcc status restrictions
            return true;
        }
        
        for (String type : content.getTypes())
        {
            if (_disabledAttributes.containsKey(type))
            {
                Map<StatusRestriction, List<String>> attributesRestrictions = _disabledAttributes.get(type);
                
                if (!_checkPilotageStatus(content, attributePath, attributesRestrictions))
                {
                    // Current pilotage status does not allow to edit this attribute
                    return false;
                }
                
                return _checkMCCStatus(content, attributePath, attributesRestrictions);
            }
        }
        
        return true;
    }
    
    
    private boolean _checkMCCStatus (Content content, String attributPath, Map<StatusRestriction, List<String>> attributesRestrictions)
    {
        UserIdentity user = _currentUserProvider.getUser();
        CacheKey cacheKey = CacheKey.of(content.getId(), user);
        
        List<String> unmodifiableAttributes = _getMCCStatusUnmodifiableAttributesCache().get(cacheKey,  __ -> _getUnmodifiableAttributesForMCCStatus(content, user, attributesRestrictions));
        return !unmodifiableAttributes.contains(attributPath);
    }
    
    private List<String> _getUnmodifiableAttributesForMCCStatus (Content content, UserIdentity user, Map<StatusRestriction, List<String>> attributesRestrictions)
    {
        List<String> mccRulesAttributes = RulesManager.isRulesEnabled() ? Optional.ofNullable(attributesRestrictions.get(StatusRestriction.RULES_VALIDATED)).orElse(List.of()) : List.of();
        List<String> mccAttributes = Optional.ofNullable(attributesRestrictions.get(StatusRestriction.MCC_VALIDATED)).orElse(List.of());
        List<String> mccOrgunitControlAttributes = Optional.ofNullable(attributesRestrictions.get(StatusRestriction.MCC_ORGUNIT_CONTROLLED)).orElse(List.of());
        List<String> mccOrgunitValidationAttributes = Optional.ofNullable(attributesRestrictions.get(StatusRestriction.MCC_ORGUNIT_VALIDATED)).orElse(List.of());
        List<String> mccCFVUControlAttributes = Optional.ofNullable(attributesRestrictions.get(StatusRestriction.MCC_CFVU_CONTROLLED)).orElse(List.of());
        List<String> mccCFVUValidationAttributes = Optional.ofNullable(attributesRestrictions.get(StatusRestriction.MCC_CFVU_VALIDATED)).orElse(List.of());
        
        List<String> unmodifiableAttributes = new ArrayList<>();
        
        Container parentContainerWithHigherMCCStatus = content instanceof ProgramItem
                ? _mccWorkflowHelper.getParentContainerWithHigherMCCStatus((ProgramItem) content)
                : _mccWorkflowHelper.getParentContainerWithHigherMCCStatus((CoursePart) content);
        
        Set<String> userRights = parentContainerWithHigherMCCStatus != null ? _rightManager.getUserRights(user, parentContainerWithHigherMCCStatus) : Set.of();
        
        if (parentContainerWithHigherMCCStatus != null && _mccWorkflowHelper.isMCCCFVUValidated(parentContainerWithHigherMCCStatus) && !userRights.contains(MCCWorkflowHelper.MCC_CFVU_VALIDATED_SUPER_RIGHT_ID))
        {
            // All MCC fields are unmodifiable
            unmodifiableAttributes.addAll(mccRulesAttributes);
            unmodifiableAttributes.addAll(mccAttributes);
            unmodifiableAttributes.addAll(mccOrgunitControlAttributes);
            unmodifiableAttributes.addAll(mccOrgunitValidationAttributes);
            unmodifiableAttributes.addAll(mccCFVUControlAttributes);
            unmodifiableAttributes.addAll(mccCFVUValidationAttributes);
        }
        else if (parentContainerWithHigherMCCStatus != null && _mccWorkflowHelper.isMCCCFVUControlled(parentContainerWithHigherMCCStatus) 
                    && !userRights.contains(MCCWorkflowHelper.MCC_CFVU_VALIDATED_SUPER_RIGHT_ID) 
                    && !userRights.contains(MCCWorkflowHelper.MCC_CFVU_CONTROLLED_SUPER_RIGHT_ID))
        {
            unmodifiableAttributes.addAll(mccRulesAttributes);
            unmodifiableAttributes.addAll(mccAttributes);
            unmodifiableAttributes.addAll(mccOrgunitControlAttributes);
            unmodifiableAttributes.addAll(mccOrgunitValidationAttributes);
            unmodifiableAttributes.addAll(mccCFVUControlAttributes);
        }
        else if (parentContainerWithHigherMCCStatus != null && _mccWorkflowHelper.isMCCOrgUnitValidated(parentContainerWithHigherMCCStatus)
                    && !userRights.contains(MCCWorkflowHelper.MCC_CFVU_VALIDATED_SUPER_RIGHT_ID) 
                    && !userRights.contains(MCCWorkflowHelper.MCC_CFVU_CONTROLLED_SUPER_RIGHT_ID)
                    && !userRights.contains(MCCWorkflowHelper.MCC_ORGUNIT_VALIDATED_SUPER_RIGHT_ID))
        {
            unmodifiableAttributes.addAll(mccRulesAttributes);
            unmodifiableAttributes.addAll(mccAttributes);
            unmodifiableAttributes.addAll(mccOrgunitControlAttributes);
            unmodifiableAttributes.addAll(mccOrgunitValidationAttributes);
        }
        else if (parentContainerWithHigherMCCStatus != null && _mccWorkflowHelper.isMCCOrgUnitControlled(parentContainerWithHigherMCCStatus)
                    && !userRights.contains(MCCWorkflowHelper.MCC_CFVU_VALIDATED_SUPER_RIGHT_ID) 
                    && !userRights.contains(MCCWorkflowHelper.MCC_CFVU_CONTROLLED_SUPER_RIGHT_ID)
                    && !userRights.contains(MCCWorkflowHelper.MCC_ORGUNIT_VALIDATED_SUPER_RIGHT_ID)
                    && !userRights.contains(MCCWorkflowHelper.MCC_ORGUNIT_CONTROLLED_SUPER_RIGHT_ID))
        {
            unmodifiableAttributes.addAll(mccRulesAttributes);
            unmodifiableAttributes.addAll(mccAttributes);
            unmodifiableAttributes.addAll(mccOrgunitControlAttributes);
        }
        else
        {
            unmodifiableAttributes.addAll(_getUnmodifiableAttributesForRules(content, user, mccRulesAttributes));
            unmodifiableAttributes.addAll(_getUnmodifiableAttributesForMCCFields(content, user, mccAttributes));
        }
        
        
        return unmodifiableAttributes;
    }
    
    private List<String> _getUnmodifiableAttributesForRules(Content content, UserIdentity user, List<String> mccRulesMentionAttributes)
    {
        List<String> unmodifiableAttributes = new ArrayList<>();
        
        Container parentContainerWithHigherMCCStatus = content instanceof ProgramItem
                ? _mccWorkflowHelper.getParentContainerWithHigherMCCStatusForRules((ProgramItem) content)
                : _mccWorkflowHelper.getParentContainerWithHigherMCCStatusForRules((CoursePart) content);
        
        if (parentContainerWithHigherMCCStatus != null && _mccWorkflowHelper.isRulesValidated(parentContainerWithHigherMCCStatus))
        {
            Set<String> userRights = _rightManager.getUserRights(user, parentContainerWithHigherMCCStatus);
            
            if (!userRights.contains(MCCWorkflowHelper.RULES_VALIDATED_SUPER_RIGHT_ID)
                    && !userRights.contains(MCCWorkflowHelper.MCC_CFVU_VALIDATED_SUPER_RIGHT_ID) 
                    && !userRights.contains(MCCWorkflowHelper.MCC_CFVU_CONTROLLED_SUPER_RIGHT_ID)
                    && !userRights.contains(MCCWorkflowHelper.MCC_ORGUNIT_VALIDATED_SUPER_RIGHT_ID)
                    && !userRights.contains(MCCWorkflowHelper.MCC_ORGUNIT_CONTROLLED_SUPER_RIGHT_ID))
            {
                unmodifiableAttributes.addAll(mccRulesMentionAttributes);
            }
        }
        
        return unmodifiableAttributes;
    }
    
    private List<String> _getUnmodifiableAttributesForMCCFields(Content content, UserIdentity user, List<String> mccMentionAttributes)
    {
        List<String> unmodifiableAttributes = new ArrayList<>();
        
        Container parentContainerWithHigherMCCStatus = content instanceof ProgramItem
                ? _mccWorkflowHelper.getParentContainerWithHigherMCCStatusForMCCFields((ProgramItem) content)
                : _mccWorkflowHelper.getParentContainerWithHigherMCCStatusForMCCFields((CoursePart) content);
        
        if (parentContainerWithHigherMCCStatus != null && _mccWorkflowHelper.isMCCValidated(parentContainerWithHigherMCCStatus))
        {
            Set<String> userRights = _rightManager.getUserRights(user, parentContainerWithHigherMCCStatus);
            
            if (!userRights.contains(MCCWorkflowHelper.MCC_VALIDATED_SUPER_RIGHT_ID)
                    && !userRights.contains(MCCWorkflowHelper.MCC_CFVU_VALIDATED_SUPER_RIGHT_ID) 
                    && !userRights.contains(MCCWorkflowHelper.MCC_CFVU_CONTROLLED_SUPER_RIGHT_ID)
                    && !userRights.contains(MCCWorkflowHelper.MCC_ORGUNIT_VALIDATED_SUPER_RIGHT_ID)
                    && !userRights.contains(MCCWorkflowHelper.MCC_ORGUNIT_CONTROLLED_SUPER_RIGHT_ID))
            {
                unmodifiableAttributes.addAll(mccMentionAttributes);
            }
        }
        
        return unmodifiableAttributes;
    }
    
    private boolean _checkPilotageStatus(Content content, String attributPath, Map<StatusRestriction, List<String>> attributesRestrictions)
    {
        UserIdentity user = _currentUserProvider.getUser();
        CacheKey cacheKey = CacheKey.of(content.getId(), user);
        
        List<String> unmodifiableAttributes = _getPilotageStatusUnmodifiableAttributesCache().get(cacheKey,  __ -> _getUnmodifiableAttributesForPilotageStatus(content, attributesRestrictions));
        return !unmodifiableAttributes.contains(attributPath);
    }
    
    private List<String> _getUnmodifiableAttributesForPilotageStatus(Content content, Map<StatusRestriction, List<String>> attributesRestrictions)
    {
        List<String> mentionsAttributes = Optional.ofNullable(attributesRestrictions.get(StatusRestriction.MENTION_VALIDATED)).orElse(List.of());
        List<String> orgunitAttributes = Optional.ofNullable(attributesRestrictions.get(StatusRestriction.ORGUNIT_VALIDATED)).orElse(List.of());
        List<String> cfvuAttributes = Optional.ofNullable(attributesRestrictions.get(StatusRestriction.CFVU_VALIDATED)).orElse(List.of());
        
        Program parentProgramWithHigherPilotageStatus = content instanceof ProgramItem
                ? _pilotageStatusHelper.getParentProgramWithHigherPilotageStatus((ProgramItem) content)
                : _pilotageStatusHelper.getParentProgramWithHigherPilotageStatus((CoursePart) content);
        
        List<String> unmodifiableAttributes = new ArrayList<>();
        
        if (parentProgramWithHigherPilotageStatus != null && !_pilotageStatusHelper.hasEditSuperRight(parentProgramWithHigherPilotageStatus))
        {
            PilotageStatus pilotageStatus = _pilotageStatusHelper.getPilotageStatus(parentProgramWithHigherPilotageStatus);
            switch (pilotageStatus)
            {
                case MENTION_VALIDATED:
                    unmodifiableAttributes.addAll(mentionsAttributes);
                    break;
                case ORGUNIT_VALIDATED:
                    unmodifiableAttributes.addAll(mentionsAttributes);
                    unmodifiableAttributes.addAll(orgunitAttributes);
                    break;
                case CFVU_VALIDATED:
                    unmodifiableAttributes.addAll(mentionsAttributes);
                    unmodifiableAttributes.addAll(orgunitAttributes);
                    unmodifiableAttributes.addAll(cfvuAttributes);
                    break;
                case NONE:
                default:
                    break;
            }
        }
        
        return unmodifiableAttributes; // no restriction regardless of pilotage status
    }
    
    static final class CacheKey extends AbstractCacheKey
    {
        private CacheKey(String contentId, UserIdentity user)
        {
            super(user, contentId);
        }

        static CacheKey of(String contentId, UserIdentity user)
        {
            return new CacheKey(contentId, user);
        }
    }
}
