/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfpilotage.tree;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.course.Course;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.program.Container;
import org.ametys.odf.tree.AbstractStaticODFTreeIndicator;
import org.ametys.odf.tree.ODFTreeIndicator;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper;
import org.ametys.plugins.odfpilotage.helper.PilotageHelper.StepHolderStatus;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;

/**
 * {@link ODFTreeIndicator} to indicates the step holder of a course
 *
 */
public class CourseStepHolderIndicator extends AbstractStaticODFTreeIndicator
{
    private AmetysObjectResolver _resolver;
    private ODFHelper _odfHelper;
    private PilotageHelper _pilotageHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
        _pilotageHelper = (PilotageHelper) smanager.lookup(PilotageHelper.ROLE);
    }
    
    public IndicatorData getIndicatorData(Content content)
    {
        if (content instanceof Course course)
        {
            String stepHolderId = Optional.of(course)
                    .map(c -> c.<ContentValue>getValue("etapePorteuse"))
                    .map(ContentValue::getContentId)
                    .orElse(null);
            
            if (stepHolderId != null)
            {
                try
                {
                    Container container = _resolver.resolveById(stepHolderId);
                    
                    boolean isPartOfStructure = _isPartOfStructure(course, container);
                    
                    Map<String, I18nizableTextParameter> i18nParams = Map.of("title", new I18nizableText(container.getTitle()),
                                                                             "code", new I18nizableText(container.getDisplayCode()));

                    if (isPartOfStructure)
                    {
                        return new IndicatorData(new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_CONTENTS_TREE_INDICATORS_STEP_HOLDING_COURSE", i18nParams), null, "ametysicon-arrow-right-out", Map.of("stepHolderId", stepHolderId));
                    }
                    else
                    {
                        return new IndicatorData(new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_CONTENTS_TREE_INDICATORS_STEP_HOLDING_COURSE_INVALID", i18nParams), null, "ametysicon-arrow-right-out red-color", Map.of("error", true));
                    }
                }
                catch (UnknownAmetysObjectException e)
                {
                    return new IndicatorData(new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_CONTENTS_TREE_INDICATORS_STEP_HOLDING_COURSE_UNKNOWN"), null, "ametysicon-arrow-right-out red-color", Map.of("error", true));
                }
            }
            else
            {
                StepHolderStatus stepHolderStatus = _pilotageHelper.getStepHolder(course).getLeft();
                if (stepHolderStatus.equals(StepHolderStatus.MULTIPLE))
                {
                    Set<Container> courseSteps = _pilotageHelper.getSteps(course);
                    
                    List<Map<String, Object>> availableStepHolders = new ArrayList<>();
                    for (Container container : courseSteps)
                    {
                        Map<String, I18nizableTextParameter> i18nParamsForStep = Map.of(
                            "title", new I18nizableText(container.getTitle()),
                            "code", new I18nizableText(container.getCode())
                        );
                        
                        availableStepHolders.add(Map.of(
                                "label", new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_CONTENTS_TREE_INDICATORS_MULTIPLE_STEP_HOLDERS_HOLDER_LABEL", i18nParamsForStep),
                                "id", container.getId()
                        ));
                    }
                    
                    return new IndicatorData(new I18nizableText("plugin.odf-pilotage", "PLUGINS_ODF_CONTENTS_TREE_INDICATORS_MULTIPLE_STEP_HOLDERS"), null, "ametysicon-arrow-right-out orange-color", Map.of("error", true, "availableStepHolders", availableStepHolders));
                }
            }
        }
        
        return null;
    }
    
    private boolean _isPartOfStructure(Course course, Container container)
    {
        List<EducationalPath> ancestorPaths = _odfHelper.getEducationalPaths(course);
        for (EducationalPath ancestorPath : ancestorPaths)
        {
            if (ancestorPath.resolveProgramItems(_resolver).anyMatch(p -> p.equals(container)))
            {
                return true;
            }
        }
        return false;
    }

}
