/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.apogee.scc.impl;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.program.Container;
import org.ametys.plugins.odfsync.apogee.scc.AbstractPreviousYearsSynchronizableContentsCollection;

/**
 * SCC for nbStudents fields for container contents.
 */
public class ContainerNbStudentsSynchronizableContentsCollection extends AbstractPreviousYearsSynchronizableContentsCollection
{
    /** The ODF reference table helper */
    protected OdfReferenceTableHelper _odfRefTableHelper;
    
    private Optional<String> _yearId = Optional.empty();
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _odfRefTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
    }
    
    @Override
    public String getIdField()
    {
        return ProgramItem.CODE;
    }
    
    @Override
    protected String getCurrentYearAttributeName()
    {
        return "numberOfStudentsCurrentYear";
    }
    
    @Override
    protected String getPrecedingYearAttributeName()
    {
        return "numberOfStudentsPrecedingYear";
    }
    
    @Override
    protected Stream<Content> getContents()
    {
        return super.getContents().filter(container -> isContainerOfTypeYear((Container) container));
    }
    
    @Override
    protected String getSyncCodeItemName()
    {
        return "apogeeSyncCodeNbStudents";
    }
    
    @Override
    protected List<Map<String, Object>> executeApogeeRequest(Map<String, Object> parameters)
    {
        return _apogeePreviousYearsFieldsDAO.getYearNbStudents(getDataSourceId(), getParameterValues(), parameters);
    }
    
    /**
     * Determine if the container nature equals to "annee"
     * @param container The container
     * @return <code>true</code> if the current container nature equals to "annee"
     * @deprecated For 5.0, use org.ametys.plugins.odfpilotage.helper.PilotageHelper#isContainerOfTypeYear(Container), this method will be removed.
     */
    @Deprecated
    private boolean isContainerOfTypeYear(Container container)
    {
        return getYearId()
                .map(id -> StringUtils.equals(id, container.getNature()))
                .orElse(false);
    }
    
    /**
     * Get the year container nature identifier.
     * @return an {@link Optional} of the year identifier
     * @deprecated For 5.0, use org.ametys.plugins.odfpilotage.helper.PilotageHelper#getYearId(Container), this method will be removed.
     */
    @Deprecated
    private synchronized Optional<String> getYearId()
    {
        if (_yearId.isEmpty())
        {
            _yearId = Optional.of(_odfRefTableHelper)
                .map(rth -> rth.getItemFromCode(OdfReferenceTableHelper.CONTAINER_NATURE, "annee"))
                .map(OdfReferenceTableEntry::getId)
                .filter(StringUtils::isNotBlank);
        }
        return _yearId;
    }
}
