/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.apogee.scc.impl;

import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;

import org.ametys.cms.data.ContentSynchronizationResult;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.odf.course.Course;
import org.ametys.odf.course.ShareableCourseHelper;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.odfsync.apogee.scc.AbstractApogeeSynchronizableContentsWithCatalogCollection;
import org.ametys.plugins.repository.data.holder.values.SynchronizationResult;

/**
 * SCC for course contents.
 */
public class CourseSynchronizableContentsCollection extends AbstractApogeeSynchronizableContentsWithCatalogCollection
{
    /** SCC model id */
    public static final String MODEL_ID = "org.ametys.plugins.odfsync.apogee.scc.course";
    
    /** The ODF reference table helper */
    protected OdfReferenceTableHelper _odfRefTableHelper;
    
    /** The shareable course helper */
    protected ShareableCourseHelper _shareableCourseHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _odfRefTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
        _shareableCourseHelper = (ShareableCourseHelper) manager.lookup(ShareableCourseHelper.ROLE);
    }
    
    @Override
    protected List<Map<String, Object>> _search(Map<String, Object> searchParameters, Logger logger)
    {
        return _convertBigDecimal(_apogeeDAO.searchCourses(getDataSourceId(), getParameterValues(), searchParameters));
    }

    @Override
    protected String getMappingName()
    {
        return "course";
    }
    
    @Override
    protected Map<String, Object> getAdditionalAttributeValues(String idValue, Content content, Map<String, Object> additionalParameters, boolean create, Logger logger)
    {
        Map<String, Object> additionalValues = super.getAdditionalAttributeValues(idValue, content, additionalParameters, create, logger);
        
        // Handle course parts children
        SynchronizableContentsCollection scc = _sccHelper.getSCCFromModelId(CoursePartSynchronizableContentsCollection.MODEL_ID);
        List<ModifiableContent> courseParts = create
                // Import mode
                ? _importChildren(idValue, scc, logger)
                // Synchronization mode
                : _synchronizeChildren(content, scc, Course.CHILD_COURSE_PARTS, logger);
        additionalValues.put(Course.CHILD_COURSE_PARTS, courseParts.toArray(new ModifiableContent[courseParts.size()]));
        
        return additionalValues;
    }
    
    @Override
    protected Pair<String, Object> getParentAttribute(ModifiableContent parent)
    {
        if (parent instanceof CourseList)
        {
            return Pair.of(Course.PARENT_COURSE_LISTS, new ModifiableContent[] {parent});
        }

        return super.getParentAttribute(parent);
    }
    
    @Override
    public ContentSynchronizationResult additionalImportOperations(ModifiableContent content, Map<String, Object> additionalParameters, Logger logger)
    {
        ContentSynchronizationResult result = super.additionalImportOperations(content, additionalParameters, logger);
        
        CourseList parentContent = getParentFromAdditionalParameters(additionalParameters)
                .map(CourseList.class::cast)
                .orElse(null);
        
        SynchronizationResult additionalResult = new SynchronizationResult();
        boolean hasChanges = _shareableCourseHelper.initializeShareableFields((Course) content, parentContent, UserPopulationDAO.SYSTEM_USER_IDENTITY, true);
        additionalResult.setHasChanged(hasChanges);
        
        result.aggregateResult(additionalResult);
        return result;
    }
    
    @Override
    protected String getChildrenSCCModelId()
    {
        return CourseListSynchronizableContentsCollection.MODEL_ID;
    }
    
    @Override
    protected String getChildrenAttributeName()
    {
        return Course.CHILD_COURSE_LISTS;
    }
}
