/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.apogee.ws.structure;

import java.rmi.RemoteException;
import java.util.List;

import org.ametys.cms.repository.Content;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramPart;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.odfsync.export.ExportReport;
import org.ametys.plugins.odfsync.export.ExportReport.ExportStatus;

import gouv.education.apogee.commun.client.ws.creationse.CreationSEMetierServiceInterface;

/**
 * The structure to export in Apogee the following program
 * <br>Program / SubProgram / Container (semester) / UE / ELP / ...
 * <br>into DIP-VDI / ETP-VET / ELP / LSE / ELP / ...
 */
public class ApogeeDefaultStructure extends AbstractApogeeStructure
{
    @Override
    public void checkProgram(Program program, ExportReport report)
    {
        // Check mandatory data for program
        checkMandatoryDataForContent(program, getDIPMandatoryData(program), report);
        checkMandatoryDataForContent(program, getVDIMandatoryData(program), report);
        
        // Check mandatory data for program orgUnits
        checkMandatoryDataForOrgunits(program, program.getOrgUnits(), getOrgUnitMandatoryDataForDIP(), report);
        
        // Check the program structure
        List<ProgramPart> programPartChildren = program.getProgramPartChildren();
        for (ProgramPart programPart : programPartChildren)
        {
            if (programPart instanceof SubProgram)
            {
                checkSubProgram((SubProgram) programPart, report);
            }
            else
            {
                // The structure is not handled by this export
                report.setStatus(ExportStatus.CONTENT_STRUCTURE_INVALID);
                break;
            }
        }
        
        if (programPartChildren.isEmpty())
        {
            // The structure is not handled by this export
            report.setStatus(ExportStatus.CONTENT_STRUCTURE_INVALID);
        }
    }
    
    @Override
    public void checkSubProgram(SubProgram subProgram, ExportReport report)
    {
        // Check mandatory data for subProgram
        checkMandatoryDataForContent(subProgram, getETPMandatoryData(subProgram), report);
        checkMandatoryDataForContent(subProgram, getVETMandatoryData(subProgram), report);
        
        // Check mandatory data for subProgram orgUnits
        checkMandatoryDataForOrgunits(subProgram, subProgram.getOrgUnits(), getOrgUnitMandatoryDataForETP(), report);
        
        // Check the subProgram structure
        List<ProgramPart> programPartChildren = subProgram.getProgramPartChildren();
        for (ProgramPart childProgramPart : programPartChildren)
        {
            if (childProgramPart instanceof Container)
            {
                Container containerChildProgramPart = (Container) childProgramPart;
                String childContainerNatureCode = getContainerNatureCode(containerChildProgramPart);
                
                checkContainerAsSemester(containerChildProgramPart, report, childContainerNatureCode);
            }
            else
            {
                // The structure is not handled by this export
                report.setStatus(ExportStatus.CONTENT_STRUCTURE_INVALID);
                break;
            }
        }
        
        if (programPartChildren.isEmpty())
        {
            // The structure is not handled by this export
            report.setStatus(ExportStatus.CONTENT_STRUCTURE_INVALID);
        }
    }
    
    @Override
    public void checkContainerAsYear(Container container, ExportReport report, String containerNatureCode)
    {
        throw new UnsupportedOperationException("No year container in this structure");
    }
    
    @Override
    public void createProgram(Program program, ExportReport report)
    {
        try
        {
            CreationSEMetierServiceInterface creationService = _apogeeWS.getCreationService();
            String codDIP = getCodeApogee(program);
            Long versionDIP = getVersionApogee(program);
            _apogeeWS.createDIP(program, null, codDIP, creationService);
            _apogeeWS.createVDI(program, null, codDIP, versionDIP, creationService);
            
            for (ProgramPart pp : program.getProgramPartChildren())
            {
                _createSubProgram((SubProgram) pp, program, creationService, report);
            }
        }
        catch (Exception e)
        {
            report.setStatus(ExportStatus.ERROR);
            getLogger().error("An error occurred exporting the program '{}' ({}) in Apogee", program.getTitle(), program.getId(), e);
        }
    }
    
    /**
     * Create a subProgram in Apogee
     * @param subProgram the subProgram to create
     * @param programParent the program parent in Apogee
     * @param creationService the service to create element in Apogee
     * @param report the Apogee export report
     * @throws RemoteException if an export error occurred
     */
    protected void _createSubProgram(SubProgram subProgram, Content programParent, CreationSEMetierServiceInterface creationService, ExportReport report) throws RemoteException
    {
        String codDIP = getCodeApogee(programParent);
        Long versionDIP = getVersionApogee(programParent);
        String codETP = getCodeApogee(subProgram);
        Long versionETP = getVersionApogee(subProgram);
        
        _apogeeWS.createETP(subProgram, null, codETP, creationService);
        _apogeeWS.createVET(subProgram, null, codETP, versionETP, creationService);
        
        _apogeeWS.createLinkDIPETP(codDIP, versionDIP, codETP, versionETP, creationService);
        
        for (ProgramPart pp : subProgram.getProgramPartChildren())
        {
            _createContainerAsELP((Container) pp, programParent, subProgram, creationService, report);
        }
    }
    
    /**
     * Create a container as ELP in Apogee
     * @param container the container to create
     * @param programParent the program parent in Apogee
     * @param parentSubProgram the parent year container
     * @param creationService the service to create element in Apogee
     * @param report the Apogee export report
     * @throws RemoteException if an export error occurred
     */
    protected void _createContainerAsELP(Container container, Content programParent, Content parentSubProgram, CreationSEMetierServiceInterface creationService, ExportReport report) throws RemoteException
    {
        String codETP = getCodeApogee(parentSubProgram);
        Long versionETP = getVersionApogee(parentSubProgram);
        String codELP = getCodeApogee(container);
        
        // Create the ELP from the container
        _apogeeWS.createELP(container, null, codELP, creationService);
        
        // Create a mandatory LSE with random code
        String codLSE = org.ametys.core.util.StringUtils.generateKey();
        _apogeeWS.createMandatoryLSE("LSE - " + parentSubProgram.getTitle(), codLSE, codELP, creationService);
        
        // Create the link between ETP and LSE (year container and semester container)
        _apogeeWS.createLinkETPELPLSE(codETP, versionETP, codLSE, null, null, null, null, creationService);
        
        for (ProgramPart pp : container.getProgramPartChildren())
        {
            _createCourseList((CourseList) pp, container, creationService, report);
        }
    }
}
