/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.apogee.scc.impl;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Stream;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;

import org.ametys.cms.repository.ModifiableContent;
import org.ametys.odf.course.Course;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.odfsync.apogee.scc.AbstractApogeeSynchronizableContentsWithCatalogCollection;

/**
 * SCC for course part contents.
 */
public class CoursePartSynchronizableContentsCollection extends AbstractApogeeSynchronizableContentsWithCatalogCollection
{
    /** SCC model id */
    public static final String MODEL_ID = "org.ametys.plugins.odfsync.apogee.scc.coursepart";
    
    @Override
    protected List<Map<String, Object>> _search(Map<String, Object> searchParameters, Logger logger)
    {
        return _convertBigDecimal(_apogeeDAO.searchCourseParts(getDataSourceId(), getParameterValues(), searchParameters));
    }

    @Override
    protected String getMappingName()
    {
        return "coursepart";
    }

    @Override
    protected String getChildrenAttributeName()
    {
        return null;
    }
    
    @Override
    protected Pair<String, Object> getParentAttribute(ModifiableContent parent)
    {
        if (parent instanceof Course)
        {
            return Pair.of(CoursePart.PARENT_COURSES, new ModifiableContent[] {parent});
        }

        return super.getParentAttribute(parent);
    }
    
    @Override
    protected Map<String, Map<String, List<Object>>> getTransformedRemoteValues(Map<String, Object> searchParameters, Logger logger)
    {
        Map<String, Map<String, List<Object>>> remoteValuesByContent = super.getTransformedRemoteValues(searchParameters, logger);
        
        SynchronizableContentsCollection courseSCC = _sccHelper.getSCCFromModelId(CourseSynchronizableContentsCollection.MODEL_ID);
        String lang = _apogeeSCCHelper.getSynchronizationLang();

        // Search and add the course holder to each value
        remoteValuesByContent.values()
            .stream()
            .forEach(remoteValues -> _transformCourseHolder(remoteValues, courseSCC, lang));
        
        return remoteValuesByContent;
    }
    
    private void _transformCourseHolder(Map<String, List<Object>> remoteValues, SynchronizableContentsCollection courseSCC, String lang)
    {
        // Get the course holder sync code from remote values
        Optional.ofNullable(remoteValues.get(CoursePart.COURSE_HOLDER))
            .map(List::stream)
            .orElseGet(Stream::of)
            .map(String.class::cast)
            .filter(StringUtils::isNotBlank)
            // Get the course corresponding to the sync code
            .map(courseHolderSyncCode -> courseSCC.getContent(lang, courseHolderSyncCode, false))
            // Check if it exists
            .filter(Objects::nonNull)
            .findFirst()
            .map(ModifiableContent::getId)
            .ifPresentOrElse(
                // Set the course holder value
                courseHolderId -> remoteValues.put(CoursePart.COURSE_HOLDER, List.of(courseHolderId)),
                // Remove the value if no course holder found
                () -> remoteValues.remove(CoursePart.COURSE_HOLDER)
            );
    }
}
