/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.apogee;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.ibatis.mapping.Environment;
import org.apache.ibatis.session.SqlSession;
import org.apache.ibatis.type.JdbcType;
import org.apache.ibatis.type.TypeHandlerRegistry;

import org.ametys.core.datasource.AbstractMyBatisDAO;
import org.ametys.core.datasource.ZonedDateTimeTypeHandler;
import org.ametys.plugins.odfsync.apogee.scc.AbstractApogeeSynchronizableContentsCollection;

/**
 * Data access object to the remote DB Apogee
 */
public abstract class AbstractApogeeDAO extends AbstractMyBatisDAO
{
    /** The name of request attribute for the datasource id */
    protected static final String DATASOURCE_REQUEST_ATTR = AbstractApogeeDAO.class.getName() + "$dataSourceId";
    
    private Context _context;
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        super.contextualize(context);
        _context = context;
    }

    @Override
    protected org.apache.ibatis.session.Configuration _getMyBatisConfiguration(Environment env)
    {
        org.apache.ibatis.session.Configuration config = super._getMyBatisConfiguration(env);
        TypeHandlerRegistry typeHandlerRegistry = config.getTypeHandlerRegistry();
        
        // Transform date types to ZonedDateTime
        typeHandlerRegistry.register(Object.class, JdbcType.DATE, ZonedDateTimeTypeHandler.class);
        typeHandlerRegistry.register(Object.class, JdbcType.TIMESTAMP, ZonedDateTimeTypeHandler.class);
        
        return config;
    }
    
    @Override
    protected void _configureDatasource(Configuration configuration) throws ConfigurationException
    {
        // Do nothing
    }
    
    @Override
    protected String _getDataSourceId()
    {
        // Get data source id from request attributes
        return (String) _getRequest().getAttribute(DATASOURCE_REQUEST_ATTR);
    }
    
    /**
     * Get the request
     * @return the request
     */
    protected Request _getRequest()
    {
        Request request = ContextHelper.getRequest(_context);
        if (request == null)
        {
            throw new IllegalStateException("Apogee DAO can not be used outside a request");
        }
        return request;
    }
    
    /**
     * Returns the Apogee objects matching the search criteria on the asked statement.
     * @param dataSourceId The id of data source. Can not be null.
     * @param sccParams the parameters values of the Apogee synchronizable collection
     * @param params The parameters for search criteria
     * @param query The query identifier to execute
     * @return the list Apogee objects matching the search criteria
     */
    protected List<Map<String, Object>> _executeSearch(String dataSourceId, Map<String, Object> sccParams, Map<String, Object> params, String query)
    {
        Map<String, Object> sqlParams = new HashMap<>(params);
        _prepareSqlRequest(dataSourceId, sccParams, sqlParams);
        
        try (SqlSession session = getSession())
        {
            return session.selectList(query, sqlParams);
        }
    }
    
    /**
     * Common method to prepare the SQL request to execute
     * @param dataSourceId The id of data source. Can not be null.
     * @param sccParams the parameters values of the used Apogee synchronizable collection
     * @param params the SQL parameters
     */
    protected void _prepareSqlRequest(String dataSourceId, Map<String, Object> sccParams, Map<String, Object> params)
    {
        // Set data source id in request attributes
        _getRequest().setAttribute(DATASOURCE_REQUEST_ATTR, dataSourceId);
        
        // Add administrative year to SQL parameters if exists
        if (sccParams.containsKey(AbstractApogeeSynchronizableContentsCollection.PARAM_YEAR))
        {
            params.put("year", sccParams.get(AbstractApogeeSynchronizableContentsCollection.PARAM_YEAR));
        }
    }
}
