/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.apogee;

import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.quartz.JobDetail;
import org.quartz.JobExecutionContext;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.core.schedule.Schedulable;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.schedule.progression.SimpleProgressionTracker;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.catalog.Catalog;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.cdmfr.CDMFRHandler;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.plugins.core.impl.schedule.AbstractStaticSchedulable;
import org.ametys.plugins.core.schedule.Scheduler;
import org.ametys.plugins.odfsync.apogee.scc.ApogeeSynchronizableContentsCollectionHelper;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * A {@link Schedulable} job which synchronizes Apogee collections.
 */
public class ApogeeSchedulable extends AbstractStaticSchedulable
{
    private ODFHelper _odfHelper;
    private ApogeeSynchronizableContentsCollectionHelper _apogeeSCCHelper;
    private CatalogsManager _catalogsManager;
    private CDMFRHandler _cdmfrHandler;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _apogeeSCCHelper = (ApogeeSynchronizableContentsCollectionHelper) manager.lookup(ApogeeSynchronizableContentsCollectionHelper.ROLE);
        _catalogsManager = (CatalogsManager) manager.lookup(CatalogsManager.ROLE);
        _cdmfrHandler = (CDMFRHandler) manager.lookup(CDMFRHandler.ROLE);
    }
    
    @Override
    public void execute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        long begin = System.currentTimeMillis();
        Request request = ContextHelper.getRequest(_context);
        
        try
        {
            _cdmfrHandler.suspendCDMFRObserver();
            
            // Save handle contents
            request.setAttribute(ApogeeSynchronizableContentsCollectionHelper.HANDLED_CONTENTS, new HashSet<>());
            
            synchronizeCollections(context, progressionTracker);
        }
        catch (Exception ex)
        {
            getLogger().error("The global synchronization have failed.", ex);
        }
        finally
        {
            @SuppressWarnings("unchecked")
            Set<String> handledContents = (Set<String>) request.getAttribute(ApogeeSynchronizableContentsCollectionHelper.HANDLED_CONTENTS);
            _cdmfrHandler.unsuspendCDMFRObserver(handledContents);
            request.removeAttribute(ApogeeSynchronizableContentsCollectionHelper.HANDLED_CONTENTS);
            
            if (getLogger().isInfoEnabled())
            {
                getLogger().info("Global synchronization ended in {} ms", System.currentTimeMillis() - begin);
            }
        }
    }
    
    /**
     * Synchronize all contents with Apogee SCC starting from programs.
     * @param context The execution context
     * @param progressionTracker The progression tracker
     */
    protected void synchronizeCollections(JobExecutionContext context, ContainerProgressionTracker progressionTracker)
    {
        String catalog = Optional.of(context)
            .map(JobExecutionContext::getJobDetail)
            .map(JobDetail::getJobDataMap)
            .map(map -> map.getString(Scheduler.PARAM_VALUES_PREFIX + "catalog"))
            .map(_catalogsManager::getCatalog)
            .map(Catalog::getName)
            .orElseGet(_catalogsManager::getDefaultCatalogName);
        
        AmetysObjectIterable<Content> programs = _odfHelper.getProgramItems(ProgramFactory.PROGRAM_CONTENT_TYPE, null, catalog, null);
        
        for (Content program : programs)
        {
            progressionTracker.addSimpleStep(program.getId(), new I18nizableText("plugin.odf-sync", "PLUGINS_ODF_SYNC_GLOBAL_SYNCHRONIZATION_SCHEDULER_PROGRAM_SYNC_STEP_LABEL", List.of(program.getTitle())));
        }
        
        for (Content program : programs)
        {
            _apogeeSCCHelper.synchronizeContent((ModifiableContent) program, getLogger());
            ((SimpleProgressionTracker) progressionTracker.getStep(program.getId())).increment();
        }
    }
}
