/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.scc.operator;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;

import org.ametys.cms.contenttype.ContentAttributeDefinition;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.data.type.ModelItemTypeConstants;
import org.ametys.plugins.contentio.synchronize.impl.DefaultSynchronizingContentOperator;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;

/**
 * The abstract ODF synchronizing content operator. Each implementation can override this to add its helper role and specific mappings.
 */
public abstract class AbstractODFSynchronizingContentOperator extends DefaultSynchronizingContentOperator implements Serviceable
{
    /** The connector conversion helper */
    protected ODFSynchronizingContentOperatorHelper _odfSCCOperatorHelper;
    
    /** The content type extension point */
    protected ContentTypeExtensionPoint _contentTypeEP;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _odfSCCOperatorHelper = (ODFSynchronizingContentOperatorHelper) manager.lookup(getHelperRole());
        _contentTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
    }

    /**
     * Get the SCC operator helper for the current implementation.
     * @return a SCC operator helper role
     */
    protected abstract String getHelperRole();
    
    @Override
    public Map<String, List<Object>> transform(ContentType contentType, Map<String, List<Object>> remoteValues, Logger logger)
    {
        Map<String, List<Object>> result = new HashMap<>();
        
        for (String attributePath : remoteValues.keySet())
        {
            List<Object> values = remoteValues.get(attributePath);
            if (values != null)
            {
                List<Object> transformedValues = new ArrayList<>(values);
                
                if (contentType.hasModelItem(attributePath))
                {
                    ModelItem modelItem = contentType.getModelItem(attributePath);
                    transformedValues = _transformAttributeValues(modelItem, transformedValues, logger);
                }
                
                result.put(attributePath, transformedValues);
            }
        }

        return result;
    }
    
    /**
     * Transform the given attribute values
     * @param definition The definition of the attribute
     * @param values The values to transform
     * @param logger The logger
     * @return The transformed values
     */
    protected List<Object> _transformAttributeValues(ModelItem definition, List<Object> values, Logger logger)
    {
        if (definition instanceof ContentAttributeDefinition contentAttributeDef)
        {
            return _transformContentAttributeValues(contentAttributeDef, values, logger);
        }
        else if (ModelItemTypeConstants.RICH_TEXT_ELEMENT_TYPE_ID.equals(definition.getType().getId()))
        {
            return _transformRichTextAttributeValues((ElementDefinition) definition, values, logger);
        }
        
        return values;
    }
    
    /**
     * Transform the given values to content IDs, using Apogée conversion if needed.
     * @param definition The definition of the attribute
     * @param values The values to transform
     * @param logger The logger
     * @return The corresponding content IDs of the values
     */
    protected List<Object> _transformContentAttributeValues(ContentAttributeDefinition definition, List<Object> values, Logger logger)
    {
        if (_contentTypeEP.getExtension(definition.getContentTypeId()).isReferenceTable())
        {
            return values.stream()
                         .filter(String.class::isInstance)
                         .map(String.class::cast)
                         .map(v -> _getReferenceTableEntryId(definition, v, logger))
                         .filter(Objects::nonNull)
                         .distinct()
                         .collect(Collectors.toList());
        }
        
        return values;
    }
    
    /**
     * Get the id of content associated with the given attribute value
     * @param definition The definition of the attribute
     * @param value The attribute value
     * @param logger The logger
     * @return The id of content or null if no match found
     */
    protected String _getReferenceTableEntryId(ContentAttributeDefinition definition, String value, Logger logger)
    {
        String referenceTableEntryId = _odfSCCOperatorHelper.getReferenceTableEntryId(definition.getContentTypeId(), value);
        if (referenceTableEntryId == null)
        {
            logger.warn("The connector code '{}' doesn't have a corresponding Ametys value for attribute '{}'", value, definition.getPath());
        }
        
        return referenceTableEntryId;
    }
    
    /**
     * Transform the given values to concatenate them and get the final rich text value
     * @param definition The definition of the attribute
     * @param values The values to transform
     * @param logger The logger
     * @return The concatenated values
     */
    protected List<Object> _transformRichTextAttributeValues(ElementDefinition definition, List<Object> values, Logger logger)
    {
        String concatenated = values.stream()
                                  .filter(Objects::nonNull)
                                  .map(Object::toString)
                                  .filter(StringUtils::isNotBlank)
                                  .collect(Collectors.joining("\r\n"));
        
        return List.of(concatenated);
    }
    
    /**
     * Get the first value as a string.
     * @param values The list of values to iterate on
     * @return The first value as a string or null
     */
    protected String _getFirstValueAsString(List<Object> values)
    {
        return Optional
                // Manage null attributeValues
                .ofNullable(values)
                // Get an empty List
                .orElseGet(Collections::emptyList)
                // Stream it
                .stream()
                // Remove null values
                .filter(Objects::nonNull)
                // Object as string
                .map(Object::toString)
                // Remove empty values
                .filter(StringUtils::isNotEmpty)
                // Get the first element
                .findFirst()
                // Or null if there is not
                .orElse(null);
    }
}
