/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.cdmfr.extractor;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Optional;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.xpath.XPathAPI;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;

import org.ametys.cms.repository.Content;
import org.ametys.core.util.dom.DOMUtils;
import org.ametys.odf.courselist.CourseList;
import org.ametys.plugins.odfsync.cdmfr.ImportCDMFrContext;
import org.ametys.plugins.odfsync.cdmfr.components.ImportCDMFrComponent;
import org.ametys.plugins.odfsync.utils.ContentWorkflowDescription;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.ModelHelper;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelItemContainer;
import org.ametys.runtime.model.ModelViewItem;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.ViewItemContainer;

/**
 * This class provides methods to extract course list's values from a CMD-fr import document
 */
public class ImportCourseListValuesExtractor extends ImportCDMFrValuesExtractor
{
    /**
     * Creates an import course list values extractor
     * @param element the DOM element containing the XML values
     * @param factory the values extractor factory
     * @param component the import CDM-fr component
     * @param context the import context
     * @param models the model of the extracted values
     */
    public ImportCourseListValuesExtractor(Element element, ImportCDMFrValuesExtractorFactory factory, ImportCDMFrComponent component, ImportCDMFrContext context, Model... models)
    {
        this(element, factory, component, context, Arrays.asList(models));
    }
    
    /**
     * Creates an import course list values extractor
     * @param element the DOM element containing the XML values
     * @param factory the values extractor factory
     * @param component the import CDM-fr component
     * @param context the imported content's context
     * @param modelItemContainers the model of the extracted values
     */
    public ImportCourseListValuesExtractor(Element element, ImportCDMFrValuesExtractorFactory factory, ImportCDMFrComponent component, ImportCDMFrContext context, Collection<? extends ModelItemContainer> modelItemContainers)
    {
        super(element, factory, component, context, modelItemContainers);
    }
    
    @SuppressWarnings("unchecked")
    @Override
    protected void _fillViewItemContainerFromXML(Element element, ViewItemContainer viewItemContainer, Collection<? extends ModelItemContainer> modelItemContainers) throws Exception
    {
        NamedNodeMap attributes = element.getAttributes();
        for (int i = 0; i < attributes.getLength(); i++)
        {
            Node attribute = attributes.item(i);

            Optional<ModelItem> optionalModelItem = _getModelItemFromNodeName(element, attribute.getNodeName(), modelItemContainers);
            if (optionalModelItem.isPresent())
            {
                ModelItem modelItem = optionalModelItem.get();

                ModelViewItem modelViewItem = new ViewElement();
                modelViewItem.setDefinition(modelItem);

                viewItemContainer.addViewItem(modelViewItem);
            }
        }
        
        if (DOMUtils.hasChildElement(element, _MULTIPLE_DATA_ITEM_TAG))
        {
            ModelItem childCoursesModelItem = ModelHelper.getModelItem(CourseList.CHILD_COURSES, modelItemContainers);
            
            ModelViewItem modelViewItem = new ViewElement();
            modelViewItem.setDefinition(childCoursesModelItem);

            viewItemContainer.addViewItem(modelViewItem);
        }
    }

    @Override
    protected boolean _hasChildForAttribute(Element element, String attributeName)
    {
        return CourseList.CHILD_COURSES.equals(attributeName)
                ? DOMUtils.hasChildElement(element, _MULTIPLE_DATA_ITEM_TAG)
                : element.hasAttribute(attributeName);
    }
    
    @Override
    protected <T> Object _extractElementValue(Element parent, ElementDefinition<T> definition, Optional<Object> additionalData) throws Exception
    {
        String attributePath = definition.getPath();
        if (CourseList.CHILD_COURSES.equals(attributePath))
        {
            Document doc = _context.getDocument();
            String lang = _context.getLang();
            List<Content> courses = new ArrayList<>();
            List<Element> children = DOMUtils.getChildElementsByTagName(parent, _MULTIPLE_DATA_ITEM_TAG);
            for (Element child : children)
            {
                String courseSyncCode = child.getTextContent().trim();
                
                Element courseElement = (Element) XPathAPI.selectSingleNode(doc.getFirstChild(), "course[@CDMid = '" + courseSyncCode + "' and @language = '" + lang + "']");
                if (courseElement == null)
                {
                    courseElement = (Element) XPathAPI.selectSingleNode(doc.getFirstChild(), "course[@CDMid = '" + courseSyncCode + "']");
                }
                
                if (courseElement != null)
                {
                    String courseCatalog = _component.getCatalogName(courseElement);
                    if (courseCatalog != null && !courseCatalog.equals(_context.getCatalog()))
                    {
                        String cdmCode = DOMUtils.getChildElementByTagName(courseElement, "cdmCode").getTextContent();
                        _context.getLogger().error("The course '{}' belongs to a different catalog than the one from the imported/synchronized program : '{}' vs '{}'. No synchronization will be done on this course.", cdmCode, courseCatalog, _context.getCatalog());
                    }
                    else
                    {
                        ImportCDMFrContext courseContext = new ImportCDMFrContext(_context);
                        String elpLang = courseElement.getAttribute("language");
                        if (StringUtils.isNotEmpty(elpLang))
                        {
                            courseContext.setLang(elpLang);
                        }
                        
                        String title = DOMUtils.getChildElementByTagName(courseElement, "title").getTextContent();
                        Content course = _component.importOrSynchronizeContent(courseElement, ContentWorkflowDescription.COURSE_WF_DESCRIPTION, title, courseSyncCode, courseContext);
                        CollectionUtils.addIgnoreNull(courses, course);
                    }
                }
            }
            return courses.toArray(new Content[courses.size()]);
        }
        else
        {
            String attribute = parent.getAttribute(attributePath);
            if (StringUtils.isNotEmpty(attribute))
            {
                return CourseList.CHOICE_TYPE.equals(attributePath)
                        ? attribute.toUpperCase()
                        : definition.getType().castValue(attribute); 
            }
            else
            {
                return null;
            }
        }
    }
}
