/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.odfsync.generic.scc;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.xml.sax.SAXException;

import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Abstract class for reading mapping files in ODF helpers.
 * First look for a project mapping files in WEB-INF/param/odf
 * Then look for the default mapping files in plugin.odf-sync
 */
public abstract class AbstractMappingHelper extends AbstractLogEnabled implements Serviceable, Configurable
{
    /** The source resolver */
    protected SourceResolver _srcResolver;
    
    /** The implementation name */
    protected String _implementationName;
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _implementationName = configuration.getChild("name").getValue();
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _srcResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
    }
    
    /**
     * Read the mapping file to convert types.
     * @param baseUri The base uri of the file
     * @return The mapping
     */
    protected Map<String, String> _readMapping(String baseUri)
    {
        // First, search in WEB-INF/param/odf/[implementationName]
        // Then in the ODF-sync plugin conversions folder
        return _parseMapping("context://WEB-INF/param/odf/" + _implementationName + baseUri)
            .or(() -> _parseMapping("plugin:odf-sync://conversions/" + _implementationName + baseUri))
            .orElseGet(Map::of);
    }
    
    private Optional<Map<String, String>> _parseMapping(String uri)
    {
        Source source = null;
        try
        {
            source = _srcResolver.resolveURI(uri);
            
            // If a project mapping file is found, read it
            if (source.exists())
            {
                return Optional.of(_parseMappingFile(source));
            }
            else
            {
                getLogger().debug("No mapping at '{}'.", uri);
            }
        }
        catch (IOException | ConfigurationException | SAXException e)
        {
            throw new RuntimeException("Unable to read the mapping file '" + uri + "'", e);
        }
        catch (Exception e)
        {
            throw new RuntimeException("An error occured while parsing the mapping file '" + uri + "'", e);
        }
        finally
        {
            _srcResolver.release(source);
        }
        
        return Optional.empty();
    }
    
    /**
     * Parse the mapping file
     * @param source The source of the file
     * @return The mapping
     * @throws Exception If an error occurs
     */
    protected Map<String, String> _parseMappingFile(Source source) throws Exception
    {
        Map<String, String> mapping = new HashMap<>();
        
        try (InputStream is = source.getInputStream())
        {
            Configuration configuration = new DefaultConfigurationBuilder().build(is);
            for (Configuration itemConf : configuration.getChildren())
            {
                String remoteValue = itemConf.getAttribute("code");
                String ametysValue = itemConf.getValue();
                
                mapping.put(remoteValue, ametysValue);
            }
        }
        
        return mapping;
    }
}
