/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.scc.operator;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.plugins.odfsync.generic.scc.AbstractMappingHelper;

/**
 * Get mapped values from a given connector to Ametys.
 */
public class ODFSynchronizingContentOperatorHelper extends AbstractMappingHelper implements Component, Initializable
{
    /** The ODF TableRef Helper */
    protected OdfReferenceTableHelper _odfRefTableHelper;
    
    /** Mappings connector code - Ametys code */
    protected Map<String, Map<String, String>> _codeMappings;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _odfRefTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
    }

    @Override
    public void initialize() throws Exception
    {
        _codeMappings = new HashMap<>();
    }
    
    /**
     * Get the id of content associated with this connector code
     * @param contentTypeId The content type id
     * @param connectorCode The code into the connector
     * @return The id of content or null if no match found
     */
    public String getReferenceTableEntryId(String contentTypeId, String connectorCode)
    {
        return Optional.ofNullable(connectorCode)
            .flatMap(code -> _getReferenceTableEntry(contentTypeId, code))
            .map(OdfReferenceTableEntry::getId)
            .orElse(null);
    }
    
    /**
     * Get the reference table entry from the connector code
     * @param contentTypeId The content type identifier
     * @param connectorCode The connector code
     * @return the table entry corresponding to the given connector code
     */
    protected Optional<OdfReferenceTableEntry> _getReferenceTableEntry(String contentTypeId, String connectorCode)
    {
        return Optional.of(connectorCode)
            .flatMap(code -> _getReferenceTableEntryFromMapping(code, contentTypeId));
    }
    
    /**
     * Get the reference table entry from the connector code in the mapped values.
     * @param connectorCode The initial connector code
     * @param contentTypeId The content type identifier
     * @return the table entry corresponding to the given connector code
     */
    protected Optional<OdfReferenceTableEntry> _getReferenceTableEntryFromMapping(String connectorCode, String contentTypeId)
    {
        return Optional.ofNullable(connectorCode)
            .map(code -> _getConvertedCode(code, contentTypeId))
            .map(code -> _odfRefTableHelper.getItemFromCode(contentTypeId, code));
    }
    
    /**
     * Get the converter code of the connector code from the mapped values. It the code does not correspond to any entry, the code itself is returned.
     * @param connectorCode The initial connector code
     * @param contentTypeId The content type identifier
     * @return The converted code or the connector code itself
     */
    protected String _getConvertedCode(String connectorCode, String contentTypeId)
    {
        String shortContentTypeId = contentTypeId.substring(contentTypeId.lastIndexOf(".") + 1);
        return _codeMappings
            .computeIfAbsent(shortContentTypeId, this::_readContentTypeMapping)
            .getOrDefault(connectorCode, connectorCode);
    }

    /**
     * Read the mapping file to convert codes.
     * @param contentTypeId The content type identifier
     * @return The mapping of the file
     */
    protected Map<String, String> _readContentTypeMapping(String contentTypeId)
    {
        String fileName = contentTypeId.toLowerCase() + "_conversion.xml";
        String baseUri = "/code/" + fileName;
        
        try
        {
            return super._readMapping(baseUri);
        }
        catch (Exception e)
        {
            getLogger().error("Unable to read the mapping file '{}' for content type '{}'", fileName, contentTypeId, e);
        }
        
        return Map.of();
    }
}
