/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.apogee.scc;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentDAO;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.course.Course;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionDAO;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionHelper;
import org.ametys.runtime.config.Config;

/**
 * Helper for Apogee Synchronizable Contents Collections.
 */
public class ApogeeSynchronizableContentsCollectionHelper implements Serviceable, Component, Contextualizable, Initializable
{
    /** The Avalon Role */
    public static final String ROLE = ApogeeSynchronizableContentsCollectionHelper.class.getName();
 
    /** Request attribute name to store handled contents during import or synchronization. */
    public static final String HANDLED_CONTENTS = AbstractApogeeSynchronizableContentsCollection.class.getName() + "$handledContents";

    /** Request attribute name to store catalog during import or synchronization. */
    public static final String CATALOG = AbstractApogeeSynchronizableContentsCollection.class.getName() + "$catalog";
    
    /** Request attribute name to store language during import or synchronization. */
    public static final String LANG = AbstractApogeeSynchronizableContentsCollection.class.getName() + "$lang";
    
    /** SCC DAO */
    protected SynchronizableContentsCollectionHelper _sccHelper;
    
    /** The ODF Helper */
    protected ODFHelper _odfHelper;
    
    /** SCC DAO */
    protected SynchronizableContentsCollectionDAO _sccDAO;
    
    /** Content DAO */
    protected ContentDAO _contentDAO;
    
    /** The catalogs manager */
    protected CatalogsManager _catalogsManager;
    
    /** Context */
    protected Context _context;
    
    /** Default language configured for ODF */
    protected String _odfLang;
    
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _odfHelper = (ODFHelper) smanager.lookup(ODFHelper.ROLE);
        _sccHelper = (SynchronizableContentsCollectionHelper) smanager.lookup(SynchronizableContentsCollectionHelper.ROLE);
        _sccDAO = (SynchronizableContentsCollectionDAO) smanager.lookup(SynchronizableContentsCollectionDAO.ROLE);
        _contentDAO = (ContentDAO) smanager.lookup(ContentDAO.ROLE);
        _catalogsManager = (CatalogsManager) smanager.lookup(CatalogsManager.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    public void initialize() throws Exception
    {
        _odfLang = Config.getInstance().getValue("odf.programs.lang");
    }
    
    /**
     * Synchronize the content or its children if the content has no Apogee SCC
     * @param content the content
     * @param logger the logger
     */
    public void synchronizeContent(ModifiableContent content, Logger logger)
    {
        boolean handled = false;
        
        AbstractApogeeSynchronizableContentsCollection scc = getContentSCC(content, logger).orElse(null);
        // The content has a SCC, so synchronize it
        if (scc != null)
        {
            try
            {
                Map<String, Object> searchParams = new HashMap<>();
                String idField = scc.getIdField();
                String syncCode = content.getValue(idField);
                if (StringUtils.isNotBlank(syncCode))
                {
                    searchParams.put(idField, syncCode);
                    if (scc.removalSync() && scc.getTotalCount(searchParams, logger) == 0)
                    {
                        _contentDAO.forceDeleteContentsWithLog(List.of(content), null, logger);
                    }
                    else
                    {
                        scc.synchronizeContent(content, logger);
                    }
                    handled = true;
                }
                else
                {
                    logger.warn("The synchronization code of the content '{}' ({}) is empty but it has the SCC '{}'. It has been skipped.", content.getTitle(), content.getId(), scc.getId());
                }
            }
            catch (Exception e)
            {
                logger.error("An error occurred synchronized content '{}' ({}) from SCC '{}'", content.getTitle(), content.getId(), scc.getId(), e);
            }
        }
        
        
        // The content has not been handled because of error or the content is manually created, so search in deeper children
        if (!handled && addToHandleContents(content.getId()) && content instanceof ProgramItem programItem)
        {
            for (ProgramItem syncChild : _odfHelper.getChildProgramItems(programItem))
            {
                synchronizeContent((ModifiableContent) syncChild, logger);
            }
            
            if (content instanceof Course course)
            {
                for (CoursePart syncChild : course.getCourseParts())
                {
                    synchronizeContent(syncChild, logger);
                }
            }
        }
    }
    
    /**
     * Get the Apogee SCC for the content if it exists
     * @param content The content to search on
     * @param logger the logger
     * @return the Apogee SCC
     */
    public Optional<AbstractApogeeSynchronizableContentsCollection> getContentSCC(Content content, Logger logger)
    {
        try
        {
            return _sccHelper.getSynchronizableCollectionIds(content)
                .stream()
                .map(_sccDAO::getSynchronizableContentsCollection)
                .filter(AbstractApogeeSynchronizableContentsCollection.class::isInstance)
                .map(AbstractApogeeSynchronizableContentsCollection.class::cast)
                .findFirst();
        }
        catch (Exception e)
        {
            logger.error("An error occurred getting ametys-internal:scc property", e);
        }
        
        return Optional.empty();
    }
    
    /**
     * Add the content ID to the handle contents list.
     * @param contentId Content ID
     * @return <code>true</code> if the content ID have been added, <code>false</code> is returned if the content ID already exists in the handle contents list.
     */
    public boolean addToHandleContents(String contentId)
    {
        return _addContentToRequestAttribute(contentId, HANDLED_CONTENTS);
    }
    
    private boolean _addContentToRequestAttribute(String contentId, String attributeName)
    {
        Request request = ContextHelper.getRequest(_context);
        @SuppressWarnings("unchecked")
        Set<String> handleContents = (Set<String>) request.getAttribute(attributeName);
        boolean added = handleContents.add(contentId);
        request.setAttribute(attributeName, handleContents);
        return added;
    }
    
    /**
     * Get the synchronization catalog, can be the one defined by the synchronized content
     * or the default catalog.
     * @return the synchronization catalog, can be null
     */
    public String getSynchronizationCatalog()
    {
        return Optional.of(ContextHelper.getRequest(_context))
            .map(r -> (String) r.getAttribute(CATALOG))
            .orElseGet(() -> _catalogsManager.getDefaultCatalogName());
    }
    
    /**
     * Get the synchronization language, can be the one defined by the synchronized content
     * or the ODF language.
     * @return the synchronization language
     */
    public String getSynchronizationLang()
    {
        return Optional.of(ContextHelper.getRequest(_context))
                .map(r -> (String) r.getAttribute(LANG))
                .orElse(_odfLang);
    }
}
