/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.cdmfr.transformers;

import java.io.IOException;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.transformation.AbstractTransformer;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;

/**
 * This transformer provides a valid docbook output for rich-text issued from CDM-fr:
 * <ul>
 *      <li>- no character (break line, bold, text, ...) outside a <code>para</code> element</li>
 *      <li>- no <code>title</code> outside a <code>section</code> element</li>
 * </ul>
 */
public class CDMRichTextTransformer extends AbstractTransformer
{
    private static final String __TAG_ARTICLE = "article";
    private static final String __TAG_SECTION = "section";
    private static final String __TAG_TITLE = "title";
    
    private static final String[] __CHARACTER_TYPES = new String[] {"emphasis", "phrase", "link"};
    private static final String[] __BLOC_TYPES = new String[] {"itemizedlist", "para", "orderedlist", "table"};
    
    int _sectionCount;
    int _titleCount;
    
    boolean _inDocbook;
    boolean _inBlock;
    boolean _inSection;
    boolean _inTitle;
    
    boolean _startPara;
    
    private void _init ()
    {
        _sectionCount = 0;
        _titleCount = 0;
        
        _inDocbook = true;
        _inBlock = false;
        _inSection = false;
        _inTitle = false;
        
        _startPara = false;
    }
    
    @Override
    public void startElement(String uri, String localName, String raw, Attributes attrs) throws SAXException
    {
        if (__TAG_ARTICLE.equals(localName))
        {
            _init();
        }
        
        if (_inDocbook && __TAG_SECTION.equals(localName))
        {
            _sectionCount++;
            
            if (_startPara)
            {
                // End <para>
                _endParaElement();
            }
        }
        
        if (_inDocbook && __TAG_TITLE.equals(localName))
        {
            _titleCount++;
            _inTitle = true;
            
            if (_titleCount > _sectionCount)
            {
                if (_startPara)
                {
                    // End <para>
                    _endParaElement();
                }
                
                _sectionCount++;
                super.startElement("", "section", "section", new AttributesImpl());
            }
        }
        
        if (ArrayUtils.contains(__BLOC_TYPES, localName))
        {
            if (_startPara)
            {
                // End <para>
                _endParaElement();
            }
            _inBlock = true;
        }
        
        // Find <emphasis>, <phrase> or <link> without outside a <listItem> or <para> parent
        if (ArrayUtils.contains(__CHARACTER_TYPES, localName) && !_inBlock)
        {
            // Insert <para> element
            _startParaElement();
        }
        
        super.startElement(uri, localName, raw, attrs);
    }
    
    @Override
    public void endElement(String uri, String localName, String raw) throws SAXException
    {
        if (__TAG_ARTICLE.equals(localName))
        {
            _inDocbook = false;
            
            if (_startPara)
            {
                // End <para>
                _endParaElement();
            }
            
            while (_sectionCount != 0)
            {
                contentHandler.endElement("", "section", "section");
                _sectionCount--;
                _titleCount--;
            }
        }
        
        if (__TAG_TITLE.equals(localName))
        {
            _inTitle = false;
        }
        
        if (__TAG_SECTION.equals(localName))
        {
            _sectionCount--;
            _titleCount--;
            
            if (_startPara)
            {
                // End <para>
                _endParaElement();
            }
        }
        
        if (_inDocbook && ArrayUtils.contains(__BLOC_TYPES, localName))
        {
            _inBlock = false;
        }
        
        super.endElement(uri, localName, raw);
    }
    
    @Override
    public void characters(char[] c, int start, int len) throws SAXException
    {
        boolean isBlank = StringUtils.isBlank(new String (c, start, len));
        
        if (!isBlank && _inDocbook && !_inBlock && !_inTitle)
        {
            _startParaElement();
        }
        super.characters(c, start, len);
    }
    
    private void _startParaElement () throws SAXException
    {
        _startPara = true;
        _inBlock = true;
        contentHandler.startElement("", "para", "para", new AttributesImpl());
    }
    
    private void _endParaElement () throws SAXException
    {
        _startPara = false;
        _inBlock = false;
        contentHandler.endElement("", "para", "para");
    }

    @Override
    public void setup(SourceResolver resolver, Map objectModel, String src, Parameters par) throws ProcessingException, SAXException, IOException
    {
        // Nothing
    }
}
