/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.pegase.ws;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;

import fr.pcscol.pegase.odf.ApiException;
import fr.pcscol.pegase.odf.api.EspacesExterneApi;
import fr.pcscol.pegase.odf.externe.model.Espace;
import fr.pcscol.pegase.odf.externe.model.Pageable;
import fr.pcscol.pegase.odf.externe.model.PagedEspaces;

/**
 * Helper for import and export from Ametys to Pégase.
 */
public class PegaseHelper implements Component, Serviceable, Initializable
{
    /** Role */
    public static final String ROLE = PegaseHelper.class.getName();
    private static final String __CACHE_ID = PegaseHelper.class.getName() + "$pegaseEspaces";
    /** The attribute name for the Pégase code in ref tables mapping */
    private static final String __PEGASE_CODE_FOR_MAPPING = "codePegase";
    
    /* Components */
    private PegaseApiManager _pegaseApiManager;
    private AbstractCacheManager _cacheManager;
    
    private UUID _espaceId;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _cacheManager = (AbstractCacheManager) manager.lookup(AbstractCacheManager.ROLE);
        _pegaseApiManager = (PegaseApiManager) manager.lookup(PegaseApiManager.ROLE);
    }
    
    public void initialize() throws Exception
    {
        if (!_cacheManager.hasCache(__CACHE_ID))
        {
            _cacheManager.createMemoryCache(__CACHE_ID,
                                      new I18nizableText("plugin.odf-sync", "PLUGINS_ODF_CACHE_PEGASE_ESPACES_LABEL"),
                                      new I18nizableText("plugin.odf-sync", "PLUGINS_ODF_CACHE_PEGASE_ESPACES_DESCRIPTION"),
                                      true,
                                      null);
        }
    }
    
    /**
     * Retrieves the Id of the Pégase working space with the configured code
     * @param structureCode The structure code
     * @return The espace Id
     * @throws PegaseExportException if an error occurs
     */
    public UUID getEspaceId(String structureCode) throws PegaseExportException
    {
        if (_espaceId == null)
        {
            String espaceCode = Config.getInstance().getValue("pegase.espace.code");
            
            try
            {
                _espaceId = getEspaceIdByCode(structureCode, espaceCode);
                
                if (_espaceId == null)
                {
                    throw new PegaseExportException("No Pégase working space was found with the given code: '" + espaceCode + "'.");
                }
            }
            catch (ApiException | IOException e)
            {
                throw new PegaseExportException("An error occured while trying to retrieve the pegase working space.", e);
            }
        }
        
        return _espaceId;
    }
    
    /**
     * Initializes the cache with the espaces Ids and codes
     * @param structureCode The pegase structure code
     * @throws ApiException If an error occurs while retrieving the espaces from Pégase
     * @throws IOException If an error occurs while retrieving the Pégase API
     */
    private synchronized void _lazyInitialize(String structureCode) throws ApiException, IOException
    {
        EspacesExterneApi espacesExterneApi = _pegaseApiManager.getEspacesExterneApi();
        
        // Retrieve the first page of Pégase espaces
        PagedEspaces pagedEspaces = _getAndCachePagedEspaces(0, structureCode, espacesExterneApi);
        
        // Go through all Pégase espaces and add them to the cache
        int nbOfPages = pagedEspaces.getTotalPages();
        for (int pageIndex = 1; pageIndex < nbOfPages; pageIndex++)
        {
            _getAndCachePagedEspaces(pageIndex, structureCode, espacesExterneApi);
        }
    }
    
    /**
     * Gets the espaces of the given page and put them in the cache
     * @param page The page
     * @param structureCode The code of the Pégase structure
     * @param espacesExterneApi The espaces Pégase API
     * @return The PagedEspaces found at the page
     * @throws ApiException If an error occurs
     */
    private PagedEspaces _getAndCachePagedEspaces(int page, String structureCode, EspacesExterneApi espacesExterneApi) throws ApiException
    {
        // Pageable to retrieve the maximum number of result for the given page
        Pageable pageable = new Pageable();
        pageable.setPage(page);
        pageable.setTaille(Integer.MAX_VALUE);
        
        // Retrieve the pagedEspaces from Pégase
        PagedEspaces pagedEspaces = espacesExterneApi.rechercherEspaces(structureCode, pageable, null, null, true);
        
        // Put the espaces in the cache
        List<Espace> espaces = pagedEspaces.getItems();
        
        Cache<String, UUID> cache = _cacheManager.<String, UUID>get(__CACHE_ID);
        
        for (Espace espace : espaces)
        {
            cache.put(espace.getCode(), espace.getId());
        }
        
        return pagedEspaces;
    }
    
    /**
     * Get the espace Ids by Pégase code
     * @return the map of espaces ids by Pégase code
     */
    public Map<String, UUID> getEspaceIdsByCodes()
    {
        return _cacheManager.<String, UUID>get(__CACHE_ID).asMap();
    }
    
    /**
     * Get the Id of an espace by its Pégase code
     * @param structureCode The code of the structure
     * @param code The code of the espace
     * @return The UUID of the espace
     * @throws ApiException If an error occurs
     * @throws IOException If an error occurs
     */
    public UUID getEspaceIdByCode(String structureCode, String code) throws ApiException, IOException
    {
        // If the cache has not been initialized, initialize it
        if (!_cacheManager.<String, UUID>get(__CACHE_ID).isInitialized())
        {
            _lazyInitialize(structureCode);
        }
        
        return _cacheManager.<String, UUID>get(__CACHE_ID).get(code);
    }
    
    /**
     * Get the Pégase code of a content, defaults to the Ametys code
     * @param content The content
     * @return The Pégase code
     */
    public String _getPegaseCodeOrCode(Content content)
    {
        return content.hasValue(__PEGASE_CODE_FOR_MAPPING)
                ? content.getValue(__PEGASE_CODE_FOR_MAPPING)
                : content.getValue(OdfReferenceTableEntry.CODE);
    }
    
    /**
     * Get the Pégase codes of the values of a content's field, defaults to the Ametys code
     * @param content The content
     * @param dataPath The data path for the field
     * @return The Pégase codes for the values at given data path, joined by commas
     */
    public String getPegaseCodeForField(Content content, String dataPath)
    {
        return _getPegaseCodes(content, dataPath).collect(Collectors.joining(","));
    }
    
    /**
     * Get the Pégase code of the first value of a content's field, defaults to the Ametys code
     * @param content The content
     * @param dataPath The data path for the field
     * @return The Pégase code for the first value of the content's field
     */
    public String getPegaseCodeForFirstValue(Content content, String dataPath)
    {
        return _getPegaseCodes(content, dataPath).findFirst().orElse(null);
    }
    
    private Stream<String> _getPegaseCodes(Content content, String dataPath)
    {
        if (content.hasValue(dataPath))
        {
            ContentValue[] values = content.isMultiple(dataPath)
                    ? content.<ContentValue[]>getValue(dataPath)
                            : new ContentValue[] {content.<ContentValue>getValue(dataPath)};
                    
            return Stream.of(values)
                    .map(ContentValue::getContentIfExists)
                    .filter(Optional::isPresent)
                    .map(Optional::get)
                    .map(this::_getPegaseCodeOrCode)
                    .filter(StringUtils::isNotEmpty);
        }
        
        return Stream.of();
    }
}
