/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.cdmfr.actions;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.DefaultContent;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.helper.DeleteODFContentHelper;
import org.ametys.odf.helper.DeleteODFContentHelper.DeleteMode;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionHelper;
import org.ametys.plugins.odfsync.cdmfr.RemoteCDMFrSynchronizableContentsCollection;
import org.ametys.plugins.repository.AmetysObjectResolver;

/**
 * Delete a program when program was removed on remote server
 */
public class DeleteRemoteProgramAction extends ServiceableAction
{
    /** The Ametys Object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The delete ODF content helper */
    protected DeleteODFContentHelper _deleteODFContentHelper;
    
    /** The odf helper */
    protected ODFHelper _odfHelper;

    /** The SCC helper */
    protected SynchronizableContentsCollectionHelper _sccHelper;
    
    /** The catalogs manager */
    protected CatalogsManager _catalogsManager;
    
    @Override
    public void service(ServiceManager sManager) throws ServiceException
    {
        super.service(sManager);
        _sccHelper = (SynchronizableContentsCollectionHelper) sManager.lookup(SynchronizableContentsCollectionHelper.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _deleteODFContentHelper = (DeleteODFContentHelper) manager.lookup(DeleteODFContentHelper.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _catalogsManager = (CatalogsManager) manager.lookup(CatalogsManager.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, Object> results = new HashMap<>();
        
        Request request = ObjectModelHelper.getRequest(objectModel);

        Set<String> errors = new HashSet<>();
        
        SynchronizableContentsCollection scc = _sccHelper.getSCCFromModelId(UploadCDMFrAction.REMOTE_CDMFR_SCC_MODEL_ID);
        if (scc == null)
        {
            errors.add("Can't find synchronizable contents collections for the remote CDMFr import with model id " + UploadCDMFrAction.REMOTE_CDMFR_SCC_MODEL_ID);
        }
        else
        {
            String code = _getParameterValue(request, ProgramItem.CODE, errors);
            String lang = _getParameterValue(request, DefaultContent.METADATA_LANGUAGE, errors);
            
            if (errors.isEmpty())
            {
                // Get the catalog from the SCC if is forced, from the remote content and test it, or use default catalog
                String catalog = _getCatalog(scc, request);
                Content content = _odfHelper.getODFContent(ProgramFactory.PROGRAM_CONTENT_TYPE, code, catalog, lang);
                
                // Delete contents bypassing the rights check
                _deleteODFContentHelper.deleteContents(Collections.singletonList(content.getId()), DeleteMode.FULL.name(), true);
            }
        }

        if (!errors.isEmpty())
        {
            results.put("errors", errors);
        }
        
        return results;
    }
    
    private String _getParameterValue(Request request, String parameterName, Set<String> errors)
    {
        String result = request.getParameter(parameterName);
        if (StringUtils.isEmpty(result))
        {
            errors.add("Metadata '" + parameterName + "' is mandatory.");
        }
        
        return result;
    }
    
    private String _getCatalog(SynchronizableContentsCollection scc, Request request)
    {
        String forcedCatalog = (String) scc.getParameterValues().get(RemoteCDMFrSynchronizableContentsCollection.PARAM_CDMFR_CATALOG);
        if (StringUtils.isNotEmpty(forcedCatalog))
        {
            return forcedCatalog;
        }

        String remoteCatalog = request.getParameter(ProgramItem.CATALOG);
        if (_catalogsManager.getCatalog(remoteCatalog) == null)
        {
            // Catalog is empty or do not exist, use the default catalog
            return _catalogsManager.getDefaultCatalogName();
        }
        
        return remoteCatalog;
    }
}
