/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.odfsync.export;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.ametys.cms.repository.Content;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * The report for an export to an external connector (Pégase, Apogée...)
 */
public class ExportReport
{
    private Content _content;
    
    private AbstractExportStructure _structure;
    
    private Map<Content, Set<I18nizableText>> _invalidDataByContent;
    
    /** The problems encountered during export */
    private Set<ProblemTypes> _problemsEncountered;
    
    /** The total number of element*/
    private int _nbTotal;
    
    private Set<Content> _elementsPartlyExported;
    
    private Set<Content> _elementsExported;
    
    private ExportStatus _status;
    
    /**
     * The status of the export
     */
    public enum ExportStatus
    {
        /** Some data from content are invalid (mandatory code, wrong code format) */
        CONTENT_DATA_INVALID(50),
        
        /** The content structure is invalid */
        CONTENT_STRUCTURE_INVALID(100),
        
        /** A program with this code already exist and is not editable */
        NON_EDITABLE_PROGRAM_ALREADY_EXISTS(50),
        
        /** A program with this code already exist and its version is not equals to the one requested or the one requested - 1 */
        PROGRAM_IMPOSSIBLE_VERSION(50),
        
        /** An error occurred during the export */
        ERROR(Integer.MAX_VALUE),
        
        /** Some problems occurred during the export */
        WARN(500),
        
        /** The export is working */
        OK(0);
        
        private int _weight;
        
        private ExportStatus(int weight)
        {
            _weight = weight;
        }
        
        int weight()
        {
            return _weight;
        }
    }
    
    /**
     * Problem types that we can encounter
     */
    public enum ProblemTypes
    {
        // Problems that lead to a failure of the export
        /** An error occurred during the export of an element */
        ELEMENT_NOT_EXPORTED,
        
        /** Some elements already exist and are not editable or mutualisable */
        ELEMENT_ALREADY_EXIST,
        
        /** Some element were not exported and it left a Groupement without children */
        GROUPEMENT_WITHOUT_CHILDREN,
        
        // Problems that lead to a partial success of the export
        /** Some elements could not be linked to each others */
        LINKS_MISSING,
        
        /** Some elements could not be unlinked */
        API_ERROR;
    }
    
    /**
     * The constructor
     * @param content The content on which the report is based
     */
    public ExportReport(Content content)
    {
        _content = content;
        _status = ExportStatus.OK;
        _invalidDataByContent = new HashMap<>();
        _problemsEncountered = new HashSet<>();
        _elementsExported = new HashSet<>();
        _elementsPartlyExported = new HashSet<>();
    }

    /**
     * Get the content in which the report is based
     * @return a content
     */
    public Content getContent()
    {
        return _content;
    }
    
    /**
     * Get the invalid data by content
     * @return the map of invalid data by content
     */
    public Map<Content, Set<I18nizableText>> getInvalidDataPathByContent()
    {
        return _invalidDataByContent;
    }
    
    /**
     * Add invalid model item for a content
     * @param content the content
     * @param invalidMessage the model item
     */
    public void addInvalidDataPath(Content content, I18nizableText invalidMessage)
    {
        updateStatus(ExportStatus.CONTENT_DATA_INVALID);
        _invalidDataByContent.computeIfAbsent(content, __ -> new HashSet<>()).add(invalidMessage);
    }
    
    /**
     * Get the structure implementation.
     * @return the structure implementation
     */
    public AbstractExportStructure getExportStructure()
    {
        return _structure;
    }
    
    /**
     * Set the structure implementation.
     * @param structure the structure implementation.
     */
    public void setExportStructure(AbstractExportStructure structure)
    {
        _structure = structure;
    }
    
    /**
     * Get the export status
     * @return the export status
     */
    public ExportStatus getStatus()
    {
        return _status;
    }
    
    /**
     * Set the export status
     * @param status the export status
     */
    public void setStatus(ExportStatus status)
    {
        _status = status;
    }
    
    /**
     * Getter of nbExported
     * @return The number of element exported
     */
    public int getNbExported()
    {
        return _elementsExported.size();
    }
    
    /**
     * Getter of nbPartiallyExported
     * @return The number of partly exported element
     */
    public int getNbPartlyExported()
    {
        return _elementsPartlyExported.size();
    }
    
    /**
     * Getter of _nbNotExported
     * @return The number of element not exported
     */
    public int getNbNotExported()
    {
        return _nbTotal - getNbExported() - getNbPartlyExported();
    }
    
    /**
     * setter of _nbTotal
     * @param nbTotal The number
     */
    public void setNbTotal(int nbTotal)
    {
        _nbTotal = nbTotal;
    }
    
    /**
     * Getter of _problemsEncountered
     * @return The problems encountered
     */
    public Set<ProblemTypes> getProblemsEncountered()
    {
        return _problemsEncountered;
    }
    
    /**
     * Add an element to the element exported
     * @param element The element to add
     */
    public void addElementExported(Content element)
    {
        if (!_elementsPartlyExported.contains(element))
        {
            _elementsExported.add(element);
        }
    }

    /**
     * Update the export status
     * @param status The export status wanted
     */
    public void updateStatus(ExportStatus status)
    {
        if (status.weight() > _status.weight())
        {
            _status = status;
        }
    }
    
    /**
     * Update the export report by setting the status and problem encountered
     * @param exportStatusWanted The export status wanted
     * @param problemEncountered The problem encountered
     */
    public void updateExportReport(ExportStatus exportStatusWanted, ProblemTypes problemEncountered)
    {
        updateStatus(exportStatusWanted);
        _problemsEncountered.add(problemEncountered);
    }
    
    /**
     * Update the export report by setting the status, problemsEncountered and removing content from elementExported and adding it to elementPartlyExported
     * @param exportStatusWanted The export status wanted
     * @param problemEncountered The problem encountered
     * @param content The content at cause
     */
    public void updateExportReport(ExportStatus exportStatusWanted, ProblemTypes problemEncountered, Content content)
    {
        updateExportReport(exportStatusWanted, problemEncountered);
        _elementsPartlyExported.add(content);
        _elementsExported.remove(content);
    }
}
