/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.cdmfr.components;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.Constants;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.Context;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.xml.dom.DOMParser;
import org.apache.excalibur.xml.xpath.XPathProcessor;
import org.slf4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.data.ContentSynchronizationResult;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.LanguageExpression;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.cms.workflow.EditContentFunction;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.catalog.CatalogsManager;
import org.ametys.odf.course.Course;
import org.ametys.odf.course.ShareableCourseHelper;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.observation.OdfObservationConstants;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.orgunit.RootOrgUnitProvider;
import org.ametys.odf.program.Program;
import org.ametys.odf.translation.TranslationHelper;
import org.ametys.odf.workflow.AbstractCreateODFContentFunction;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionDataProvider;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionHelper;
import org.ametys.plugins.contentio.synchronize.workflow.EditSynchronizedContentFunction;
import org.ametys.plugins.odfsync.cdmfr.CDMFrSyncExtensionPoint;
import org.ametys.plugins.odfsync.cdmfr.ImportCDMFrContext;
import org.ametys.plugins.odfsync.cdmfr.extractor.ImportCDMFrValuesExtractor;
import org.ametys.plugins.odfsync.cdmfr.extractor.ImportCDMFrValuesExtractorFactory;
import org.ametys.plugins.odfsync.cdmfr.transformers.CDMFrSyncTransformer;
import org.ametys.plugins.odfsync.utils.ContentWorkflowDescription;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.external.ExternalizableDataProvider.ExternalizableDataStatus;
import org.ametys.plugins.repository.data.extractor.ModelAwareValuesExtractor;
import org.ametys.plugins.repository.data.holder.values.SynchronizationContext;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.plugins.repository.lock.LockAwareAmetysObject;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workflow.AbstractWorkflowComponent;
import org.ametys.plugins.workflow.AbstractWorkflowComponent.ConditionFailure;
import org.ametys.plugins.workflow.component.CheckRightsCondition;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.View;

import com.opensymphony.workflow.InvalidActionException;
import com.opensymphony.workflow.WorkflowException;

/**
 * Abstract class of a component to import a CDM-fr input stream.
 */
public abstract class AbstractImportCDMFrComponent implements ImportCDMFrComponent, Serviceable, Initializable, Contextualizable, Configurable, Component
{
    /** Tag to identify a program */
    protected static final String _TAG_PROGRAM = "program";

    /** Tag to identify a subprogram */
    protected static final String _TAG_SUBPROGRAM = "subProgram";
    
    /** Tag to identify a container */
    protected static final String _TAG_CONTAINER = "container";
    
    /** Tag to identify a courseList */
    protected static final String _TAG_COURSELIST = "coursesReferences";
    
    /** Tag to identify a coursePart */
    protected static final String _TAG_COURSEPART = "coursePart";

    /** The synchronize workflow action id */
    protected static final int _SYNCHRONIZE_WORKFLOW_ACTION_ID = 800;
    
    /** The Cocoon context */
    protected Context _cocoonContext;
    
    /** The DOM parser */
    protected DOMParser _domParser;

    /** The XPath processor */
    protected XPathProcessor _xPathProcessor;

    /** Extension point to transform CDM-fr */
    protected CDMFrSyncExtensionPoint _cdmFrSyncExtensionPoint;

    /** Default language configured for ODF */
    protected String _odfLang;
    
    /** The catalog manager */
    protected CatalogsManager _catalogsManager;

    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;

    /** The ODF TableRef Helper */
    protected OdfReferenceTableHelper _odfRefTableHelper;

    /** The content type extension point */
    protected ContentTypeExtensionPoint _contentTypeEP;

    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;

    /** The observation manager */
    protected ObservationManager _observationManager;
    
    /** The root orgunit provider */
    protected RootOrgUnitProvider _rootOUProvider;

    /** The ODF Helper */
    protected ODFHelper _odfHelper;
    
    /** The SCC helper */
    protected SynchronizableContentsCollectionHelper _sccHelper;
    
    /** The content workflow helper */
    protected ContentWorkflowHelper _contentWorkflowHelper;
    
    /** The shareable course helper */
    protected ShareableCourseHelper _shareableCourseHelper;
    
    /** the {@link ImportCDMFrValuesExtractor} factory */
    protected ImportCDMFrValuesExtractorFactory _valuesExtractorFactory;
    
    
    /** List of imported contents */
    protected Map<String, Integer> _importedContents;
    
    /** List of synchronized contents */
    protected Set<String> _synchronizedContents;

    /** Number of errors encountered */
    protected int _nbError;
    /** The prefix of the contents */
    protected String _contentPrefix;
    /** Synchronized fields by content type */
    protected Map<String, Set<String>> _syncFieldsByContentType;

    public void initialize() throws Exception
    {
        _odfLang = Config.getInstance().getValue("odf.programs.lang");
    }

    @Override
    public void contextualize(org.apache.avalon.framework.context.Context context) throws ContextException
    {
        _cocoonContext = (Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }

    public void configure(Configuration configuration) throws ConfigurationException
    {
        _parseSynchronizedFields();
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _domParser = (DOMParser) manager.lookup(DOMParser.ROLE);
        _xPathProcessor = (XPathProcessor) manager.lookup(XPathProcessor.ROLE);
        _cdmFrSyncExtensionPoint = (CDMFrSyncExtensionPoint) manager.lookup(CDMFrSyncExtensionPoint.ROLE);
        _catalogsManager = (CatalogsManager) manager.lookup(CatalogsManager.ROLE);
        _odfRefTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
        _contentTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _rootOUProvider = (RootOrgUnitProvider) manager.lookup(RootOrgUnitProvider.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _sccHelper = (SynchronizableContentsCollectionHelper) manager.lookup(SynchronizableContentsCollectionHelper.ROLE);
        _contentWorkflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
        _shareableCourseHelper = (ShareableCourseHelper) manager.lookup(ShareableCourseHelper.ROLE);
        _valuesExtractorFactory = (ImportCDMFrValuesExtractorFactory) manager.lookup(ImportCDMFrValuesExtractorFactory.ROLE);
    }

    @Override
    public String getIdField()
    {
        return "cdmfrSyncCode";
    }

    /**
     * Get the synchronized metadata from the configuration file
     * @throws ConfigurationException if the configuration is not valid.
     */
    private void _parseSynchronizedFields() throws ConfigurationException
    {
        _syncFieldsByContentType = new HashMap<>();
        
        File cdmfrMapping = new File(_cocoonContext.getRealPath("/WEB-INF/param/odf-synchro.xml"));
        try (InputStream is = !cdmfrMapping.isFile()
                            ? getClass().getResourceAsStream("/org/ametys/plugins/odfsync/cdmfr/odf-synchro.xml")
                            : new FileInputStream(cdmfrMapping))
        {
            Configuration cfg = new DefaultConfigurationBuilder().build(is);

            Configuration[] cTypesConf = cfg.getChildren("content-type");
            for (Configuration cTypeConf : cTypesConf)
            {
                String contentType = cTypeConf.getAttribute("id");
                Set<String> syncAttributes = _configureSynchronizedFields(cTypeConf, StringUtils.EMPTY);
                _syncFieldsByContentType.put(contentType, syncAttributes);
            }
        }
        catch (Exception e)
        {
            throw new ConfigurationException("Error while parsing odf-synchro.xml", e);
        }
    }

    private Set<String> _configureSynchronizedFields(Configuration configuration, String prefix) throws ConfigurationException
    {
        Set<String> syncAttributes = new HashSet<>();
        Configuration[] attributesConf = configuration.getChildren("attribute");
        
        if (attributesConf.length > 0)
        {
            for (Configuration attributeConf : attributesConf)
            {
                if (attributeConf.getChildren("attribute").length > 0)
                {
                    // composite
                    syncAttributes.addAll(_configureSynchronizedFields(attributeConf, prefix + attributeConf.getAttribute("name") + ModelItem.ITEM_PATH_SEPARATOR));
                }
                else
                {
                    syncAttributes.add(prefix + attributeConf.getAttribute("name"));
                }
            }
        }
        else if (configuration.getAttribute("name", null) != null)
        {
            syncAttributes.add(prefix + configuration.getAttribute("name"));
        }
        
        return syncAttributes;
    }
    
    @Override
    @SuppressWarnings("unchecked")
    public synchronized Map<String, Object> handleInputStream(InputStream input, Map<String, Object> parameters, SynchronizableContentsCollection scc, Logger logger) throws ProcessingException
    {
        List<ModifiableContent> importedPrograms = new ArrayList<>();
        
        _importedContents = new HashMap<>();
        _synchronizedContents = (Set<String>) parameters.getOrDefault("updatedContents", new HashSet<>());
        int nbCreatedContents = (int) parameters.getOrDefault("nbCreatedContents", 0);
        int nbSynchronizedContents = (int) parameters.getOrDefault("nbSynchronizedContents", 0);
        _nbError = (int) parameters.getOrDefault("nbError", 0);
        _contentPrefix = (String) parameters.getOrDefault("contentPrefix", "cdmfr-");
        additionalParameters(parameters);

        Map<String, Object> resultMap = new HashMap<>();
        
        try
        {
            Document doc = _domParser.parseDocument(new InputSource(input));
            doc = transformDocument(doc, new HashMap<>(), logger);
            
            if (doc != null)
            {
                String defaultLang = _getXPathString(doc, "CDM/@language", _odfLang);
                
                NodeList nodes = doc.getElementsByTagName(_TAG_PROGRAM);
                
                for (int i = 0; i < nodes.getLength(); i++)
                {
                    Element contentElement = (Element) nodes.item(i);
                    String syncCode = _xPathProcessor.evaluateAsString(contentElement, "@CDMid");
                    String contentLang = _getXPathString(contentElement, "@language", defaultLang);
                    contentLang = StringUtils.substring(contentLang, 0, 2).toLowerCase(); // on keep the language from the locale
                    
                    String catalog = getCatalogName(contentElement);

                    ImportCDMFrContext context = new ImportCDMFrContext(scc, doc, contentLang, catalog, logger);
                    importedPrograms.add(importOrSynchronizeContent(contentElement, ContentWorkflowDescription.PROGRAM_WF_DESCRIPTION, syncCode, syncCode, context));
                }
                
                // Validate newly imported contents
                if (validateAfterImport())
                {
                    for (String contentId : _importedContents.keySet())
                    {
                        WorkflowAwareContent content = _resolver.resolveById(contentId);
                        Integer validationActionId = _importedContents.get(contentId);
                        if (validationActionId > 0)
                        {
                            validateContent(content, validationActionId, logger);
                        }
                    }
                }
            }
        }
        catch (IOException | ProcessingException e)
        {
            throw new ProcessingException("An error occured while transforming the stream.", e);
        }
        catch (SAXException e)
        {
            throw new ProcessingException("An error occured while parsing the stream.", e);
        }
        catch (Exception e)
        {
            throw new ProcessingException("An error occured while synchronizing values on contents.", e);
        }

        resultMap.put("importedContents", _importedContents.keySet());
        resultMap.put("nbCreatedContents", nbCreatedContents + _importedContents.size());
        resultMap.put("updatedContents", _synchronizedContents);
        resultMap.put("nbSynchronizedContents", nbSynchronizedContents + _synchronizedContents.size());
        resultMap.put("nbError", _nbError);
        resultMap.put("importedPrograms", importedPrograms);

        return resultMap;
    }

    /**
     * True to validate the contents after import
     * @return True to validate the contents after import
     */
    protected abstract boolean validateAfterImport();

    /**
     * When returns true, a content created by a previous synchro will be removed if it does not exist anymore during the current synchro.
     * @return true if a content created by a previous synchro has to be removed if it does not exist anymore during the current synchro.
     */
    protected abstract boolean removalSync();
    
    /**
     * Additional parameters for specific treatments.
     * @param parameters The parameters map to get
     */
    protected abstract void additionalParameters(Map<String, Object> parameters);
    
    /**
     * Transform the document depending of it structure.
     * @param document Document to transform.
     * @param parameters Optional parameters for transformation
     * @param logger The logger
     * @return The transformed document.
     * @throws IOException if an error occurs.
     * @throws SAXException if an error occurs.
     * @throws ProcessingException if an error occurs.
     */
    protected Document transformDocument(Document document, Map<String, Object> parameters, Logger logger) throws IOException, SAXException, ProcessingException
    {
        Optional<CDMFrSyncTransformer> transformer = _cdmFrSyncExtensionPoint.getFirstSupportingExtension(document);
        
        if (transformer.isEmpty())
        {
            logger.error("Cannot match a CDM-fr transformer to this file structure.");
            return null;
        }
        
        return transformer.get().transform(document, parameters);
    }

    public String getCatalogName(Element contentElement)
    {
        String defaultCatalog = _catalogsManager.getDefaultCatalogName();
        
        String contentCatalog = _getXPathString(contentElement, "catalog", defaultCatalog);
        if (_catalogsManager.getCatalog(contentCatalog) == null)
        {
            // Catalog is empty or do not exist, use the default catalog
            return defaultCatalog;
        }
        
        return contentCatalog;
    }
    
    public ModifiableContent importOrSynchronizeContent(Element contentElement, ContentWorkflowDescription wfDescription, String title, String syncCode, ImportCDMFrContext context)
    {
        ModifiableContent content = _getOrCreateContent(wfDescription, title, syncCode, context);
        
        if (content != null)
        {
            try
            {
                _sccHelper.updateLastSynchronizationProperties(content);
                content.saveChanges();
                _synchronizeContent(contentElement, content, wfDescription.getContentType(), syncCode, context);
            }
            catch (Exception e)
            {
                _nbError++;
                context.getLogger().error("Failed to synchronize data for content {} and language {}.", content, context.getLang(), e);
            }
        }
        
        return content;
    }
    
    /**
     * Get or create the content from the workflow description, the synchronization code and the import context.
     * @param wfDescription The workflow description
     * @param title The title
     * @param syncCode The synchronization code
     * @param context The import context
     * @return the retrieved or created content
     */
    protected ModifiableContent _getOrCreateContent(ContentWorkflowDescription wfDescription, String title, String syncCode, ImportCDMFrContext context)
    {
        ModifiableContent receivedContent = getContent(wfDescription.getContentType(), syncCode, context);
        if (receivedContent != null)
        {
            return receivedContent;
        }
        
        try
        {
            context.getLogger().info("Creating content '{}' with the content type '{}' for language {}", title, wfDescription.getContentType(), context.getLang());
            
            ModifiableContent content = _createContent(wfDescription, title, context);
            if (content != null)
            {
                _sccHelper.updateSCCProperty(content, context.getSCC().getId());
                content.setValue(getIdField(), syncCode);
                content.saveChanges();
                _importedContents.put(content.getId(), wfDescription.getValidationActionId());
            }
            
            return content;
        }
        catch (WorkflowException e)
        {
            context.getLogger().error("Failed to initialize workflow for content {} and language {}", title, context.getLang(), e);
            _nbError++;
            return null;
        }
    }

    public ModifiableContent getContent(String contentType, String syncCode, ImportCDMFrContext context)
    {
        List<Expression> expList = _getExpressionsList(contentType, syncCode, context);
        AndExpression andExp = new AndExpression(expList.toArray(new Expression[expList.size()]));
        String xPathQuery = ContentQueryHelper.getContentXPathQuery(andExp);

        AmetysObjectIterable<ModifiableContent> contents = _resolver.query(xPathQuery);

        if (contents.getSize() > 0)
        {
            return contents.iterator().next();
        }
        
        return null;
    }
    
    /**
     * Create the content from the workflow description and the import context.
     * @param wfDescription The workflow description
     * @param title The title
     * @param context The import context
     * @return the created content
     * @throws WorkflowException if an error occurs while creating the content
     */
    protected ModifiableContent _createContent(ContentWorkflowDescription wfDescription, String title, ImportCDMFrContext context) throws WorkflowException
    {
        String contentName = NameHelper.filterName(_contentPrefix + "-" + title + "-" + context.getLang());
        
        Map<String, Object> inputs = _getInputsForContentCreation(wfDescription, context);
        Map<String, Object> result = _contentWorkflowHelper.createContent(
                wfDescription.getWorkflowName(),
                wfDescription.getInitialActionId(),
                contentName,
                title,
                new String[] {wfDescription.getContentType()},
                null,
                context.getLang(),
                inputs);
        
        return _resolver.resolveById((String) result.get("contentId"));
    }
    
    /**
     * Retrieves the inputs to give for content creation
     * @param wfDescription The workflow description
     * @param context The import context
     * @return the inputs to give for content creation
     */
    protected Map<String, Object> _getInputsForContentCreation(ContentWorkflowDescription wfDescription, ImportCDMFrContext context)
    {
        Map<String, Object> inputs = new HashMap<>();
        
        ContentType contentType = _contentTypeEP.getExtension(wfDescription.getContentType());
        if (contentType.hasModelItem(ProgramItem.CATALOG) || contentType.hasModelItem(CoursePart.CATALOG))
        {
            inputs.put(AbstractCreateODFContentFunction.CONTENT_CATALOG_KEY, context.getCatalog());
        }
        
        return inputs;
    }
    
    /**
     * Synchronize content
     * @param contentElement the DOM content element
     * @param content The content to synchronize
     * @param contentTypeId The content type ID
     * @param syncCode The synchronization code
     * @param context the import context
     * @throws Exception if an error occurs while synchronizing the content values
     */
    protected void _synchronizeContent(Element contentElement, ModifiableContent content, String contentTypeId, String syncCode, ImportCDMFrContext context) throws Exception
    {
        Logger logger = context.getLogger();
        
        // Avoid a treatment twice or more
        if (_synchronizedContents.add(content.getId()))
        {
            logger.info("Synchronization of the content '{}' with the content type '{}'", content.getTitle(), contentTypeId);
            
            if (content instanceof LockAwareAmetysObject && ((LockAwareAmetysObject) content).isLocked())
            {
                logger.warn("The content '{}' ({}) is currently locked by user {}: it cannot be synchronized", content.getTitle(), content.getId(), ((LockAwareAmetysObject) content).getLockOwner());
            }
            else if (content instanceof WorkflowAwareContent)
            {
                ContentType contentType = _contentTypeEP.getExtension(contentTypeId);
                ModelAwareValuesExtractor valuesExtractor = _valuesExtractorFactory.getValuesExtractor(contentElement, this, content, contentType, syncCode, context);
                
                // Extract the values
                Map<String, Object> values = valuesExtractor.extractValues();
                values.putAll(_getAdditionalValuesToSynchronize(content, syncCode, context));
                
                // Modify the content with the extracted values
                boolean create = _importedContents.containsKey(content.getId());
                Set<String> notSynchronizedContentIds = _getNotSynchronizedRelatedContentIds(content, syncCode, context);
                _editContent((WorkflowAwareContent) content, Optional.empty(), values, create, notSynchronizedContentIds, context);
                
                if (content instanceof OrgUnit)
                {
                    _setOrgUnitParent((WorkflowAwareContent) content, context);
                }
                
                // Create translation links
                _linkTranslationsIfExist(content, contentTypeId, context);
            }
        }
    }

    /**
     * Retrieves additional values to synchronize for the content
     * @param content the content
     * @param syncCode the content synchronization code
     * @param context the import context
     * @return the additional values
     */
    protected Map<String, Object> _getAdditionalValuesToSynchronize(ModifiableContent content, String syncCode, ImportCDMFrContext context)
    {
        Map<String, Object> additionalValues = new HashMap<>();
        additionalValues.put(getIdField(), syncCode);
        return additionalValues;
    }
    
    /**
     * Retrieves the ids of the contents related to the given content but that are not part of the synchronization
     * @param content the content
     * @param syncCode the content synchronization code
     * @param context the import context
     * @return the not synchronized content ids
     */
    protected Set<String> _getNotSynchronizedRelatedContentIds(ModifiableContent content, String syncCode, ImportCDMFrContext context)
    {
        return new HashSet<>();
    }
    
    /**
     * Synchronize the content with given values.
     * @param content The content to synchronize
     * @param view the view containing the item to edit
     * @param values the values
     * @param create <code>true</code> if content is creating, false if it is updated
     * @param notSynchronizedContentIds the ids of the contents related to the given content but that are not part of the synchronization
     * @param context the import context
     * @throws WorkflowException if an error occurs
     */
    protected void _editContent(WorkflowAwareContent content, Optional<View> view, Map<String, Object> values, boolean create, Set<String> notSynchronizedContentIds, ImportCDMFrContext context) throws WorkflowException
    {
        SynchronizationContext synchronizationContext = SynchronizationContext.newInstance()
                                                                              .withStatus(ExternalizableDataStatus.EXTERNAL)
                                                                              .withExternalizableDataContextEntry(SynchronizableContentsCollectionDataProvider.SCC_ID_CONTEXT_KEY, context.getSCC().getId())
                                                                              .withIncompatibleValuesIgnored(true);

        if (view.map(v -> content.hasDifferences(v, values, synchronizationContext))
                .orElseGet(() -> content.hasDifferences(values, synchronizationContext)))
        {
            context.getLogger().info("Some changes were detected for content '{}' and language {}", content.getTitle(), context.getLang());
            
            Map<String, Object> inputs = new HashMap<>();
            inputs.put(EditSynchronizedContentFunction.SCC_KEY, context.getSCC());
            inputs.put(EditSynchronizedContentFunction.SCC_LOGGER_KEY, context.getLogger());
            inputs.put(EditSynchronizedContentFunction.NOT_SYNCHRONIZED_RELATED_CONTENT_IDS_KEY, notSynchronizedContentIds);
            if (ignoreRights())
            {
                inputs.put(CheckRightsCondition.FORCE, true);
            }
    
            Map<String, Object> params = new HashMap<>();
            // Remove catalog data, this value is forced at creation and should not be modified
            values.remove(ProgramItem.CATALOG);
            params.put(EditContentFunction.VALUES_KEY, values);
            view.ifPresent(v -> params.put(EditContentFunction.VIEW, v));
            params.put(EditContentFunction.QUIT, true);
            params.put(EditSynchronizedContentFunction.IMPORT, create);
            inputs.put(AbstractWorkflowComponent.CONTEXT_PARAMETERS_KEY, params);
            
            _contentWorkflowHelper.doAction(content, _SYNCHRONIZE_WORKFLOW_ACTION_ID, inputs);
        }
        else
        {
            context.getLogger().info("No changes detected for content '{}' and language {}", content.getTitle(), context.getLang());
        }
    }
    
    public ContentSynchronizationResult additionalOperations(ModifiableContent content, Map<String, Object> additionalParameters, Logger logger)
    {
        ContentSynchronizationResult result = new ContentSynchronizationResult();

        if (content instanceof Program)
        {
            List<ModifiableContent> modifiedContents = _initializeShareableCoursesFields((Program) content);
            
            result.addModifiedContents(modifiedContents);
            result.setHasChanged(!modifiedContents.isEmpty());
        }
        
        return result;
    }
    
    /**
     * Initialize shareable fields for the courses under the given {@link ProgramItem}
     * @param programItem the program item
     * @return the list of contents that have been modified during the initialization
     */
    protected List<ModifiableContent> _initializeShareableCoursesFields(ProgramItem programItem)
    {
        List<ModifiableContent> modifiedContents = new ArrayList<>();

        List<ProgramItem> children = _odfHelper.getChildProgramItems(programItem);
        for (ProgramItem child : children)
        {
            if (child instanceof Course && programItem instanceof CourseList)
            {
                if (_shareableCourseHelper.initializeShareableFields((Course) child, (CourseList) programItem,  UserPopulationDAO.SYSTEM_USER_IDENTITY, true))
                {
                    modifiedContents.add((Course) child);
                }
            }
            
            modifiedContents.addAll(_initializeShareableCoursesFields(child));
        }

        return modifiedContents;
    }
    
    /**
     * Search for translated contents
     * @param importedContent The imported content
     * @param contentType The content type
     * @param context the import context
     */
    protected void _linkTranslationsIfExist(ModifiableContent importedContent, String contentType, ImportCDMFrContext context)
    {
        if (importedContent instanceof ProgramItem)
        {
            Expression expression = _getTranslationExpression(importedContent, contentType);
            String xPathQuery = ContentQueryHelper.getContentXPathQuery(expression);
    
            AmetysObjectIterable<ModifiableContent> contents = _resolver.query(xPathQuery);
            
            Map<String, String> translations = new HashMap<>();
            for (ModifiableContent content : contents)
            {
                translations.put(content.getLanguage(), content.getId());
            }
            
            for (ModifiableContent content : contents)
            {
                TranslationHelper.setTranslations(content, translations);
    
                Map<String, Object> eventParams = new HashMap<>();
                eventParams.put(ObservationConstants.ARGS_CONTENT, content);
                eventParams.put(ObservationConstants.ARGS_CONTENT_ID, content.getId());
                _observationManager.notify(new Event(OdfObservationConstants.ODF_CONTENT_TRANSLATED, _currentUserProvider.getUser(), eventParams));
            }
        }
    }
    
    private Expression _getTranslationExpression(ModifiableContent content, String contentType)
    {
        List<Expression> expList = new ArrayList<>();
        
        if (StringUtils.isNotBlank(contentType))
        {
            expList.add(new ContentTypeExpression(Operator.EQ, contentType));
        }
        
        String catalog = content.getValue(ProgramItem.CATALOG);
        if (StringUtils.isNotBlank(catalog))
        {
            expList.add(new StringExpression(ProgramItem.CATALOG, Operator.EQ, catalog));
        }
        
        List<Expression> codeExpressionList = new ArrayList<>();
        String syncValue = content.getValue(getIdField());
        if (StringUtils.isNotBlank(syncValue))
        {
            codeExpressionList.add(new StringExpression(getIdField(), Operator.EQ, syncValue));
        }

        String code = content.getValue(ProgramItem.CODE);
        if (StringUtils.isNotBlank(syncValue))
        {
            codeExpressionList.add(new StringExpression(ProgramItem.CODE, Operator.EQ, code));
        }
        
        if (!codeExpressionList.isEmpty())
        {
            expList.add(new OrExpression(codeExpressionList.toArray(Expression[]::new)));
        }
        
        return new AndExpression(expList.toArray(Expression[]::new));
    }

    /**
     * Set the orgUnit parent to rootOrgUnit.
     * @param orgUnit The orgunit to link
     * @param context the import context
     * @throws Exception if an error occurs while synchronizing the content values
     */
    protected void _setOrgUnitParent(WorkflowAwareContent orgUnit, ImportCDMFrContext context) throws Exception
    {
        // Set the orgUnit parent (if no parent is set)
        if (!orgUnit.hasValue(OrgUnit.PARENT_ORGUNIT))
        {
            OrgUnit rootOrgUnit = _rootOUProvider.getRoot();
            Map<String, Object> values = new HashMap<>();
            values.put(OrgUnit.PARENT_ORGUNIT, rootOrgUnit);
            _editContent(orgUnit, Optional.empty(), values, false, Set.of(rootOrgUnit.getId()), context);
        }
    }
    
    /**
     * Validates a content after import
     * @param content The content to validate
     * @param validationActionId Validation action ID to use for this content
     * @param logger The logger
     */
    protected void validateContent(WorkflowAwareContent content, int validationActionId, Logger logger)
    {
        Map<String, Object> inputs = new HashMap<>();
        if (ignoreRights())
        {
            inputs.put(CheckRightsCondition.FORCE, true);
        }
        
        try
        {
            _contentWorkflowHelper.doAction(content, validationActionId, inputs);
            logger.info("The content {} has been validated after import", content);
        }
        catch (WorkflowException | InvalidActionException e)
        {
            String failuresAsString = _getActionFailuresAsString(inputs);
            logger.error("The content {} cannot be validated after import{}", content, failuresAsString, e);
        }
    }
    
    private String _getActionFailuresAsString(Map<String, Object> actionInputs)
    {
        String failuresAsString = "";
        if (actionInputs.containsKey(AbstractWorkflowComponent.FAIL_CONDITIONS_KEY))
        {
            @SuppressWarnings("unchecked")
            List<ConditionFailure> failures = (List<ConditionFailure>) actionInputs.get(AbstractWorkflowComponent.FAIL_CONDITIONS_KEY);
            if (!failures.isEmpty())
            {
                failuresAsString = ", due to the following error(s):\n" + String.join("\n", failures.stream().map(ConditionFailure::text).toList());
            }
        }
        
        return failuresAsString;
    }
    
    public String getIdFromCDMThenCode(String tableRefId, String cdmCode)
    {
        OdfReferenceTableEntry entry = _odfRefTableHelper.getItemFromCDM(tableRefId, cdmCode);
        if (entry == null)
        {
            entry = _odfRefTableHelper.getItemFromCode(tableRefId, cdmCode);
        }
        return entry != null ? entry.getId() : null;
    }
    
    private String _getXPathString(Node metadataNode, String xPath, String defaultValue)
    {
        String value = _xPathProcessor.evaluateAsString(metadataNode, xPath);
        if (StringUtils.isEmpty(value))
        {
            value = defaultValue;
        }
        return value;
    }

    /**
     * If true, bypass the rights check during the import process
     * @return True if the rights check are bypassed during the import process
     */
    protected boolean ignoreRights()
    {
        return false;
    }
    
    /**
     * Construct the query to retrieve the content.
     * @param contentTypeId The content type
     * @param syncCode The synchronization code
     * @param context the import context
     * @return The {@link List} of {@link Expression}
     */
    protected List<Expression> _getExpressionsList(String contentTypeId, String syncCode, ImportCDMFrContext context)
    {
        List<Expression> expList = new ArrayList<>();
        
        if (StringUtils.isNotBlank(contentTypeId))
        {
            expList.add(new ContentTypeExpression(Operator.EQ, contentTypeId));
            
            if (StringUtils.isNotBlank(context.getCatalog()))
            {
                ContentType contentType = _contentTypeEP.getExtension(contentTypeId);
                if (contentType.hasModelItem(ProgramItem.CATALOG) || contentType.hasModelItem(CoursePart.CATALOG))
                {
                    expList.add(new StringExpression(ProgramItem.CATALOG, Operator.EQ, context.getCatalog()));
                }
            }
        }
        
        if (StringUtils.isNotBlank(syncCode))
        {
            expList.add(new StringExpression(getIdField(), Operator.EQ, syncCode));
        }
        
        if (StringUtils.isNotBlank(context.getLang()))
        {
            expList.add(new LanguageExpression(Operator.EQ, context.getLang()));
        }
        
        return expList;
    }

    @Override
    public Set<String> getLocalAndExternalFields(Map<String, Object> additionalParameters)
    {
        if (additionalParameters == null || !additionalParameters.containsKey("contentTypes"))
        {
            throw new IllegalArgumentException("Content types shouldn't be null.");
        }

        @SuppressWarnings("unchecked")
        List<String> contentTypeIds = (List<String>) additionalParameters.get("contentTypes");
        Set<String> allSyncFields = new HashSet<>();
        
        for (String contentTypeId : contentTypeIds)
        {
            Set<String> syncFields = _syncFieldsByContentType.computeIfAbsent(contentTypeId, k -> new HashSet<>());
            allSyncFields.addAll(syncFields);
        }
        
        return allSyncFields;
    }
}
