/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.cdmfr.components.impl;

import java.lang.reflect.Array;
import java.text.Normalizer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.w3c.dom.Element;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.data.holder.impl.DefaultModifiableModelAwareDataHolder;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.repository.LanguageExpression;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.ProgramPart;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.program.SubProgramFactory;
import org.ametys.plugins.odfsync.cdmfr.ImportCDMFrContext;
import org.ametys.plugins.odfsync.cdmfr.RemoteCDMFrSynchronizableContentsCollection;
import org.ametys.plugins.odfsync.utils.ContentWorkflowDescription;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.extractor.ModelAwareValuesExtractor;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.ModifiableModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.group.ModelAwareComposite;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.repository.data.holder.impl.DataHolderHelper;
import org.ametys.plugins.repository.data.holder.values.SynchronizationContext;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.model.CompositeDefinition;
import org.ametys.plugins.repository.model.ViewHelper;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.ViewItemContainer;
import org.ametys.runtime.model.type.ElementType;

import com.opensymphony.workflow.WorkflowException;


/**
 * Component to import a CDM-fr input stream from a remote server with co-accredited mode.
 */
public class CoAccreditedRemoteImportCDMFrComponent extends RemoteImportCDMFrComponent
{
    /** The name of the JCR node holding the shared metadata  */
    protected static final String SHARED_PROGRAMS_NODE_NAME = "shared-programs";
    
    /** The list of attributes to copy for mention program */
    protected Set<String> _mentionAttributePaths;
    
    /** The list of attributes to merge */
    protected Set<String> _attributePathsToMerge;
    
    /** The content type of the mention */
    protected ContentType _mentionContentType;
    
    /** The mention id */
    protected String _mentionId;
    
    /** The program code to link to the mention */
    protected String _programToLinkCode;
    
    /** The content type of the sub program */
    protected ContentType _subProgramContentType;
    
    /** Attribute to handle shared sub programs or not */
    protected boolean _handleSharedSubPrograms;
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        super.configure(configuration);

        if ("co-accredited".equals(configuration.getName()))
        {
            _configureCoAccreditedParams(configuration);
        }
        else
        {
            _configureCoAccreditedParams(configuration.getChild("co-accredited"));
        }
    }
    
    /**
     * Configure the co-accredited params
     * @param configuration the configuration
     * @throws ConfigurationException if an error occurred
     */
    protected void _configureCoAccreditedParams(Configuration configuration) throws ConfigurationException
    {
        _mentionAttributePaths = new HashSet<>();
        Configuration mentionConf = configuration.getChild("mention");
        if (mentionConf != null)
        {
            Configuration attributesConf = mentionConf.getChild("attributes-to-copy");
            if (attributesConf != null)
            {
                for (Configuration attributeConf : attributesConf.getChildren())
                {
                    String attributePath = attributeConf.getAttribute("path");
                    _mentionAttributePaths.add(attributePath);
                }
            }
        }
        
        _attributePathsToMerge = new HashSet<>();
        Configuration sharedWithConf = configuration.getChild("shared-with");
        if (sharedWithConf != null)
        {
            Configuration attributesConf = sharedWithConf.getChild("attributes-to-merge");
            if (attributesConf != null)
            {
                for (Configuration attributeConf : attributesConf.getChildren())
                {
                    String attributePath = attributeConf.getAttribute("path");
                    _attributePathsToMerge.add(attributePath);
                }
            }
        }
    }
    
    @Override
    public void initialize() throws Exception
    {
        super.initialize();
        _mentionContentType = _contentTypeEP.getExtension(ContentWorkflowDescription.PROGRAM_WF_DESCRIPTION.getContentType());
        _subProgramContentType = _contentTypeEP.getExtension(ContentWorkflowDescription.SUBPROGRAM_WF_DESCRIPTION.getContentType());
    }

    @Override
    protected void additionalParameters(Map<String, Object> parameters)
    {
        _mentionId = null;
        String sharedSubProgramType = (String) parameters.get(RemoteCDMFrSynchronizableContentsCollection.PARAM_SHARED_WITH_TYPE);
        _handleSharedSubPrograms = sharedSubProgramType != null && !"NONE".equalsIgnoreCase(sharedSubProgramType);
        
        super.additionalParameters(parameters);
    }
    
    @Override
    public ModifiableContent importOrSynchronizeContent(Element contentElement, ContentWorkflowDescription wfDescription, String title, String syncCode, ImportCDMFrContext context)
    {
        ModifiableContent content = null;
        if (contentElement.getLocalName().equals(_TAG_PROGRAM))
        {
            String educationKind = _xPathProcessor.evaluateAsString(contentElement, AbstractProgram.EDUCATION_KIND);
            String mention = _xPathProcessor.evaluateAsString(contentElement, AbstractProgram.MENTION);

            // This is a co-accredited program
            if (StringUtils.isNotBlank(mention) && "parcours".equals(educationKind))
            {
                try
                {
                    // Get or create the mention from the program
                    Program mentionProgram = _getOrCreateMention(contentElement, mention, syncCode, context);
                    _mentionId = mentionProgram.getId();
                    
                    // Store the content to link to the mention
                    _programToLinkCode = syncCode;
                    
                    // Handle shared subPrograms
                    if (_handleSharedSubPrograms)
                    {
                        content = _importOrSynchronizeSharedSubProgram(mentionProgram, contentElement, title, syncCode, context);
                    }
                }
                catch (CoAccreditedRemoteImportException e)
                {
                    context.getLogger().error(e.getMessage(), e.getArguments());
                    _nbError++;
                    return null;
                }
            }
        }

        if (content == null)
        {
            content = super.importOrSynchronizeContent(contentElement, wfDescription, title, syncCode, context);
        }
        
        if (_mentionId != null && content instanceof SubProgram && _programToLinkCode.equals(content.getValue(getIdField())))
        {
            // Mettre les composantes de la mention à jour
            Program mention = _resolver.resolveById(_mentionId);
            _updateMentionAttributes(mention, (SubProgram) content, context);
        }
        
        return content;
    }
    
    /**
     * Get of create the mention program
     * @param contentElement the content element
     * @param mentionCode the mention code
     * @param syncCode the content synchronization code
     * @param context the import context
     * @return the mention program
     * @throws CoAccreditedRemoteImportException if an explained error occurs during import
     */
    protected Program _getOrCreateMention(Element contentElement, String mentionCode, String syncCode, ImportCDMFrContext context) throws CoAccreditedRemoteImportException
    {
        String degreeCodeCDM = _xPathProcessor.evaluateAsString(contentElement, AbstractProgram.DEGREE);
        if (StringUtils.isEmpty(degreeCodeCDM))
        {
            throw new CoAccreditedRemoteImportException("Le diplôme n'est pas défini sur la formation {}, elle n'a pas été importée.", syncCode);
        }
        
        OdfReferenceTableEntry degree = _odfRefTableHelper.getItemFromCDM(OdfReferenceTableHelper.DEGREE, degreeCodeCDM);
        if (degree == null)
        {
            throw new CoAccreditedRemoteImportException("Le diplôme {} n'existe pas, la formation {} n'a pas été importée.", degreeCodeCDM, syncCode);
        }
        String degreeCode = degree.getCode();
        
        String mentionType = _odfRefTableHelper.getMentionForDegree(degree.getId());
        if (mentionType == null)
        {
            throw new CoAccreditedRemoteImportException("Il n'y a pas de type de mention (licence, licence pro, master) associée au diplôme {}, la formation {} n'a pas été importée.", degreeCode, syncCode);
        }
        
        OdfReferenceTableEntry mention = _odfRefTableHelper.getItemFromCDM(mentionType, mentionCode);
        if (mention == null)
        {
            throw new CoAccreditedRemoteImportException("Il n'y a pas de mention de type '{}' (pour le diplôme {}) avec le code {}, La formation {} n'a pas été importée.", mentionType, degreeCode, mentionCode, syncCode);
        }
        
        Program mentionProgram = _getMention(mention.getId(), degree.getId(), context);
        if (mentionProgram == null)
        {
            mentionProgram = _createMention(contentElement, mention, syncCode, context);
            _importedContents.put(mention.getId(), ContentWorkflowDescription.PROGRAM_WF_DESCRIPTION.getValidationActionId());
        }
        return mentionProgram;
    }
    
    /**
     * Get the mention program
     * @param mentionId the mention content id
     * @param degreeId the degree content id
     * @param context the import context
     * @return the mention program or <code>null</code> if it doesn't exist
     */
    protected Program _getMention(String mentionId, String degreeId, ImportCDMFrContext context)
    {
        List<Expression> expList = new ArrayList<>();
        expList.add(new ContentTypeExpression(Operator.EQ, ProgramFactory.PROGRAM_CONTENT_TYPE));
        expList.add(new LanguageExpression(Operator.EQ, context.getLang()));
        expList.add(new StringExpression(ProgramItem.CATALOG, Operator.EQ, context.getCatalog()));
        expList.add(new StringExpression(AbstractProgram.DEGREE, Operator.EQ, degreeId));
        expList.add(new StringExpression(AbstractProgram.MENTION, Operator.EQ, mentionId));

        AndExpression andExp = new AndExpression(expList.toArray(new Expression[expList.size()]));
        String xPathQuery = ContentQueryHelper.getContentXPathQuery(andExp);

        AmetysObjectIterable<Program> contents = _resolver.query(xPathQuery);
        
        if (contents.getSize() > 0)
        {
            return contents.iterator().next();
        }
        
        return null;
    }
    
    /**
     * Create the mention program
     * @param contentElement the content element
     * @param mention the mention
     * @param syncCode the content synchronization code
     * @param context the import context
     * @return the created mention
     */
    protected Program _createMention(Element contentElement, OdfReferenceTableEntry mention, String syncCode, ImportCDMFrContext context)
    {
        String contentTitle = mention.getLabel(context.getLang());
        ContentWorkflowDescription wfDescription = ContentWorkflowDescription.PROGRAM_WF_DESCRIPTION;
        
        try
        {
            Program mentionProgram = (Program) _createContent(wfDescription, contentTitle, context);
            if (mentionProgram != null)
            {
                try
                {
                    // Set the SCC property
                    _sccHelper.updateSCCProperty(mentionProgram, context.getSCC().getId());
                    mentionProgram.saveChanges();
                    
                    // Extract the values
                    ModelAwareValuesExtractor valuesExtractor = _valuesExtractorFactory.getMentionValuesExtractor(contentElement, this, context);
                    View mentionView = _createMentionView();
                    Map<String, Object> values = valuesExtractor.extractValues(mentionView);
                    // Add mention
                    values.put(AbstractProgram.MENTION, mention.getId());

                    // Synchronize the content using the extracted values
                    _editContent(mentionProgram, Optional.empty(), values, true, Set.of(), context);
                }
                catch (Exception e)
                {
                    _nbError++;
                    context.getLogger().error("Failed to synchronize data for mention {} and language {}.", mentionProgram, context.getLang(), e);
                }
            }
            return mentionProgram;
        }
        catch (WorkflowException e)
        {
            context.getLogger().error("Failed to initialize workflow for content {} and language {}", contentTitle, context.getLang(), e);
            _nbError++;
            return null;
        }
    }
    
    /**
     * Create the view to use to extract values for mention creation
     * @return the created view
     */
    protected View _createMentionView()
    {
        Set<String> itemPaths = new HashSet<>(_mentionAttributePaths);
        itemPaths.add(AbstractProgram.DEGREE);
        itemPaths.add(AbstractProgram.DOMAIN);
        
        return View.of(_mentionContentType, itemPaths.toArray(new String[itemPaths.size()]));
    }
    
    /**
     * Import or synchronized shared subPrograms
     * @param mentionProgram the mention program
     * @param contentElement the content element
     * @param title the title of the subprogram
     * @param syncCode the synchronisation code
     * @param context the import context
     * @return the imported of synchronized content
     */
    protected ModifiableContent _importOrSynchronizeSharedSubProgram(Program mentionProgram, Element contentElement, String title, String syncCode, ImportCDMFrContext context)
    {
        // Search for the imported sub program
        ModifiableContent subProgram = getContent(ContentWorkflowDescription.SUBPROGRAM_WF_DESCRIPTION.getContentType(), syncCode, context);
        
        // Search for a subprogram with the same title.
        SubProgram mainSubProgram = _getSameSubProgram(mentionProgram, contentElement, context);
        
        boolean create = false;
        
        if (subProgram != null || mainSubProgram == null)
        {
            // The imported sub program already exists or there is no main subProgram yet
            // Import or synchronize the sub program, it is the main sub program
            mainSubProgram = (SubProgram) super.importOrSynchronizeContent(contentElement, ContentWorkflowDescription.SUBPROGRAM_WF_DESCRIPTION, title, syncCode, context);
            
            if (subProgram == null)
            {
                create = true;
            }
        }
        
        View mergeView = _createMergeView(_attributePathsToMerge, context);
        String sharedSubProgramCode = _xPathProcessor.evaluateAsString(contentElement, ProgramItem.CODE);
        Map<String, Object> valuesToMerge = _extractValuesToMerge(contentElement, mergeView, sharedSubProgramCode, context);
        
        try
        {
            // Synchronize the attributes to merge on a sub program node under the main sub program
            _synchronizeAttributesToMergeOnSharedSubProgramNode(mainSubProgram, sharedSubProgramCode, mergeView, valuesToMerge, context);

            // Get the merge of all shared sub programs' values
            Map<String, Object> mergedValues = _getMergedAttributes(mainSubProgram, mergeView);
         
            // Modify the main sub program with the merged values
            _editContent(mainSubProgram, Optional.of(mergeView), mergedValues, create, Set.of(), context);
        }
        catch (WorkflowException e)
        {
            _nbError++;
            context.getLogger().warn("Impossible de synchronizer les attributs partagés pour le parcours principal '" + mainSubProgram + "'");
        }
        
        return mainSubProgram;
    }
    
    /**
     * Search for a sub program corresponding to the one represented by the content element
     * @param mentionProgram the mention program
     * @param contentElement the content element
     * @param context the import context
     * @return the corresponding subProgram, or <code>null</code> if none is found
     */
    protected SubProgram _getSameSubProgram(Program mentionProgram, Element contentElement, ImportCDMFrContext context)
    {
        String contentTitle = _xPathProcessor.evaluateAsString(contentElement, "title");
        
        for (ProgramPart child : mentionProgram.getProgramPartChildren())
        {
            if (child instanceof SubProgram)
            {
                SubProgram subProgram  = (SubProgram) child;
                if (_getNormalizeTitle(subProgram.getTitle()).equals(_getNormalizeTitle(contentTitle)))
                {
                    return subProgram;
                }
            }
        }
        
        // No corresponding subprogram found
        return null;
    }
    
    private String _getNormalizeTitle(String title)
    {
        // To lower case
        // then remove accents
        return Normalizer.normalize(title.toLowerCase(), Normalizer.Form.NFD).replaceAll("[\\p{InCombiningDiacriticalMarks}]", "");
    }
    
    @Override
    protected Map<String, Object> _getAdditionalValuesToSynchronize(ModifiableContent content, String syncCode, ImportCDMFrContext context)
    {
        Map<String, Object> additionalValues = super._getAdditionalValuesToSynchronize(content, syncCode, context);
        
        if (_mentionId != null && content instanceof SubProgram && syncCode.equals(_programToLinkCode))
        {
            additionalValues.put(ProgramPart.PARENT_PROGRAM_PARTS, new String[] {_mentionId});
        }
        
        return additionalValues;
    }
    
    @Override
    protected Set<String> _getNotSynchronizedRelatedContentIds(ModifiableContent content, String syncCode, ImportCDMFrContext context)
    {
        Set<String> contentIds = super._getNotSynchronizedRelatedContentIds(content, syncCode, context);
        
        if (_mentionId != null && content instanceof SubProgram && syncCode.equals(_programToLinkCode))
        {
            contentIds.add(_mentionId);
        }
        
        return contentIds;
    }
    
    /**
     * Creates a view with attributes to merge
     * @param attributePathsToMerge the attributes to put in the view
     * @param context the import context
     * @return the created view
     */
    protected View _createMergeView(Set<String> attributePathsToMerge, ImportCDMFrContext context)
    {
        Set<String> attributePaths = new HashSet<>();
        for (String attributePathToMerge : attributePathsToMerge)
        {
            if (isValidAttributePath(_subProgramContentType, attributePathToMerge, context))
            {
                attributePaths.add(attributePathToMerge);
            }
        }
        
        return View.of(_subProgramContentType, attributePaths.toArray(new String[attributePaths.size()]));
    }
    
    /**
     * Determines if the attribute path is a valid path from the given content type and eligible to merge
     * An attribute is considered as valid if
     * <ul>
     * <li>it is part of the content type</li>
     * <li>it is multiple (to be able to merge all data from main and shared subprograms)</li>
     * </ul>
     * @param contentType The content type
     * @param attributePath The attribute path
     * @param context the import context
     * @return <code>true</code> if the attribute can be merged, <code>false</code> otherwise
     */
    protected boolean isValidAttributePath (ContentType contentType, String attributePath, ImportCDMFrContext context)
    {
        Logger logger = context.getLogger();
        
        if (!contentType.hasModelItem(attributePath))
        {
            logger.warn("L'attribut '{}' n'existe pas. il sera ignoré lors de la fusion de parcours partagés.", attributePath);
            return false;
        }
        
        ModelItem modelItem = contentType.getModelItem(attributePath);
        if (
                modelItem instanceof CompositeDefinition
                || modelItem instanceof ElementDefinition elementDefinition && !elementDefinition.isMultiple()
        )
        {
            logger.warn("L'attribut '{}' n'est pas multiple. il sera ignoré lors de la fusion de parcours partagés.", attributePath);
            return false;
        }
        
        return true;
    }
    
    /**
     * Extract the values to merge with the main program
     * @param contentElement the content element
     * @param mergeView the view containing the attributes to merge
     * @param sharedSubProgramCode the shared sub program code
     * @param context the import context
     * @return the extracted values
     */
    protected Map<String, Object> _extractValuesToMerge(Element contentElement, View mergeView, String sharedSubProgramCode, ImportCDMFrContext context)
    {
        try
        {
            ModelAwareValuesExtractor valuesExtractor = _valuesExtractorFactory.getSharedSubProgramValuesExtractor(contentElement, this, context);
            return valuesExtractor.extractValues(mergeView);
        }
        catch (Exception e)
        {
            _nbError++;
            context.getLogger().error("Failed to extract values to merge from content {} and language {}.", sharedSubProgramCode, context.getLang(), e);
            return Map.of();
        }
    }
    
    /**
     * Synchronize the attributes to merge on the shared sub program data
     * @param mainSubProgram the main sub program
     * @param sharedSubProgramCode the code of the shared sub program
     * @param mergeView the view containing the attributes to merge
     * @param valuesToMerge the values to merge
     * @param context the import context
     */
    protected void _synchronizeAttributesToMergeOnSharedSubProgramNode(SubProgram mainSubProgram, String sharedSubProgramCode, View mergeView, Map<String, Object> valuesToMerge, ImportCDMFrContext context)
    {
        // Convert values for synchronization
        Map<String, Object> convertedValues = DataHolderHelper.convertValues(mergeView, valuesToMerge);
        
        ModifiableRepositoryData rootRepositoryData = _getSharedSubProgramRootRepositoryData(mainSubProgram);
        ModifiableModelAwareDataHolder sharedDataHolder = _getSharedSubProgramDataHolder(rootRepositoryData, mainSubProgram, sharedSubProgramCode);
        sharedDataHolder.synchronizeValues(mergeView, convertedValues, SynchronizationContext.newInstance());
    }
    
    /**
     * Retrieves the merged attributes of all sub programs shared with the given main sub program
     * @param mainSubProgram the main sub program
     * @param mergeView the view containing the attributes to merge
     * @return the merged values
     */
    protected Map<String, Object> _getMergedAttributes(SubProgram mainSubProgram, View mergeView)
    {
        Collection<ModelAwareDataHolder> sharedSubPrograms = new ArrayList<>();
        ModifiableRepositoryData rootRepositoryData = _getSharedSubProgramRootRepositoryData(mainSubProgram);
        for (String subProgramCode : rootRepositoryData.getDataNames(StringUtils.EMPTY))
        {
            ModelAwareDataHolder sharedSubProgram = _getSharedSubProgramDataHolder(rootRepositoryData, mainSubProgram, subProgramCode);
            sharedSubPrograms.add(sharedSubProgram);
        }
        
        return _getMergedAttributes(sharedSubPrograms, mergeView);
    }
    
    private Map<String, Object> _getMergedAttributes(Collection<? extends ModelAwareDataHolder> dataHolders, ViewItemContainer viewItemContainer)
    {
        Map<String, Object> results = new HashMap<>();
        
        ViewHelper.visitView(viewItemContainer,
            (element, definition) -> {
                // simple element
                String name = definition.getName();
                Set<Object> result = new LinkedHashSet<>();
                
                for (ModelAwareDataHolder dataHolder : dataHolders)
                {
                    if (dataHolder.hasValue(name))
                    {
                        // Only multiple values are allowed, the value has to be an array
                        Object values = dataHolder.getValue(name);
                        
                        // Add each value to the result avoiding duplicates
                        for (int i = 0; i < Array.getLength(values); i++)
                        {
                            Object value = Array.get(values, i);
                            if (!_containsValue(definition, result, value))
                            {
                                result.add(value);
                            }
                        }
                    }
                }

                results.put(name, result);
            },
            (group, definition) -> {
                // composite
                String name = definition.getName();
                
                Collection<ModelAwareComposite> composites = new ArrayList<>();
                for (ModelAwareDataHolder dataHolder : dataHolders)
                {
                    if (dataHolder.hasValue(name))
                    {
                        ModelAwareComposite composite = dataHolder.getComposite(name);
                        composites.add(composite);
                    }
                }
                
                results.put(name, _getMergedAttributes(composites, group));
            },
            (group, definition) -> {
                // repeater
                String name = definition.getName();
                
                List<Map<String, Object>> result = new ArrayList<>();
                for (ModelAwareDataHolder dataHolder : dataHolders)
                {
                    if (dataHolder.hasValue(name))
                    {
                        ModelAwareRepeater repeater = dataHolder.getRepeater(name);
                        for (ModelAwareRepeaterEntry repeaterEntry : repeater.getEntries())
                        {
                            Map<String, Object> repeaterEntryAsMap = repeaterEntry.dataToMap();
                            result.add(repeaterEntryAsMap);
                        }
                    }
                }
                
                results.put(name, result);
            },
            group -> results.putAll(_getMergedAttributes(dataHolders, group)));
        
        return results;
    }
    
    private boolean _containsValue(ElementDefinition definition, Set<Object> values, Object valueToCheck)
    {
        ElementType type = definition.getType();
        
        for (Object value : values)
        {
            // Use type#compareValues method in order to detect duplicates for all possible types
            if (type.compareValues(value, valueToCheck).count() <= 0)
            {
                // The value to check is the same as the current value
                return true;
            }
        }
        
        // No match found, the given Set does not contain a value corresponding to the value to check
        return false;
    }
    
    /**
     * Get the shared program repository data holding the shared subprogram data
     * @param mainSubProgram the main program
     * @return root repository data
     */
    protected ModifiableRepositoryData _getSharedSubProgramRootRepositoryData(SubProgram mainSubProgram)
    {
        ModifiableRepositoryData repositoryData = mainSubProgram.getRepositoryData();
        if (repositoryData.hasValue(SHARED_PROGRAMS_NODE_NAME, RepositoryConstants.NAMESPACE_PREFIX_INTERNAL))
        {
            return repositoryData.getRepositoryData(SHARED_PROGRAMS_NODE_NAME, RepositoryConstants.NAMESPACE_PREFIX_INTERNAL);
        }
        else
        {
            mainSubProgram.setLockInfoOnCurrentContext();
            return repositoryData.addRepositoryData(SHARED_PROGRAMS_NODE_NAME, RepositoryConstants.COMPOSITE_METADTA_NODETYPE, RepositoryConstants.NAMESPACE_PREFIX_INTERNAL);
        }
    }
    
    /**
     * Get the data holder for the shared attributes for subprogram of given code
     * @param rootRepositoryData the repository data containing the values of all shared sub programs
     * @param mainSubProgram the main sub program
     * @param subProgramCode the sub program code
     * @return the shared data holder
     */
    protected ModifiableModelAwareDataHolder _getSharedSubProgramDataHolder(ModifiableRepositoryData rootRepositoryData, SubProgram mainSubProgram, String subProgramCode)
    {
        ModifiableRepositoryData repositoryData;
        if (rootRepositoryData.hasValue(subProgramCode, StringUtils.EMPTY))
        {
            repositoryData = rootRepositoryData.getRepositoryData(subProgramCode, StringUtils.EMPTY);
        }
        else
        {
            mainSubProgram.setLockInfoOnCurrentContext();
            repositoryData = rootRepositoryData.addRepositoryData(subProgramCode, RepositoryConstants.COMPOSITE_METADTA_NODETYPE, StringUtils.EMPTY);
        }
        
        ContentType contentType = _contentTypeEP.getExtension(SubProgramFactory.SUBPROGRAM_CONTENT_TYPE);
        return new DefaultModifiableModelAwareDataHolder(repositoryData, contentType, Optional.of(mainSubProgram));
    }
    
    /**
     * Update mention attributes depending on the sub program
     * @param mention the mention
     * @param subProgram the subProgram
     * @param context the import context
     */
    protected void _updateMentionAttributes(Program mention, SubProgram subProgram, ImportCDMFrContext context)
    {
        // Mettre les composantes de la mention à jour
        List<String> orgUnits = mention.getOrgUnits();
        List<String> orgUnitsToAdd = subProgram.getOrgUnits();
        if (!CollectionUtils.containsAll(orgUnits, orgUnitsToAdd))
        {
            try
            {
                View view = View.of(_mentionContentType, ProgramItem.ORG_UNITS_REFERENCES);
    
                List<String> newOrgUnitIds = ListUtils.union(orgUnits, orgUnitsToAdd);
                Map<String, Object> values = new HashMap<>();
                values.put(ProgramItem.ORG_UNITS_REFERENCES, newOrgUnitIds.toArray(new String[newOrgUnitIds.size()]));
                
                // Synchronize the content using the extracted values
                _editContent(mention, Optional.of(view), values, false, Set.of(), context);
            }
            catch (WorkflowException e)
            {
                context.getLogger().error("Failed to synchronize data for mention {} and language {}.", mention, context.getLang(), e);
                _nbError++;
            }
        }
    }
    
    private static class CoAccreditedRemoteImportException extends Throwable
    {
        private String _message;
        private Object[] _arguments;
        
        protected CoAccreditedRemoteImportException(String message, Object... arguments)
        {
            super();
            _message = message;
            _arguments = arguments;
        }
        
        @Override
        public String getMessage()
        {
            return _message;
        }
        
        protected Object[] getArguments()
        {
            return _arguments;
        }
    }
}
