/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.cdmfr.extractor;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.w3c.dom.Element;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.odf.course.Course;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.program.TraversableProgramPart;
import org.ametys.plugins.odfsync.cdmfr.ImportCDMFrContext;
import org.ametys.plugins.odfsync.cdmfr.components.ImportCDMFrComponent;
import org.ametys.plugins.odfsync.utils.ContentWorkflowDescription;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.extractor.ModelAwareValuesExtractor;

/**
 * Factory for {@link ImportCDMFrValuesExtractor}
 * Creates the right instance of {@link ImportCDMFrValuesExtractor} according to the imported content
 * Provides components to the values extractor
 */
public class ImportCDMFrValuesExtractorFactory implements Component, Serviceable
{
    /** The component role. */
    public static final String ROLE = ImportCDMFrValuesExtractorFactory.class.getName();
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The content type extension point */
    protected ContentTypeExtensionPoint _contentTypeExtensionPoint;
    
    /** The ODF TableRef Helper */
    private OdfReferenceTableHelper _odfRefTableHelper;

    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _contentTypeExtensionPoint = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _odfRefTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
    }
    
    /**
     * Retrieves the values extractor corresponding to the given content
     * @param contentElement the DOM content element
     * @param component The import CDM-fr component
     * @param content The content to synchronize
     * @param contentType The content type
     * @param syncCode The synchronization code
     * @param context the import context
     * @return the values extractor corresponding to the given content
     */
    public ModelAwareValuesExtractor getValuesExtractor(Element contentElement, ImportCDMFrComponent component, Content content, ContentType contentType, String syncCode, ImportCDMFrContext context)
    {
        if (content instanceof TraversableProgramPart)
        {
            return new ImportTraversableProgramPartValuesExtractor(contentElement, this, component, syncCode, context, contentType);
        }
        else if (content instanceof CourseList)
        {
            return new ImportCourseListValuesExtractor(contentElement, this, component, context, contentType);
        }
        else if (content instanceof Course)
        {
            return new ImportCourseValuesExtractor(contentElement, this, component, syncCode, context, contentType);
        }
        else if (content instanceof CoursePart)
        {
            return new ImportCoursePartValuesExtractor(contentElement, this, component, context, contentType);
        }
        else
        {
            return new ImportCDMFrValuesExtractor(contentElement, this, component, context, contentType);
        }
    }
    
    /**
     * Retrieves the values extractor for mentions
     * @param contentElement the DOM content element
     * @param component The import CDM-fr component
     * @param context the import context
     * @return the values extractor corresponding to the given content
     */
    public ModelAwareValuesExtractor getMentionValuesExtractor(Element contentElement, ImportCDMFrComponent component, ImportCDMFrContext context)
    {
        ContentType contentType = _contentTypeExtensionPoint.getExtension(ContentWorkflowDescription.PROGRAM_WF_DESCRIPTION.getContentType());
        return new ImportCDMFrValuesExtractor(contentElement, this, component, context, contentType);
    }
    
    /**
     * Retrieves the values extractor for mentions
     * @param contentElement the DOM content element
     * @param component The import CDM-fr component
     * @param context the import context
     * @return the values extractor corresponding to the given content
     */
    public ModelAwareValuesExtractor getSharedSubProgramValuesExtractor(Element contentElement, ImportCDMFrComponent component, ImportCDMFrContext context)
    {
        ContentType contentType = _contentTypeExtensionPoint.getExtension(ContentWorkflowDescription.SUBPROGRAM_WF_DESCRIPTION.getContentType());
        return new ImportCDMFrValuesExtractor(contentElement, this, component, context, contentType);
    }
    
    /**
     * Retrieves the ametys object resolver
     * @return the ametys object resolver
     */
    public AmetysObjectResolver getAmetysObjectResolver()
    {
        return _resolver;
    }
    
    /**
     * Retrieves the content type extension point
     * @return the content type extension point
     */
    public ContentTypeExtensionPoint getContentTypeExtensionPoint()
    {
        return _contentTypeExtensionPoint;
    }

    /**
     * Retrieves the {@link OdfReferenceTableHelper}
     * @return the {@link OdfReferenceTableHelper}
     */
    public OdfReferenceTableHelper getODFReferenceTableHelper()
    {
        return _odfRefTableHelper;
    }
}
