/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.odfsync.export;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.util.I18nUtils;
import org.ametys.odf.enumeration.OdfReferenceTableHelper;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * The abstract class to handle an export in a connector (Pégase, Apogée...)
 */
public abstract class AbstractExportStructure extends AbstractLogEnabled implements Serviceable
{
    /** The i18n utils */
    protected I18nUtils _i18nUtils;
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The ODF reference table helper */
    protected OdfReferenceTableHelper _odfRefTableHelper;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _odfRefTableHelper = (OdfReferenceTableHelper) manager.lookup(OdfReferenceTableHelper.ROLE);
    }
    
    /**
     * Get the code of the container nature
     * @param container the container
     * @return the code of the nature
     */
    protected String getContainerNatureCode(Container container)
    {
        String nature = container.getNature();
        
        if (StringUtils.isNotBlank(nature))
        {
            return _odfRefTableHelper.getItemCode(nature);
        }
        
        return null;
    }
    
    /**
     * Check if the program structure is matching
     * @param program the program
     * @param report the export report
     */
    public abstract void checkProgram(Program program, ExportReport report);
    
    /**
     * Create a program through the export connector 
     * @param program the program to export
     * @param report the export report
     */
    public abstract void createProgram(Program program, ExportReport report);
}
