/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.apogee;

import java.util.List;
import java.util.Map;

/**
 * Data access object to the remote DB Apogee
 */
public class ApogeeDAO extends AbstractApogeeDAO
{
    /** Avalon ROLE */
    public static final String ROLE = ApogeeDAO.class.getName();

    /**
     * Returns Apogee programs matching the search criteria
     * @param dataSourceId The id of data source. Can not be null.
     * @param sccParams the parameters values of the Apogee synchronizable collection
     * @param params The parameters for search criteria: DIP code, VDI code, title
     * @return the list Apogee course lists matching the search criteria
     */
    public List<Map<String, Object>> searchPrograms(String dataSourceId, Map<String, Object> sccParams, Map<String, Object> params)
    {
        return _executeSearch(dataSourceId, sccParams, params, "Apogee.searchPrograms");
    }
    
    /**
     * Returns joint org units linked to a program
     * @param dataSourceId The id of data source. Can not be null.
     * @param sccParams the parameters values of the Apogee synchronizable collection
     * @param params The parameters for search criteria: Program ID
     * @return The list of Apogee joint org units matching the program
     */
    public List<Map<String, Object>> getJointOrgUnits(String dataSourceId, Map<String, Object> sccParams, Map<String, Object> params)
    {
        return _executeSearch(dataSourceId, sccParams, params, "Apogee.getJointOrgUnits");
    }

    /**
     * Returns form of teaching linked to a program
     * @param dataSourceId The id of data source. Can not be null.
     * @param sccParams the parameters values of the Apogee synchronizable collection
     * @param params The parameters for search criteria: Program ID
     * @return The list of Apogee form of teaching matching the program
     */
    public List<Map<String, Object>> getFormOfTeachingOrg(String dataSourceId, Map<String, Object> sccParams, Map<String, Object> params)
    {
        return _executeSearch(dataSourceId, sccParams, params, "Apogee.getFormOfTeachingOrg");
    }
    
    /**
     * Returns the add elements (rich texts) to a program
     * @param dataSourceId The id of data source. Can not be null.
     * @param sccParams the parameters values of the Apogee synchronizable collection
     * @param params The parameters for search criteria: Program ID
     * @return The list of Apogee add elements matching the program
     */
    public List<Map<String, Object>> getAddElements(String dataSourceId, Map<String, Object> sccParams, Map<String, Object> params)
    {
        return _executeSearch(dataSourceId, sccParams, params, "Apogee.getAddElements");
    }
    
    /**
     * Returns the domains to a program
     * @param dataSourceId The id of data source. Can not be null.
     * @param sccParams the parameters values of the Apogee synchronizable collection
     * @param params The parameters for search criteria: Program ID
     * @return The list of Apogee domains matching the program
     */
    public List<Map<String, Object>> getDomains(String dataSourceId, Map<String, Object> sccParams, Map<String, Object> params)
    {
        return _executeSearch(dataSourceId, sccParams, params, "Apogee.getDomains");
    }
    
    /**
     * Returns Apogee organizational units matching the search criteria
     * @param dataSourceId The id of data source. Can not be null.
     * @param sccParams the parameters values of the Apogee synchronizable collection
     * @param params The parameters for search criteria: cod_cmp, cod_tpc, title
     * @return The list of Apogee organizational units matching the search criteria
     */
    public List<Map<String, Object>> searchOrgUnits(String dataSourceId, Map<String, Object> sccParams, Map<String, Object> params)
    {
        return _executeSearch(dataSourceId, sccParams, params, "Apogee.searchOrgUnits");
    }
    
    /**
     * Returns Apogee containers matching the search criteria
     * @param dataSourceId The id of data source. Can not be null.
     * @param sccParams the parameters values of the Apogee synchronizable collection
     * @param params The parameters for search criteria: code, type, title
     * @return The list of Apogee containers matching the search criteria
     */
    public List<Map<String, Object>> searchContainers(String dataSourceId, Map<String, Object> sccParams, Map<String, Object> params)
    {
        return _executeSearch(dataSourceId, sccParams, params, "Apogee.searchContainers");
    }
    
    /**
     * Returns Apogee subprograms matching the search criteria
     * @param dataSourceId The id of data source. Can not be null.
     * @param sccParams the parameters values of the Apogee synchronizable collection
     * @param params The parameters for search criteria: "dip", "vdi", "vdi-ter", "etp-code", "vet-code", "vet-title"
     * @return The list of Apogee subprograms matching the search criteria
     */
    public List<Map<String, Object>> searchSubPrograms(String dataSourceId, Map<String, Object> sccParams, Map<String, Object> params)
    {
        return _executeSearch(dataSourceId, sccParams, params, "Apogee.searchSubPrograms");
    }
    
    /**
     * Returns the Apogee educational elements matching the search criteria
     * @param dataSourceId The id of data source. Can not be null.
     * @param sccParams the parameters values of the Apogee synchronizable collection
     * @param params The parameters for search criteria: ELP code (code), ELP type (type), title (title)
     * @return the list of Apogee educational elements matching the search criteria
     */
    public List<Map<String, Object>> searchCourses(String dataSourceId, Map<String, Object> sccParams, Map<String, Object> params)
    {
        return _executeSearch(dataSourceId, sccParams, params, "Apogee.searchCourses");
    }
    
    /**
     * Returns the course parts (hours) to a course
     * @param dataSourceId The id of data source. Can not be null.
     * @param sccParams the parameters values of the Apogee synchronizable collection
     * @param params The parameters for search criteria: Course ID
     * @return The list of Apogee course parts matching the course
     */
    public List<Map<String, Object>> searchCourseParts(String dataSourceId, Map<String, Object> sccParams, Map<String, Object> params)
    {
        return _executeSearch(dataSourceId, sccParams, params, "Apogee.searchCourseParts");
    }
    
    /**
     * Returns the Apogee course lists matching the search criteria
     * @param dataSourceId The id of data source. Can not be null.
     * @param sccParams the parameters values of the Apogee synchronizable collection
     * @param params The parameters for search criteria: ELP code (code), NEL code (cod_nel), title (title)
     * @return the list Apogee course lists matching the search criteria
     */
    public List<Map<String, Object>> searchCourseLists(String dataSourceId, Map<String, Object> sccParams, Map<String, Object> params)
    {
        return _executeSearch(dataSourceId, sccParams, params, "Apogee.searchCourseLists");
    }
}
