/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.apogee.ws.structure;

import java.util.List;

import org.ametys.odf.course.Course;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramPart;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.odfsync.export.ExportReport;
import org.ametys.plugins.odfsync.export.ExportReport.ExportStatus;

import gouv.education.apogee.commun.client.ws.creationse.CreationSEMetierServiceInterface;

/**
 * The structure to export in Apogee the following program
 * <br>Program into DIP-VDI
 */
public class ApogeeSingleProgramStructure extends AbstractApogeeStructure
{
    @Override
    public void checkProgram(Program program, ExportReport report)
    {
        // Check mandatory data for program
        checkMandatoryDataForContent(program, getDIPMandatoryData(program), report);
        checkMandatoryDataForContent(program, getVDIMandatoryData(program), report);
        
        // Check mandatory data for program orgUnits
        checkMandatoryDataForOrgunits(program, program.getOrgUnits(), getOrgUnitMandatoryDataForDIP(), report);
        
        // Check the program structure
        List<ProgramPart> programPartChildren = program.getProgramPartChildren();
        if (!programPartChildren.isEmpty())
        {
            // The structure is not handled by this export
            report.setStatus(ExportStatus.CONTENT_STRUCTURE_INVALID);
        }
    }
    
    @Override
    public void checkSubProgram(SubProgram subProgram, ExportReport report)
    {
        throw new UnsupportedOperationException("No subprogram in this structure");
    }
    
    @Override
    public void checkContainerAsYear(Container container, ExportReport report, String containerNatureCode)
    {
        throw new UnsupportedOperationException("No year container in this structure");
    }
    
    @Override
    public void checkCourseList(CourseList courseList, ExportReport report)
    {
        throw new UnsupportedOperationException("No course list in this structure");
    }
    
    @Override
    public void checkCourse(Course course, ExportReport report)
    {
        throw new UnsupportedOperationException("No course in this structure");
    }
    
    @Override
    public void createProgram(Program program, ExportReport report)
    {
        try
        {
            CreationSEMetierServiceInterface creationService = _apogeeWS.getCreationService();
            String codDIP = getCodeApogee(program);
            Long versionDIP = getVersionApogee(program);
            _apogeeWS.createDIP(program, null, codDIP, creationService);
            _apogeeWS.createVDI(program, null, codDIP, versionDIP, creationService);
        }
        catch (Exception e)
        {
            report.setStatus(ExportStatus.ERROR);
            getLogger().error("An error occurred exporting the program '{}' ({}) in Apogee", program.getTitle(), program.getId(), e);
        }
    }
}
