/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.odfsync.export;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.clientsideelement.AbstractContentClientSideElement;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.odfsync.export.ExportReport.ExportStatus;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Client side element for connector export buttons
 */
public abstract class AbstractExportClientSideElement extends AbstractContentClientSideElement implements Initializable
{
    /** The avalon service manager */
    protected ServiceManager _manager;
    
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** The content type extension point */
    protected ContentTypeExtensionPoint _ctExtPoint;
    
    /** The structure component (depending of the current implementation */
    protected AbstractStructureComponent _structureComponent;
    
    private boolean _isActive;
    
    public void initialize() throws Exception
    {
        _isActive = Config.getInstance().getValue(getActivateParam(), true, false);
    }
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _manager = smanager;
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _ctExtPoint = (ContentTypeExtensionPoint) smanager.lookup(ContentTypeExtensionPoint.ROLE);
        _structureComponent = (AbstractStructureComponent) _manager.lookup(getStructureComponentRole());
    }
    
    @Override
    public List<Script> getScripts(boolean ignoreRights, Map<String, Object> contextParameters)
    {
        if (_isActive)
        {
            return super.getScripts(ignoreRights, contextParameters);
        }
        return new ArrayList<>();
    }

    /**
     * Get the name of the configuration parameter to check to associated buttons.
     * @return The name of the configuration parameter
     */
    protected abstract String getActivateParam();
    
    /**
     * Get the role of the structure component, depending of the implementation.
     * @return The role of the structure component
     */
    protected abstract String getStructureComponentRole();
    
    /**
     * Get the report informations of the export of the program id
     * @param programId the program id to export
     * @return the export results
     */
    public Map<String, Object> getExportReportInfo(String programId)
    {
        Map<String, Object> results = new HashMap<>();
        
        Program program = _resolver.resolveById(programId);
        if (hasRight(program))
        {
            ExportReport report = _structureComponent.getExportReport(program);
            ExportStatus exportStatus = report.getStatus();
            
            results.put("status", exportStatus.name());
            if (exportStatus == ExportStatus.CONTENT_DATA_INVALID)
            {
                Map<String, Object> programReport = new HashMap<>();
                Map<String, Object> subProgramReport = new HashMap<>();
                Map<String, Object> containerReport = new HashMap<>();
                Map<String, Object> courseListReport = new HashMap<>();
                Map<String, Object> courseReport = new HashMap<>();
                Map<String, Object> orgUnitReport = new HashMap<>();
                Map<String, Object> otherContentReport = new HashMap<>();
                
                Map<Content, Set<I18nizableText>> invalidDataByContent = report.getInvalidDataPathByContent();
                for (Content content : invalidDataByContent.keySet())
                {
                    if (content instanceof Program)
                    {
                        programReport.put(content.getId(), _getContentParams(content, invalidDataByContent.get(content)));
                    }
                    else if (content instanceof SubProgram)
                    {
                        subProgramReport.put(content.getId(), _getContentParams(content, invalidDataByContent.get(content)));
                    }
                    else if (content instanceof Container)
                    {
                        containerReport.put(content.getId(), _getContentParams(content, invalidDataByContent.get(content)));
                    }
                    else if (content instanceof CourseList)
                    {
                        courseListReport.put(content.getId(), _getContentParams(content, invalidDataByContent.get(content)));
                    }
                    else if (content instanceof Course)
                    {
                        courseReport.put(content.getId(), _getContentParams(content, invalidDataByContent.get(content)));
                    }
                    else if (content instanceof OrgUnit)
                    {
                        orgUnitReport.put(content.getId(), _getContentParams(content, invalidDataByContent.get(content)));
                    }
                    else
                    {
                        otherContentReport.put(content.getId(), _getContentParams(content, invalidDataByContent.get(content)));
                    }
                }
                
                results.put("program", programReport);
                results.put("subprogram", subProgramReport);
                results.put("container", containerReport);
                results.put("courselist", courseListReport);
                results.put("course", courseReport);
                results.put("orgunit", orgUnitReport);
                results.put("other", otherContentReport);
            }
        }
        else
        {
            results.put("status", ExportStatus.ERROR.name());
        }
        
        return results;
    }
    
    /**
     * Export a program to external connector
     * @param programId the program id to export
     * @return the export results
     */
    public Map<String, Object> exportProgram(String programId)
    {
        Map<String, Object> results = new HashMap<>();
        
        Program program = _resolver.resolveById(programId);
        if (hasRight(program))
        {
            ExportReport exportReport = _structureComponent.getExportReport(program);
            if (exportReport.getStatus() == ExportStatus.OK)
            {
                exportReport.getExportStructure().createProgram(program, exportReport);
            }
            
            results.put("nbExported", exportReport.getNbExported());
            results.put("nbNotExported", exportReport.getNbNotExported());
            results.put("nbPartlyExported", exportReport.getNbPartlyExported());
            results.put("problemsEncountered", exportReport.getProblemsEncountered());
            results.put("status", exportReport.getStatus().name());
        }
        else
        {
            results.put("status", ExportStatus.ERROR.name());
        }

        results.put("contentId", program.getId());
        results.put("contentTitle", program.getTitle());
        
        return results;
    }
    
    /**
     * Get content params
     * @param content The content
     * @param invalidMessages The modelItems
     * @return parameters of the content
     */
    protected Map<String, Object> _getContentParams(Content content, Set<I18nizableText> invalidMessages)
    {
        Map<String, Object> params = new HashMap<>();
        params.put("contentTitle", content.getTitle());
        if (content instanceof ProgramItem)
        {
            params.put("contentCode", ((ProgramItem) content).getCode());
        }
        else if (content instanceof OrgUnit)
        {
            params.put("contentCode", ((OrgUnit) content).getUAICode());
        }
        else
        {
            String contentTypeId = content.getTypes()[0];
            ContentType contentType = _ctExtPoint.getExtension(contentTypeId);
            params.put("contentTypeLabel", contentType.getLabel());
            params.put("contentTypeId", contentTypeId);
            params.put("isTableRef", true);
        }
        
        params.put("attributes", invalidMessages);
        
        return params;
    }
}
