/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.apogee.ws;

import java.math.BigDecimal;
import java.rmi.RemoteException;
import java.time.LocalDate;
import java.util.List;
import java.util.stream.Stream;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.ContentDataHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.core.util.HttpUtils;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.odfsync.apogee.ws.structure.AbstractApogeeStructure;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import gouv.education.apogee.commun.client.ws.creationse.CreationSEMetierServiceInterface;
import gouv.education.apogee.commun.client.ws.creationse.CreationSEMetierServiceInterfaceServiceLocator;
import gouv.education.apogee.commun.transverse.dto.offreformation.creerse.CentreInsPedagogiDTO;
import gouv.education.apogee.commun.transverse.dto.offreformation.creerse.CentreInsPedagogiDTO2;
import gouv.education.apogee.commun.transverse.dto.offreformation.creerse.CmpHabiliteeVdiDTO;
import gouv.education.apogee.commun.transverse.dto.offreformation.creerse.ComposanteCGEDTO;
import gouv.education.apogee.commun.transverse.dto.offreformation.creerse.DiplomeDTO;
import gouv.education.apogee.commun.transverse.dto.offreformation.creerse.ElementPedagogiDTO4;
import gouv.education.apogee.commun.transverse.dto.offreformation.creerse.ElementPedagogiDTO6;
import gouv.education.apogee.commun.transverse.dto.offreformation.creerse.EtapeDTO;
import gouv.education.apogee.commun.transverse.dto.offreformation.creerse.LienVetElpLseDTO;
import gouv.education.apogee.commun.transverse.dto.offreformation.creerse.ListeElementPedagogiDTO3;
import gouv.education.apogee.commun.transverse.dto.offreformation.creerse.Nullable2Int;
import gouv.education.apogee.commun.transverse.dto.offreformation.creerse.Nullable3Int;
import gouv.education.apogee.commun.transverse.dto.offreformation.creerse.NullableDecimal;
import gouv.education.apogee.commun.transverse.dto.offreformation.creerse.NullableInt;
import gouv.education.apogee.commun.transverse.dto.offreformation.creerse.RegimeDTO;
import gouv.education.apogee.commun.transverse.dto.offreformation.creerse.VdiFractionnerVetDTO;
import gouv.education.apogee.commun.transverse.dto.offreformation.creerse.VersionDiplomeDTO;
import gouv.education.apogee.commun.transverse.dto.offreformation.creerse.VersionEtapeDTO;


/**
 * The component to create ODF element in Apogee
 * In most cases :
 * DIP / VDI (diplome / version de diplome) : it is a program
 * ETP / VET (etape / version d'etape) : it is a subprogram or a container of type year
 * LSE (liste d'elements) : it is a course list
 * ELP (element pedagogique) : it is a course or a container of type semester
 */
public class ApogeeWS extends AbstractLogEnabled implements Serviceable, Component
{
    /** Avalon ROLE */
    public static final String ROLE = ApogeeWS.class.getName();
    
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    /**
     * Get the service to create element in Apogee
     * @return the service
     * @throws javax.xml.rpc.ServiceException if an error occurred getting the service
     */
    public CreationSEMetierServiceInterface getCreationService() throws javax.xml.rpc.ServiceException
    {
        String wsURL = HttpUtils.sanitize(Config.getInstance().getValue("apogee.ws.url"));
        CreationSEMetierServiceInterfaceServiceLocator locator = new CreationSEMetierServiceInterfaceServiceLocator();
        locator.setCreationSEMetierEndpointAddress(wsURL); // TODO user and mdp ????
        
        return locator.getCreationSEMetier();
    }
    
    /**
     * Create a Apogee DIP from a content
     * @param content the content
     * @param title the DIP title. Can be null, in this case we take the content title
     * @param codDip the code DIP
     * @throws RemoteException if a web service error occurred
     * @throws javax.xml.rpc.ServiceException if an error occurred getting the service
     */
    public void createDIP(Content content, String title, String codDip) throws RemoteException, javax.xml.rpc.ServiceException
    {
        createDIP(content, title, codDip, getCreationService());
    }
    
    /**
     * Create a Apogee DIP from a content
     * @param content the content
     * @param title the DIP title. Can be null, in this case we take the content title
     * @param codDip the code DIP
     * @param creationSEMetierServiceInterface the service to create element in Apogee
     * @throws RemoteException if a web service error occurred
     */
    public void createDIP(Content content, String title, String codDip, CreationSEMetierServiceInterface creationSEMetierServiceInterface) throws RemoteException
    {
        // Code diplome
        String computedTitle = StringUtils.isNotBlank(title) ? title : content.getTitle();
        
        // codDip='Code diplôme (obligatoire)' "
        // libDip='Libellé long diplôme (non obligatoire)' "
        // licDip='Libellé court diplôme (obligatoire)' "
        // codNdi='Code nature diplôme (obligatoire)'"
        // codCycle='Code cycle SISE (obligatoire)'"
        // codTypDip='Code type diplôme (obligatoire)'"
        DiplomeDTO dip = new DiplomeDTO();
        dip.setCodDip(codDip);
        dip.setLibDip(computedTitle);
        dip.setLicDip(StringUtils.substring(computedTitle, 0, 24)); // need to cut to 25 max
        
        String educationKindId = ContentDataHelper.getContentIdFromContentData(content, AbstractProgram.EDUCATION_KIND);
        Content educationKindContent = _resolver.resolveById(educationKindId);
        dip.setCodNdi(educationKindContent.getValue(AbstractApogeeStructure.CODE_APOGEE_ATTRIBUTE_NAME));

        dip.setCodCycle(content.getValue("cycleApogee/codeApogee"));

        String degreeCodeId = ContentDataHelper.getContentIdFromContentData(content, AbstractProgram.DEGREE);
        Content degreeContent = _resolver.resolveById(degreeCodeId);
        dip.setCodTypDip(degreeContent.getValue(AbstractApogeeStructure.CODE_APOGEE_ATTRIBUTE_NAME));
        
        creationSEMetierServiceInterface.creerDIP(dip);
    }
    
    /**
     * Create a Apogee VDI from a content
     * @param content the content
     * @param title the VDI title. Can be null, in this case we take the content title
     * @param codDip the code of the diplome in Apogee
     * @param versionDIP the version of the diplome in Apogee
     * @throws RemoteException if a web service error occurred
     * @throws javax.xml.rpc.ServiceException if an error occurred getting the service
     */
    public void createVDI(Content content, String title, String codDip, Long versionDIP) throws RemoteException, javax.xml.rpc.ServiceException
    {
        createVDI(content, title, codDip, versionDIP, getCreationService());
    }
    
    /**
     * Create a Apogee VDI from a content
     * @param content the content
     * @param title the VDI title. Can be null, in this case we take the content title
     * @param codDip the code of the diplome in Apogee
     * @param versionDIP the version of the diplome in Apogee
     * @param creationSEMetierServiceInterface the service to create element in Apogee
     * @throws RemoteException if a web service error occurred
     */
    public void createVDI(Content content, String title, String codDip, Long versionDIP, CreationSEMetierServiceInterface creationSEMetierServiceInterface) throws RemoteException
    {
        String computedTitle = StringUtils.isNotBlank(title) ? title : content.getTitle();
        
        // codDip='Code diplôme (obligatoire)' "
        // codVersionDip='Code version diplôme (obligatoire)' "
        // licVersionDip='Libellé court version diplôme (non obligatoire)' "
        // debutRecrutement='Année universitaire début recrutement (obligatoire)' "
        // finRecrutement='Année universitaire fin recrutement (obligatoire)' "
        // debutValidation='Année universitaire début validation (obligatoire)' "
        // finValidation='Année universitaire fin validation (obligatoire)' "
        // listComposantesHabilitees='Code composante 1,Code composante 2,... (obligatoire)' "
        VersionDiplomeDTO vdi = new VersionDiplomeDTO();
        vdi.setCodDip(codDip);
        
        vdi.setCodVersionDip(new NullableInt(versionDIP.intValue()));
        vdi.setLicVersionDip(StringUtils.substring(computedTitle, 0, 24));
        
        LocalDate startRecruitmentDate = content.getValue("start-date-recruitment");
        LocalDate endRecruitmentDate = content.getValue("end-date-recruitment");
        vdi.setDebutRecrutement(String.valueOf(startRecruitmentDate.getYear()));
        vdi.setFinRecrutement(String.valueOf(endRecruitmentDate.getYear()));
        
        LocalDate startValidationDate = content.getValue("start-date-validation");
        LocalDate endValidationDate = content.getValue("end-date-validation");
        vdi.setDebutValidation(String.valueOf(startValidationDate.getYear()));
        vdi.setFinValidation(String.valueOf(endValidationDate.getYear()));
        
        // Create tab of orgUnits VDI
        CmpHabiliteeVdiDTO[] tabCmpHabiliteeVDIs = ContentDataHelper.getContentIdsListFromMultipleContentData(content, ProgramItem.ORG_UNITS_REFERENCES)
                    .stream()
                    .map(id -> _getOrgUnitById(id))
                    .filter(o -> o != null)
                    .map(o -> _createCmpHabiliteeVdiDTO(o))
                    .toArray(CmpHabiliteeVdiDTO[]::new);
        vdi.setListComposantesHabilitees(tabCmpHabiliteeVDIs);
        
        creationSEMetierServiceInterface.creerVDI(vdi);
    }
    
    /**
     * Create a Apogee ETP from a content
     * @param content the content
     * @param title the ETP title. Can be null, in this case we take the content title
     * @param codETP the code ETP
     * @throws RemoteException if a web service error occurred
     * @throws javax.xml.rpc.ServiceException if an error occurred getting the service
     */
    public void createETP(Content content, String title, String codETP) throws RemoteException, javax.xml.rpc.ServiceException
    {
        createETP(content, title, codETP, getCreationService());
    }
    
    /**
     * Create a Apogee ETP from a content
     * @param content the content
     * @param title the ETP title. Can be null, in this case we take the content title
     * @param codETP the code ETP
     * @param creationSEMetierServiceInterface the service to create element in Apogee
     * @throws RemoteException if a web service error occurred
     */
    public void createETP(Content content, String title, String codETP, CreationSEMetierServiceInterface creationSEMetierServiceInterface) throws RemoteException
    {
        String computedTitle = StringUtils.isNotBlank(title) ? title : content.getTitle();
        
        // codEtp='Code étape (obligatoire)' "
        // licEtp='Libellé court étape (obligatoire)' "
        // libEtp='Libellé long étape (non obligatoire)' "
        // codCycle='Code cycle SISE (obligatoire)' "
        // listComposantesCGE='[Code composante 1 (obligatoire),Code CGE 1 (obligatoire),capacité accueil 1 (non obligatoire)];[Code composante 2 (obligatoire),Code CGE 2 (obligatoire),capacité accueil 2 (non obligatoire)];... (obligatoire)' "
        EtapeDTO etp = new EtapeDTO();
        etp.setCodEtp(codETP);
        etp.setLicEtp(StringUtils.substring(computedTitle, 0, 24)); // need to cut to 25 max
        etp.setLibEtp(computedTitle);
        
        etp.setCodCycle(content.getValue("cycleApogee/codeApogee"));
        
        if (content instanceof SubProgram subProgram)
        {
            // Create tab of orgUnits CGE
            ComposanteCGEDTO[] tabComp = subProgram.getOrgUnits()
                        .stream()
                        .map(id -> _getOrgUnitById(id))
                        .filter(o -> o != null)
                        .map(o -> _createComposanteCGEDTO(o))
                        .toArray(ComposanteCGEDTO[]::new);
            etp.setListComposantesCGE(tabComp);
        }
        else if (content instanceof Container)
        {
            ContentValue value = content.getValue("orgUnit");
            ComposanteCGEDTO[] tabComp = new ComposanteCGEDTO[1];
            tabComp[0] = _createComposanteCGEDTO((OrgUnit) value.getContent());
            etp.setListComposantesCGE(tabComp);
        }
        
        creationSEMetierServiceInterface.creerETP(etp);
    }
    
    /**
     * Create a Apogee VET from a content
     * @param content the content
     * @param title the VET title. Can be null, in this case we take the content title
     * @param codETP the ETP code in Apogee
     * @param versionETP the version ETP in Apogee
     * @throws RemoteException if a web service error occurred
     * @throws javax.xml.rpc.ServiceException if an error occurred getting the service
     */
    public void createVET(Content content, String title, String codETP, Long versionETP) throws RemoteException, javax.xml.rpc.ServiceException
    {
        createVET(content, title, codETP, versionETP, getCreationService());
    }
    
    /**
     * Create a Apogee VET from a content
     * @param content the content
     * @param title the VET title. Can be null, in this case we take the content title
     * @param codETP the ETP code in Apogee
     * @param versionETP the version ETP in Apogee
     * @param creationSEMetierServiceInterface the service to create element in Apogee
     * @throws RemoteException if a web service error occurred
     */
    public void createVET(Content content, String title, String codETP, Long versionETP, CreationSEMetierServiceInterface creationSEMetierServiceInterface) throws RemoteException
    {
        String computedTitle = StringUtils.isNotBlank(title) ? title : content.getTitle();
        
        // codEtp='Code étape (obligatoire)' "
        // codVersionEtp='Code version étape (obligatoire)' "
        // libWeb='Libellé web version étape (non obligatoire)' "
        // codDureeEtp='Code durée étape' (obligatoire)' "
        // codComposante='Code composante organisatrice (obligatoire)' "
        // listRegimes='Code régime 1,Code régime 2,... (obligatoire)' "
        // listCIP='[Code centre d'inscription pédagogique 1 (obligatoire),Témoin IA avant IP 1 (non obligatoire),Témoin inscription automatique aux éléments obligatoires 1 (non obligatoire)];[Code centre d'inscription pédagogique 2 (obligatoire),Témoin ...];... (obligatoire)' "
        VersionEtapeDTO vet = new VersionEtapeDTO();
        vet.setCodEtp(codETP);
        vet.setCodVersionEtp(new NullableInt(versionETP.intValue()));
        vet.setLibWeb(computedTitle);
        
        vet.setCodDureeEtp(content.getValue("duration-apogee/codeApogee"));

        if (content instanceof SubProgram subProgram) // TODO prendre orgunit principale
        {
            String orgUnitId = subProgram.getOrgUnits().get(0);
            OrgUnit orgUnit = _getOrgUnitById(orgUnitId);
            vet.setCodComposante(orgUnit.getValue(AbstractApogeeStructure.CODE_APOGEE_ATTRIBUTE_NAME));
        }
        else if (content instanceof Container)
        {
            ContentValue value = content.getValue("orgUnit");
            vet.setCodComposante(value.getContent().getValue(AbstractApogeeStructure.CODE_APOGEE_ATTRIBUTE_NAME));
        }
        
        // Create tab of regime
        String[] inscriptionTypes = content.getValue("inscription-types/codeApogee", true);
        RegimeDTO[] tabRegimes = Stream.of(inscriptionTypes)
                    .map(it -> _createRegime(it))
                    .toArray(RegimeDTO[]::new);
        vet.setListRegimes(tabRegimes);

        // Create tab of CIP
        String[] cips = content.getValue("cips/codeApogee", true);
        CentreInsPedagogiDTO2[] tabCIP = Stream.of(cips)
                    .map(cip -> _createCIPVET(cip))
                    .toArray(CentreInsPedagogiDTO2[]::new);
        vet.setListCIP(tabCIP);

        creationSEMetierServiceInterface.creerVET(vet);
    }
    
    /**
     * Create in Apogee the link between a DIP and a ETP
     * @param codDIP the DIP code in Apogee
     * @param versionDIP the DIP version in Apogee
     * @param codETP the ETP code in Apogee
     * @param versionETP the ETP version in Apogee
     * @throws RemoteException if a web service error occurred
     * @throws javax.xml.rpc.ServiceException if an error occurred getting the service
     */
    public void createLinkDIPETP(String codDIP, Long versionDIP, String codETP, Long versionETP) throws RemoteException, javax.xml.rpc.ServiceException
    {
        createLinkDIPETP(codDIP, versionDIP, codETP, versionETP, getCreationService());
    }
    
    /**
     * Create in Apogee the link between a DIP and a ETP
     * @param codDIP the DIP code in Apogee
     * @param versionDIP the DIP version in Apogee
     * @param codETP the ETP code in Apogee
     * @param versionETP the ETP version in Apogee
     * @param creationSEMetierServiceInterface the service to create element in Apogee
     * @throws RemoteException if a web service error occurred
     */
    public void createLinkDIPETP(String codDIP, Long versionDIP, String codETP, Long versionETP, CreationSEMetierServiceInterface creationSEMetierServiceInterface) throws RemoteException
    {
        // codDip='Code diplôme (obligatoire)' "
        // codVersionDip='Code version diplôme (obligatoire)' "
        // codEtp='Code étape (obligatoire)' "
        // codVersionEtp='Code version étape (obligatoire)'"
        // anMinEtpDip='Année minimale étape pour diplôme (obligatoire)' "
        // anMaxEtpDip='Année maximale étape pour diplôme (obligatoire)' "
        VdiFractionnerVetDTO lien = new VdiFractionnerVetDTO();
        lien.setCodDip(codDIP);
        lien.setCodVersionDip(new Nullable3Int(versionDIP.intValue()));
        lien.setCodEtp(codETP);
        lien.setCodVersionEtp(new Nullable3Int(versionETP.intValue()));
        lien.setAnMinEtpDip(new NullableInt("1")); //TODO
        lien.setAnMaxEtpDip(new NullableInt("2")); //TODO

        creationSEMetierServiceInterface.creerLienVetVdi(lien);
    }
    
    /**
     * Create in Apogee the link between a ETP and a LSE or ELP
     * @param codETP the ETP code in Apogee. Can be null if codELP is set.
     * @param versionETP the ETP version in Apogee. Can be null if codELP is set.
     * @param codLSE the LSE code in Apogee.
     * @param codELP the ELP version in Apogee. Can be null if codETP and versionETP is set.
     * @param nbELP the number of ELP. Can be null if the LSE is mandatory or optional.
     * @param ectsMin the min number of ects. Can be null if the LSE is mandatory or optional.
     * @param ectsMax the max number of ects. Can be null if the LSE is mandatory or optional.
     * @throws RemoteException if a web service error occurred
     * @throws javax.xml.rpc.ServiceException if an error occurred getting the service
     */
    public void createLinkETPELPLSE(String codETP, Long versionETP, String codLSE, String codELP, Long nbELP, Double ectsMin, Double ectsMax) throws RemoteException, javax.xml.rpc.ServiceException
    {
        createLinkETPELPLSE(codETP, versionETP, codLSE, codELP, nbELP, ectsMin, ectsMax, getCreationService());
    }
    
    /**
     * Create in Apogee the link between a ETP and a LSE or ELP
     * @param codETP the ETP code in Apogee. Can be null if codELP is set.
     * @param versionETP the ETP version in Apogee. Can be null if codELP is set.
     * @param codLSE the LSE code in Apogee.
     * @param codELP the ELP version in Apogee. Can be null if codETP and versionETP is set.
     * @param nbELP the number of ELP. Can be null if the LSE is mandatory or optional.
     * @param ectsMin the min number of ects. Can be null if the LSE is mandatory or optional.
     * @param ectsMax the max number of ects. Can be null if the LSE is mandatory or optional.
     * @param creationSEMetierServiceInterface the service to create element in Apogee
     * @throws RemoteException if a web service error occurred
     */
    public void createLinkETPELPLSE(String codETP, Long versionETP, String codLSE, String codELP, Long nbELP, Double ectsMin, Double ectsMax, CreationSEMetierServiceInterface creationSEMetierServiceInterface) throws RemoteException
    {
        // codElp='Code élément pédagogique (obligatoire)' "
        // codEtp='Code étape (obligatoire)'"
        // codVersionEtp='Code version étape (obligatoire)'"
        // codListeElp='Code Liste d'éléments pédagogiques (obligatoire)'"
        // nbrMinElp='Nombre minimum d'éléments à choisir dans la liste'"
        // nbrMaxElp='Nombre maximum d'éléments à choisir dans la liste'"
        // nbrMinCredits='Nombre minimum de crédits ECTS à choisir dans la liste'"
        // nbrMaxCredits='Nombre maximum de crédits ECTS à choisir dans la liste'"
        
        LienVetElpLseDTO lien = new LienVetElpLseDTO();
        if (StringUtils.isNotBlank(codETP))
        {
            lien.setCodEtp(codETP);
        }
        
        if (versionETP != null)
        {
            lien.setCodVersionEtp(new Nullable3Int(versionETP.intValue()));
        }
        else
        {
            lien.setCodVersionEtp(new Nullable3Int());
        }

        lien.setCodListeElp(codLSE);
        lien.setCodElp(codELP);
        
        if (nbELP != null)
        {
            lien.setNbrMinElp(new Nullable2Int(nbELP.intValue()));
            lien.setNbrMaxElp(new Nullable2Int(nbELP.intValue()));
            
            lien.setNbrMinCredits(new NullableDecimal(new BigDecimal(ectsMin)));
            lien.setNbrMaxCredits(new NullableDecimal(new BigDecimal(ectsMax)));
        }
        creationSEMetierServiceInterface.creerLienVetElpLse(lien);
    }
    
    /**
     * Create a Apogee LSE from a course list
     * @param courseList the course list
     * @param title the LSE title. Can be null, in this case we take the content title
     * @param codLSE the code LSE
     * @throws RemoteException if a web service error occurred
     * @throws javax.xml.rpc.ServiceException if an error occurred getting the service
     */
    public void createLSE(CourseList courseList, String title, String codLSE) throws RemoteException, javax.xml.rpc.ServiceException
    {
        createLSE(courseList, title, codLSE, getCreationService());
    }
    
    /**
     * Create a Apogee LSE from a course list
     * @param courseList the course list
     * @param title the LSE title. Can be null, in this case we take the content title
     * @param codLSE the code LSE
     * @param creationSEMetierServiceInterface the service to create element in Apogee
     * @throws RemoteException if a web service error occurred
     */
    public void createLSE(CourseList courseList, String title, String codLSE, CreationSEMetierServiceInterface creationSEMetierServiceInterface) throws RemoteException
    {
        String computedTitle = StringUtils.isNotBlank(title) ? title : courseList.getTitle();
        
        // Création du tableau d'ELP
        int i = 0;
        List<Course> courses = courseList.getCourses();
        ElementPedagogiDTO4[] tabELP = new ElementPedagogiDTO4[courses.size()];
        for (Course course : courses)
        {
            tabELP[i] = new ElementPedagogiDTO4(course.getValue(AbstractApogeeStructure.CODE_APOGEE_ATTRIBUTE_NAME));
            i++;
        }

        // codListeElp='Code Liste d'éléments pédagogiques (obligatoire)' "
        // typListeElp='Code type de liste (obligatoire)' "
        // libCourtListeElp='Libellé court liste d'éléments pédagogiques (obligatoire)'"
        // libListeElp='Libelle long liste d'éléments pédagogiques (obligatoire)'"
        // listElementPedagogi='Code élément pédagogique,Code élément pédagogique 2,Code élément pédagogique 3,... (au moins 1 codElp, obligatoire)'"
        ListeElementPedagogiDTO3 lse = new ListeElementPedagogiDTO3();
        lse.setCodListeElp(codLSE);
        
        String choiceTypeAmetys = courseList.getValue("choiceType");
        String typList = "O";
        if (choiceTypeAmetys.equals("MANDATORY"))
        {
            typList = "F";
        }
        else if (choiceTypeAmetys.equals("CHOICE"))
        {
            typList = "X";
        }
        lse.setTypListeElp(typList);
        lse.setLibCourtListeElp(StringUtils.substring(computedTitle, 0, 24));
        lse.setLibListeElp(computedTitle);
        lse.setListElementPedagogi(tabELP);
        
        creationSEMetierServiceInterface.creerLSE(lse);
    }
    
    /**
     * Create a Apogee mandatory LSE with his ELP
     * @param titleLSE the title of the LSE
     * @param codLSE the code of the LSE
     * @param codELP the code of the ELP
     * @throws RemoteException if a web service error occurred
     * @throws javax.xml.rpc.ServiceException if an error occurred getting the service
     */
    public void createMandatoryLSE(String titleLSE, String codLSE, String codELP) throws RemoteException, javax.xml.rpc.ServiceException
    {
        createMandatoryLSE(titleLSE, codLSE, codELP, getCreationService());
    }
    
    /**
     * Create a Apogee mandatory LSE with his ELP
     * @param titleLSE the title of the LSE
     * @param codLSE the code of the LSE
     * @param codELP the code of the ELP
     * @param creationSEMetierServiceInterface the service to create element in Apogee
     * @throws RemoteException if a web service error occurred
     */
    public void createMandatoryLSE(String titleLSE, String codLSE, String codELP, CreationSEMetierServiceInterface creationSEMetierServiceInterface) throws RemoteException
    {
        // Création du tableau d'ELP
        ElementPedagogiDTO4[] tabELP = new ElementPedagogiDTO4[1];
        tabELP[0] = new ElementPedagogiDTO4(codELP);

        // codListeElp='Code Liste d'éléments pédagogiques (obligatoire)' "
        // typListeElp='Code type de liste (obligatoire)' "
        // libCourtListeElp='Libellé court liste d'éléments pédagogiques (obligatoire)'"
        // libListeElp='Libelle long liste d'éléments pédagogiques (obligatoire)'"
        // listElementPedagogi='Code élément pédagogique,Code élément pédagogique 2,Code élément pédagogique 3,... (au moins 1 codElp, obligatoire)'"
        ListeElementPedagogiDTO3 lse = new ListeElementPedagogiDTO3();
        lse.setCodListeElp(codLSE);
        
        lse.setTypListeElp("F");
        lse.setLibCourtListeElp(StringUtils.substring(titleLSE, 0, 24));
        lse.setLibListeElp(titleLSE);
        lse.setListElementPedagogi(tabELP);
        
        creationSEMetierServiceInterface.creerLSE(lse);
    }
    
    /**
     * Create a Apogee ELP from a content
     * @param content the content
     * @param title the ELP title. Can be null, in this case we take the content title
     * @param codELP the code ELP
     * @throws RemoteException if a web service error occurred
     * @throws javax.xml.rpc.ServiceException if an error occurred getting the service
     */
    public void createELP(Content content, String title, String codELP) throws RemoteException, javax.xml.rpc.ServiceException
    {
        createELP(content, title, codELP, getCreationService());
    }
    
    /**
     * Create a Apogee ELP from a content
     * @param content the content
     * @param title the ELP title. Can be null, in this case we take the content title
     * @param codELP the code ELP
     * @param creationSEMetierServiceInterface the service to create element in Apogee
     * @throws RemoteException if a web service error occurred
     */
    public void createELP(Content content, String title, String codELP, CreationSEMetierServiceInterface creationSEMetierServiceInterface) throws RemoteException
    {
        String computedTitle = StringUtils.isNotBlank(title) ? title : content.getTitle();
        
        // Create the ELP DTO
        // codElp='Code élément pédagogique (obligatoire)' "
        // libCourtElp='Libellé court élément pédagogique pédagogiques (obligatoire)'"
        // libElp='Libelle long élément pédagogique (obligatoire)'"
        // codNatureElp='Code de la nature ELP' (obligatoire)"
        // codComposante='Code composante' (obligatoire)"
        // listCentreInsPedagogi='Code CIP 1,Code CIP 2,Code CIP 3, ... (au moins 1 codCip, obligatoire)'"
        ElementPedagogiDTO6 elp = new ElementPedagogiDTO6();
        elp.setCodElp(codELP);
        elp.setLibElp(computedTitle);
        elp.setLibCourtElp(StringUtils.substring(computedTitle, 0, 24));
        
        if (content instanceof Container)
        {
            ContentValue value = content.getValue("nature");
            elp.setCodNatureElp(value.getContent().getValue(AbstractApogeeStructure.CODE_APOGEE_ATTRIBUTE_NAME));
        }
        else if (content instanceof Course)
        {
            ContentValue value = content.getValue("courseType");
            elp.setCodNatureElp(value.getContent().getValue(AbstractApogeeStructure.CODE_APOGEE_ATTRIBUTE_NAME));
        }
        
        if (content instanceof Course) // TODO prendre orgunit principale
        {
            String orgUnitId = ((Course) content).getOrgUnits().get(0);
            OrgUnit orgUnit = _getOrgUnitById(orgUnitId);
            elp.setCodComposante(orgUnit.getValue(AbstractApogeeStructure.CODE_APOGEE_ATTRIBUTE_NAME));
        }
        else if (content instanceof Container)
        {
            ContentValue value = content.getValue("orgUnit");
            elp.setCodComposante(value.getContent().getValue(AbstractApogeeStructure.CODE_APOGEE_ATTRIBUTE_NAME));
        }
        
        // Create tab of CIP
        String[] cips = content.getValue("cips/codeApogee", true);
        CentreInsPedagogiDTO[] tabCIP = Stream.of(cips)
                    .map(cip -> _createCIPELP(cip))
                    .toArray(CentreInsPedagogiDTO[]::new);
        elp.setListCentreInsPedagogi(tabCIP);
        
        creationSEMetierServiceInterface.creerELP_v2(elp);
    }
    
    /**
     * Get orgUnit content from id
     * @param orgUnitId the orgUnit id
     * @return the orgUnit content
     */
    private OrgUnit _getOrgUnitById(String orgUnitId)
    {
        try
        {
            OrgUnit orgUnit = _resolver.resolveById(orgUnitId);
            return orgUnit;
        }
        catch (Exception e)
        {
            getLogger().warn("Can't get orgunit from id " + orgUnitId, e);
            return null;
        }
    }
    
    /**
     * Create CIP object for VET Apogee
     * @param code the code of the CIP
     * @return the CIP object
     */
    private CentreInsPedagogiDTO2 _createCIPVET(String code)
    {
        CentreInsPedagogiDTO2 cip = new CentreInsPedagogiDTO2();
        cip.setCodCIP(code);
        
        return cip;
    }
    
    /**
     * Create CIP object for ELP Apogee
     * @param code the code of the CIP
     * @return the CIP object
     */
    private CentreInsPedagogiDTO _createCIPELP(String code)
    {
        return new CentreInsPedagogiDTO(code);
    }
    
    /**
     * Create Regime object for Apogee
     * @param code the code of the regime
     * @return the Regime object
     */
    private RegimeDTO _createRegime(String code)
    {
        RegimeDTO regime = new RegimeDTO();
        regime.setCodRegime(code);
        
        return regime;
    }
    
    /**
     * Create the orgUnit CGE for Apogee
     * @param orgUnit the Ametys orgUnit
     * @return the orgUnit CGE
     */
    private ComposanteCGEDTO _createComposanteCGEDTO(OrgUnit orgUnit)
    {
        ComposanteCGEDTO comp = new ComposanteCGEDTO();
        comp.setCodComposante(orgUnit.getValue(AbstractApogeeStructure.CODE_APOGEE_ATTRIBUTE_NAME));
        comp.setCodCGE(orgUnit.getValue("codeCGE"));
        
        return comp;
    }
    
    /**
     * Create the orgUnit VDI for Apogee
     * @param orgUnit the Ametys orgUnit
     * @return the orgUnit VDI
     */
    private CmpHabiliteeVdiDTO _createCmpHabiliteeVdiDTO(OrgUnit orgUnit)
    {
        CmpHabiliteeVdiDTO comp = new CmpHabiliteeVdiDTO();
        comp.setCodComposanteHabilitee(orgUnit.getValue(AbstractApogeeStructure.CODE_APOGEE_ATTRIBUTE_NAME));
        
        return comp;
    }
}
