/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.cdmfr.rights;

import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.repository.Content;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.AccessController;
import org.ametys.core.right.AccessExplanation;
import org.ametys.core.right.RightsException;
import org.ametys.core.right.RightsExtensionPoint;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.odf.rights.ODFContentHierarchicalAccessController;
import org.ametys.odf.tree.ODFContentsTreeHelper;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionDAO;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionHelper;
import org.ametys.plugins.odfsync.cdmfr.RemoteCDMFrSynchronizableContentsCollection;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.PluginAware;

/**
 * {@link AccessController} for contents imported by remote CDM-fr SCC
 */
public class RemoteCDMFrSCCAccessController implements AccessController, Serviceable, PluginAware
{
    /** The ODF contents tree helper */
    protected ODFContentsTreeHelper _odfContentsTreeHelper;
    private SynchronizableContentsCollectionHelper _sccHelper;
    private SynchronizableContentsCollectionDAO _collectionsDAO;
    private RightsExtensionPoint _rightsExtensionPoint;
    private String _id;

    public void service(ServiceManager manager) throws ServiceException
    {
        _odfContentsTreeHelper = (ODFContentsTreeHelper) manager.lookup(ODFContentsTreeHelper.ROLE);
        _sccHelper = (SynchronizableContentsCollectionHelper) manager.lookup(SynchronizableContentsCollectionHelper.ROLE);
        _collectionsDAO = (SynchronizableContentsCollectionDAO) manager.lookup(SynchronizableContentsCollectionDAO.ROLE);
        _rightsExtensionPoint = (RightsExtensionPoint) manager.lookup(RightsExtensionPoint.ROLE);
    }
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _id = id;
    }

    public AccessResult getPermission(UserIdentity user, Set<GroupIdentity> userGroups, String rightId, Object object)
    {
        return UserPopulationDAO.SYSTEM_USER_IDENTITY.equals(user) ? AccessResult.USER_ALLOWED : AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermission(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        return UserPopulationDAO.SYSTEM_USER_IDENTITY.equals(user) ? AccessResult.USER_ALLOWED : AccessResult.UNKNOWN;
    }

    public Map<String, AccessResult> getPermissionByRight(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        if (UserPopulationDAO.SYSTEM_USER_IDENTITY.equals(user))
        {
            return _rightsExtensionPoint.getExtensionsIds().stream().collect(Collectors.toMap(rightId -> rightId, rightId -> AccessResult.USER_ALLOWED));
        }
        else
        {
            return Collections.EMPTY_MAP;
        }
    }

    public AccessResult getPermissionForAnonymous(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnonymous(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getPermissionForAnyConnectedUser(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnyConnectedUser(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public Map<UserIdentity, AccessResult> getPermissionByUser(String rightId, Object object)
    {
        return Map.of(UserPopulationDAO.SYSTEM_USER_IDENTITY, AccessResult.USER_ALLOWED);
    }

    public Map<UserIdentity, AccessResult> getReadAccessPermissionByUser(Object object)
    {
        return getPermissionByUser(null, object);
    }

    public Map<GroupIdentity, AccessResult> getPermissionByGroup(String rightId, Object object)
    {
        return Collections.EMPTY_MAP;
    }

    public Map<GroupIdentity, AccessResult> getReadAccessPermissionByGroup(Object object)
    {
        return Collections.EMPTY_MAP;
    }

    public boolean hasUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups, String rightId)
    {
        return UserPopulationDAO.SYSTEM_USER_IDENTITY.equals(user);
    }

    public boolean hasUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups)
    {
        return hasUserAnyPermissionOnWorkspace(workspacesContexts, user, userGroups, null);
    }

    public boolean hasAnonymousAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnonymousAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }

    public boolean supports(Object object)
    {
        if (object instanceof Content)
        {
            Content content = (Content) object;
            Set<String> collectionIds = _sccHelper.getSynchronizableCollectionIds(content);
            for (String collectionId : collectionIds)
            {
                SynchronizableContentsCollection scc = _collectionsDAO.getSynchronizableContentsCollection(collectionId);
                if (scc instanceof RemoteCDMFrSynchronizableContentsCollection)
                {
                    return true;
                }
            }
        }

        return false;
    }
    
    @Override
    public AccessExplanation explainReadAccessPermission(UserIdentity user, Set<GroupIdentity> groups, Object context)
    {
        if (UserPopulationDAO.SYSTEM_USER_IDENTITY.equals(user))
        {
            return new AccessExplanation(getId(), AccessResult.USER_ALLOWED,
                    new I18nizableText("plugin.odf-sync", "PLUGINS_ODF_SYNC_CDMFR_ACCESS_CONTROLLER_SYSTEM_USER_ALLOWED",
                            Map.of("title", new I18nizableText(((Content) context).getTitle()))));
        }
        return AccessController.getDefaultAccessExplanation(getId(), AccessResult.UNKNOWN);
    }
    
    @Override
    public AccessExplanation explainPermission(UserIdentity user, Set<GroupIdentity> groups, String rightId, Object context)
    {
        if (UserPopulationDAO.SYSTEM_USER_IDENTITY.equals(user))
        {
            return new AccessExplanation(getId(), AccessResult.USER_ALLOWED,
                    new I18nizableText("plugin.odf-sync", "PLUGINS_ODF_SYNC_CDMFR_ACCESS_CONTROLLER_SYSTEM_USER_ALLOWED",
                            Map.of("title", getObjectLabel(context))));
        }
        return AccessController.getDefaultAccessExplanation(getId(), AccessResult.UNKNOWN);
    }
    
    public I18nizableText getObjectLabel(Object object)
    {
        if (object instanceof Content content)
        {
            return ODFContentHierarchicalAccessController.getContentObjectLabel(content, _odfContentsTreeHelper);
        }
        throw new RightsException("Unsupported context: " + object.toString());
    }
    
    @Override
    public Map<ExplanationObject, Map<Permission, AccessExplanation>> explainAllPermissions(UserIdentity identity, Set<GroupIdentity> groups, Set<Object> workspacesContexts)
    {
        // Only grant permission in admin context
        return Map.of();
    }
    
    public Map<Permission, AccessExplanation> explainAllPermissionsForAnonymous(Object object)
    {
        // Only used in admin context
        return Map.of();
    }
    
    public Map<Permission, AccessExplanation> explainAllPermissionsForAnyConnected(Object object)
    {
        // Only used in admin context
        return Map.of();
    }
    
    public Map<UserIdentity, Map<Permission, AccessExplanation>> explainAllPermissionsByUser(Object object)
    {
        // Only used in admin context
        return Map.of();
    }

    public Map<GroupIdentity, Map<Permission, AccessExplanation>> explainAllPermissionsByGroup(Object object)
    {
        // Only used in admin context
        return Map.of();
    }

    public I18nizableText getObjectCategory(Object object)
    {
        return ODFContentHierarchicalAccessController.ODF_CONTEXT_CATEGORY;
    }
    
    public String getId()
    {
        return _id;
    }
}
