/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.utils;

import java.util.stream.Stream;

import org.ametys.odf.course.CourseFactory;
import org.ametys.odf.courselist.CourseListFactory;
import org.ametys.odf.coursepart.CoursePartFactory;
import org.ametys.odf.orgunit.OrgUnitFactory;
import org.ametys.odf.person.PersonFactory;
import org.ametys.odf.program.ContainerFactory;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.SubProgramFactory;

/**
 * Object to describe content workflow elements.
 */
public enum ContentWorkflowDescription
{
    /** Program workflow description */
    PROGRAM_WF_DESCRIPTION(ProgramFactory.PROGRAM_CONTENT_TYPE, "program", 1, 4),
    /** Subprogram workflow description */
    SUBPROGRAM_WF_DESCRIPTION(SubProgramFactory.SUBPROGRAM_CONTENT_TYPE, "subprogram", 1, 4),
    /** Container workflow description */
    CONTAINER_WF_DESCRIPTION(ContainerFactory.CONTAINER_CONTENT_TYPE, "container", 1, 4),
    /** Course list workflow description */
    COURSELIST_WF_DESCRIPTION(CourseListFactory.COURSE_LIST_CONTENT_TYPE, "courselist", 1, 4),
    /** Course workflow description */
    COURSE_WF_DESCRIPTION(CourseFactory.COURSE_CONTENT_TYPE, "course", 1, 4),
    /** Course part workflow description */
    COURSEPART_WF_DESCRIPTION(CoursePartFactory.COURSE_PART_CONTENT_TYPE, "course-part", 1, -1),
    /** Orgunit workflow description */
    ORGUNIT_WF_DESCRIPTION(OrgUnitFactory.ORGUNIT_CONTENT_TYPE, "orgunit", 1, 4),
    /** Person workflow description */
    PERSON_WF_DESCRIPTION(PersonFactory.PERSON_CONTENT_TYPE, "person", 1, 4);
    
    private String _contentType;
    private String _workflowName;
    private int _initialActionId;
    private int _validationActionId;
    
    ContentWorkflowDescription(String contentType, String workflowName, int initialActionId, int validationActionId)
    {
        _contentType = contentType;
        _workflowName = workflowName;
        _initialActionId = initialActionId;
        _validationActionId = validationActionId;
    }
    
    /**
     * Get the content type.
     * @return the content type ID
     */
    public String getContentType()
    {
        return _contentType;
    }
    
    /**
     * Get the workflow name.
     * @return the workflow name
     */
    public String getWorkflowName()
    {
        return _workflowName;
    }
    
    /**
     * Get the initial action ID.
     * @return the initial action ID
     */
    public int getInitialActionId()
    {
        return _initialActionId;
    }
    
    /**
     * Get the validation action ID.
     * @return the validation action ID
     */
    public int getValidationActionId()
    {
        return _validationActionId;
    }
    
    /**
     * Retrieve the workflow description from the content type
     * @param contentType The content type
     * @return The corresponding workflow description
     */
    public static ContentWorkflowDescription getByContentType(String contentType)
    {
        return Stream.of(values())
            .filter(wd -> wd.getContentType().equals(contentType))
            .findFirst()
            .orElse(null);
    }
}
