/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.cdmfr.actions;

import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.environment.http.HttpRequest;

import org.ametys.core.authentication.AuthenticateAction;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.population.UserPopulationDAO;
import org.ametys.core.util.AvalonLoggerAdapter;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionHelper;
import org.ametys.plugins.core.user.UserDAO;

/**
 * Import a CDM-fr file from a remote source.
 */
public class UploadCDMFrAction extends ServiceableAction
{
    /** The remote CDMFr SCC model id */
    public static final String REMOTE_CDMFR_SCC_MODEL_ID = "org.ametys.plugins.odfsync.remote.cdmfr.scc";
    
    /** The name of the key to get the CDMFr input stream */
    public static final String CDMFR_IMPUT_STREAM_KEY_NAME = "cdmfr-input-stream";
    
    /** The SCC helper */
    protected SynchronizableContentsCollectionHelper _sccHelper;
    
    /** Current user provider. */
    protected CurrentUserProvider _currentUserProvider;
    
    @Override
    public void service(ServiceManager sManager) throws ServiceException
    {
        super.service(sManager);
        _sccHelper = (SynchronizableContentsCollectionHelper) sManager.lookup(SynchronizableContentsCollectionHelper.ROLE);
        _currentUserProvider = (CurrentUserProvider) sManager.lookup(CurrentUserProvider.ROLE);
    }

    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        if (request instanceof HttpRequest httpRequest)
        {
            SynchronizableContentsCollection scc = _sccHelper.getSCCFromModelId(REMOTE_CDMFR_SCC_MODEL_ID);
            if (scc == null)
            {
                String message = "Can't find synchronizable contents collections for the remote CDMFr import with model id " + REMOTE_CDMFR_SCC_MODEL_ID;
                getLogger().error(message);
                throw new IllegalStateException(message);
            }
            else
            {
                try (InputStream is = httpRequest.getInputStream())
                {
                    UserIdentity user = _currentUserProvider.getUser();
                    if (user == null)
                    {
                        AuthenticateAction.setUserIdentityInSession(request, UserPopulationDAO.SYSTEM_USER_IDENTITY, new UserDAO.ImpersonateCredentialProvider(), true);
                    }
                    
                    getLogger().info("Start of upload CDM-FR file");
                    
                    Map<String, Object> importParams = new HashMap<>();
                    importParams.put(CDMFR_IMPUT_STREAM_KEY_NAME, is);
                    scc.importContent(null, importParams, new AvalonLoggerAdapter(getLogger()));
                    
                    getLogger().info("End of upload CDM-FR file");
                }
            }
        }
        
        return EMPTY_MAP;
    }
}
