/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.cdmfr;

import java.io.InputStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.slf4j.Logger;

import org.ametys.cms.repository.ModifiableContent;
import org.ametys.plugins.odfsync.cdmfr.actions.UploadCDMFrAction;
import org.ametys.plugins.odfsync.cdmfr.components.ImportCDMFrComponent;
import org.ametys.plugins.odfsync.cdmfr.components.impl.RemoteImportCDMFrComponent;

/**
 * Class to import remote CDMFr contents
 */
public class RemoteCDMFrSynchronizableContentsCollection extends AbstractCDMFrSynchronizableContentsCollection
{
    /** The name of the param to get the shared with type */
    public static final String PARAM_SHARED_WITH_TYPE = "shared.subprogram.type";
    
    /** The name of the param to get the role of the CDM-fr importer */
    public static final String PARAM_CDMFR_IMPORTER_ROLE = "remote.implementation";
    
    /** The name of the param to get the catalog */
    public static final String PARAM_CDMFR_CATALOG = "remote.catalog";
    
    /** The name of the param to validate after import */
    public static final String PARAM_CDMFR_VALIDATE_AFTER_IMPORT = "remote.validate.after.import";
    
    /** The service manager */
    protected ServiceManager _manager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _manager = manager;
    }
    
    @Override
    protected void configureStaticParams(Configuration configuration) throws ConfigurationException
    {
        super.configureStaticParams(configuration);
        try
        {
            _importCDMFrComponent = (ImportCDMFrComponent) _manager.lookup(getImportCDMFrRole());
        }
        catch (ServiceException e)
        {
            throw new ConfigurationException("An error occurred getting the CDM-fr importer", e);
        }
    }

    @Override
    @SuppressWarnings("unchecked")
    public List<ModifiableContent> importContent(String idValue, Map<String, Object> additionalParameters, Logger logger) throws Exception
    {
        if (additionalParameters.containsKey(UploadCDMFrAction.CDMFR_IMPUT_STREAM_KEY_NAME))
        {
            Map<String, Object> parameters = new HashMap<>();
            
            try
            {
                _startHandleCDMFR();

                parameters.put(PARAM_SHARED_WITH_TYPE, getSharedWithType());
                parameters.put(PARAM_CDMFR_CATALOG, getCatalog());
                parameters.put(PARAM_CDMFR_VALIDATE_AFTER_IMPORT, validateAfterImport());
                
                InputStream is = (InputStream) additionalParameters.get(UploadCDMFrAction.CDMFR_IMPUT_STREAM_KEY_NAME);
                Map<String, Object> resultMap = _importCDMFrComponent.handleInputStream(is, parameters, this, logger);
                
                return (List<ModifiableContent>) resultMap.get("importedPrograms");
            }
            catch (ProcessingException e)
            {
                logger.error("An error occurred. Can't import remote program.", e);
            }
            finally
            {
                _endHandleCDMFR((Set<String>) parameters.get("updatedContents"));
            }
        }
        else
        {
            logger.error("An error occurred because no input stream has been sent");
        }
        
        return null;
    }

    /**
     * Get the defined way to detect shared program
     * @return The defined way to detect shared program
     */
    protected String getSharedWithType()
    {
        return (String) getParameterValues().get(PARAM_SHARED_WITH_TYPE);
    }
    
    /**
     * Get the role of the CDM-fr importer
     * @return the role of the CDM-fr importer
     */
    protected String getImportCDMFrRole()
    {
        return (String) getParameterValues().getOrDefault(PARAM_CDMFR_IMPORTER_ROLE, RemoteImportCDMFrComponent.ROLE);
    }
    
    /**
     * Get the catalog
     * @return the catalog
     */
    protected String getCatalog()
    {
        return (String) getParameterValues().get(PARAM_CDMFR_CATALOG);
    }
}
