/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.cdmfr.transformers;

import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.components.source.SourceUtil;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.xml.dom.DOMParser;
import org.apache.excalibur.xml.xpath.PrefixResolver;
import org.apache.excalibur.xml.xpath.XPathProcessor;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

/**
 * Default implementation of CDMFrSyncTransformer.
 */
public class DefaultCDMFrSyncTransformer implements CDMFrSyncTransformer, Serviceable, Configurable
{
    /** The default XML namespace prefixes. */
    public static final Map<String, String> DEFAULT_NAMESPACES = new HashMap<>();
    static
    {
        DEFAULT_NAMESPACES.put("cdm", "http://cdm-fr.fr/2011/CDM");
        DEFAULT_NAMESPACES.put("cdmfr", "http://cdm-fr.fr/2011/CDM-frSchema");
    }
    
    /** The DOM parser. */
    protected DOMParser _domParser;
    /** Ametys object resolver. */
    protected SourceResolver _sourceResolver;
    /** The XPath processor. */
    protected XPathProcessor _xPathProcessor;
    
    /** The XSL file to transform the document. */
    protected String _xslFile;

    /** Set of applications supported by the transformer. */
    protected Set<Pattern> _applications = new HashSet<>();
    
    /** The prefix resolver. */
    protected PrefixResolver _prefixResolver;
    
    /** The priority */
    protected Integer _priority;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _domParser = (DOMParser) manager.lookup(DOMParser.ROLE);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _xPathProcessor = (XPathProcessor) manager.lookup(XPathProcessor.ROLE);
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        Configuration[] applications = configuration.getChildren("app");
        for (Configuration application : applications)
        {
            _applications.add(Pattern.compile(application.getValue(), Pattern.DOTALL));
        }

        _xslFile = configuration.getChild("configuration").getChild("xsl-transform").getValue(StringUtils.EMPTY);
        _priority = configuration.getChild("priority").getValueAsInteger(50);
        
        configureNamespaces(configuration.getChild("namespaces", false));
    }
    
    /**
     * Configure the namespace to use.
     * @param configuration the namespaces configuration, can be null.
     */
    protected void configureNamespaces(Configuration configuration)
    {
        Map<String, String> namespaces = DEFAULT_NAMESPACES;
        
        if (configuration != null)
        {
            namespaces = new HashMap<>();
            
            for (Configuration nsConf : configuration.getChildren("namespace"))
            {
                String prefix = nsConf.getAttribute("prefix", StringUtils.EMPTY);
                String namespace = nsConf.getAttribute("value", StringUtils.EMPTY);
                
                namespaces.put(prefix, namespace);
            }
        }
        
        _prefixResolver = new CDMFrPrefixResolver(namespaces);
    }

    @Override
    public boolean supports(Document document)
    {
        String application = _xPathProcessor.evaluateAsString(document, "/cdm:CDM/cdm:properties/cdm:datasource", _prefixResolver);
        if (StringUtils.isNotBlank(application))
        {
            for (Pattern pattern : _applications)
            {
                if (pattern.matcher(application).matches())
                {
                    return true;
                }
            }
        }
        return false;
    }

    @Override
    public Document transform(Document document, Map<String, Object> parameters) throws IOException, SAXException, ProcessingException
    {
        parameters.put("source", document);
        parameters.put("xslFile", _xslFile);
        Source src = _sourceResolver.resolveURI("cocoon://_plugins/odf-sync/cdmfr/import/file", null, parameters);
        return SourceUtil.toDOM(src);
    }
    
    /**
     * XML prefix resolver which declares CDMFr namespaces.
     */
    public static class CDMFrPrefixResolver implements PrefixResolver
    {
        
        /** List of namespaces, indexed by prefix. */
        private Map<String, String> _namespaces;
        
        /**
         * Constructor.
         * @param namespaces the namespaces to resolve, indexed by prefix.
         */
        public CDMFrPrefixResolver(Map<String, String> namespaces)
        {
            _namespaces = namespaces;
        }
        
        public String prefixToNamespace(String prefix)
        {
            return _namespaces.get(prefix);
        }
    }

    public int getPriority()
    {
        return _priority;
    }
}
