/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.apogee.ws.structure;

import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.type.ModelItemTypeConstants;
import org.ametys.cms.repository.Content;
import org.ametys.odf.course.Course;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.ProgramPart;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.odfsync.apogee.ws.ApogeeStructureComponent;
import org.ametys.plugins.odfsync.apogee.ws.ApogeeWS;
import org.ametys.plugins.odfsync.export.AbstractExportStructure;
import org.ametys.plugins.odfsync.export.ExportReport;
import org.ametys.plugins.odfsync.export.ExportReport.ExportStatus;
import org.ametys.runtime.i18n.I18nizableText;

import gouv.education.apogee.commun.client.ws.creationse.CreationSEMetierServiceInterface;

/**
 * The abstract class to handle an export in Apogee
 */
public abstract class AbstractApogeeStructure extends AbstractExportStructure
{
    /** Avalon Role */
    public static final String ROLE = ApogeeStructureComponent.class.getName();
    
    /** The attribute name for the code Apogee */
    public static final String CODE_APOGEE_ATTRIBUTE_NAME = "codeApogee";
    
    /** The false attribute name for the version Apogee */
    public static final String VERSION_APOGEE_ATTRIBUTE_NAME = "versionApogee";
    
    /** The separator between the code and the version */
    public static final String CODE_APOGEE_SEPARATOR = "-";
    
    /** The apogee WS */
    protected ApogeeWS _apogeeWS;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _apogeeWS = (ApogeeWS) manager.lookup(ApogeeWS.ROLE);
    }
    
    /**
     * Mandatory data to export a content in a DIP in Apogee
     * @param content the content to export
     * @return the list of mandatory data
     */
    public List<String> getDIPMandatoryData(Content content)
    {
        List<String> mandatoryData = new ArrayList<>();
        mandatoryData.add(CODE_APOGEE_ATTRIBUTE_NAME);
        mandatoryData.add(AbstractProgram.EDUCATION_KIND);
        mandatoryData.add("cycleApogee");
        mandatoryData.add(AbstractProgram.DEGREE);
        
        return mandatoryData;
    }
    
    /**
     * Mandatory data to export a content in a VDI in Apogee
     * @param content the content to export
     * @return the list of mandatory data
     */
    public List<String> getVDIMandatoryData(Content content)
    {
        // The orgunit Apogee Code is mandatory too
        
        List<String> mandatoryData = new ArrayList<>();
        mandatoryData.add(VERSION_APOGEE_ATTRIBUTE_NAME);
        mandatoryData.add("start-date-recruitment");
        mandatoryData.add("end-date-recruitment");
        mandatoryData.add("start-date-validation");
        mandatoryData.add("end-date-validation");
        
        return mandatoryData;
    }
    
    /**
     * Mandatory data to export a content in a ETP in Apogee
     * @param content the content to export
     * @return the list of mandatory data
     */
    public List<String> getETPMandatoryData(Content content)
    {
        // The orgunit Apogee Code is mandatory too
        
        List<String> mandatoryData = new ArrayList<>();
        mandatoryData.add(CODE_APOGEE_ATTRIBUTE_NAME); 
        mandatoryData.add("cycleApogee");
        
        return mandatoryData;
    }
    
    /**
     * Mandatory data to export a content in a VET in Apogee
     * @param content the content to export
     * @return the list of mandatory data
     */
    public List<String> getVETMandatoryData(Content content)
    {
        // The orgunit Apogee Code is mandatory too
        
        List<String> mandatoryData = new ArrayList<>();
        mandatoryData.add(VERSION_APOGEE_ATTRIBUTE_NAME);
        mandatoryData.add("duration-apogee");
        mandatoryData.add("inscription-types"); 
        mandatoryData.add("cips"); 
        
        return mandatoryData;
    }
    
    /**
     * Mandatory data to export a content in a LSE in Apogee
     * @param content the content to export
     * @return the list of mandatory data
     */
    public List<String> getLSEMandatoryData(Content content)
    {
        List<String> mandatoryData = new ArrayList<>();
        mandatoryData.add(CODE_APOGEE_ATTRIBUTE_NAME); 
        mandatoryData.add("choiceType"); 
        
        String choiceTypeAmetys = content.getValue("choiceType");
        if (choiceTypeAmetys.equals("CHOICE"))
        {
            mandatoryData.add("min"); 
        }
        
        return mandatoryData;
    }
    
    /**
     * Mandatory data to export a content in a ELP in Apogee
     * @param content the content to export
     * @return the list of mandatory data
     */
    public List<String> getELPMandatoryData(Content content)
    {
        // The orgunit Apogee Code is mandatory too
        
        List<String> mandatoryData = new ArrayList<>();
        mandatoryData.add(CODE_APOGEE_ATTRIBUTE_NAME); 
        mandatoryData.add("cips"); 
        mandatoryData.add("orgUnit"); 
        if (content instanceof Container)
        {
            mandatoryData.add("nature"); 
        }
        else if (content instanceof Course)
        {
            mandatoryData.add("courseType"); 
        }
        
        return mandatoryData;
    }
    
    /**
     * Mandatory data to export an orgunit for a DIP in Apogee
     * @return the list of mandatory data
     */
    public List<String> getOrgUnitMandatoryDataForDIP()
    {
        List<String> mandatoryData = new ArrayList<>();
        mandatoryData.add(CODE_APOGEE_ATTRIBUTE_NAME); 
        
        return mandatoryData;
    }
    
    /**
     * Mandatory data to export an orgunit for a ETP in Apogee
     * @return the list of mandatory data
     */
    public List<String> getOrgUnitMandatoryDataForETP()
    {
        List<String> mandatoryData = new ArrayList<>();
        mandatoryData.add(CODE_APOGEE_ATTRIBUTE_NAME); 
        mandatoryData.add("codeCGE"); 
        
        return mandatoryData;
    }
    
    /**
     * Mandatory data to export an orgunit for a ELP in Apogee
     * @return the list of mandatory data
     */
    public List<String> getOrgUnitMandatoryDataForELP()
    {
        List<String> mandatoryData = new ArrayList<>();
        mandatoryData.add(CODE_APOGEE_ATTRIBUTE_NAME); 
        
        return mandatoryData;
    }
    
    /**
     * Get the code Apogee of the content
     * @param content the content
     * @return the code Apogee
     */
    public String getCodeApogee(Content content)
    {
        String codeApogee = content.getValue(CODE_APOGEE_ATTRIBUTE_NAME);
        if (StringUtils.isNotBlank(codeApogee) && codeApogee.contains(CODE_APOGEE_SEPARATOR))
        {
            return StringUtils.substringBefore(codeApogee, CODE_APOGEE_SEPARATOR);
        }
        else
        {
            return codeApogee;
        }
    }
    
    /**
     * Get the version, Apogee of the content
     * @param content the content
     * @return the version Apogee
     */
    public Long getVersionApogee(Content content)
    {
        String codeApogee = content.getValue(CODE_APOGEE_ATTRIBUTE_NAME);
        if (StringUtils.isNotBlank(codeApogee) && codeApogee.contains(CODE_APOGEE_SEPARATOR))
        {
            return Long.parseLong(StringUtils.substringAfterLast(codeApogee, CODE_APOGEE_SEPARATOR));
        }
        else
        {
            return null;
        }
    }
    
    /**
     * Check if the subProgram has the good data and structure to be export in Apogee
     * @param subProgram the subProgram to check
     * @param report the Apogee export report
     */
    public abstract void checkSubProgram(SubProgram subProgram, ExportReport report);
    
    /**
     * Check if the container as year has the good data and structure to be export in Apogee
     * @param container the container to check
     * @param report the Apogee export report
     * @param containerNatureCode The container nature code
     */
    public void checkContainerAsYear(Container container, ExportReport report, String containerNatureCode)
    {
        if (checkContainerYear(containerNatureCode, report))
        {
            // The container is a semester so check the data as a semester (ELP in Apogee)
            checkMandatoryDataForContent(container, getETPMandatoryData(container), report);
            checkMandatoryDataForContent(container, getVETMandatoryData(container), report);
            
            List<String> orgUnits = container.getOrgUnits();
            checkMandatoryDataForOrgunits(container, orgUnits, getOrgUnitMandatoryDataForETP(), report);
            
            // Check the container structure
            List<ProgramPart> programPartChildren = container.getProgramPartChildren();
            for (ProgramPart childProgramPart : programPartChildren)
            {
                if (childProgramPart instanceof Container)
                {
                    Container containerChildProgramPart = (Container) childProgramPart;
                    String childContainerNatureCode = getContainerNatureCode(containerChildProgramPart);
                    
                    checkContainerAsSemester(containerChildProgramPart, report, childContainerNatureCode);
                }
                else
                {
                    report.setStatus(ExportStatus.CONTENT_STRUCTURE_INVALID);
                    break;
                }
            }
            
            if (programPartChildren.isEmpty())
            {
                // The structure is not handled by this export
                report.setStatus(ExportStatus.CONTENT_STRUCTURE_INVALID);
            }
        }
        else
        {
            // The structure is not handled by this export
            report.setStatus(ExportStatus.CONTENT_STRUCTURE_INVALID);
        }
    }
    
    /**
     * Check if the container as semester has the good data and structure to be export in Apogee
     * @param container the container to check
     * @param report the Apogee export report
     * @param containerNatureCode The container nature code
     */
    public void checkContainerAsSemester(Container container, ExportReport report, String containerNatureCode)
    {
        if (checkContainerSemester(containerNatureCode, report))
        {
            // The container is a semester so check the data as a semester (ELP in Apogee)
            checkMandatoryDataForContent(container, getELPMandatoryData(container), report);
            
            // Check the container structure
            for (ProgramPart childProgramPart : container.getProgramPartChildren())
            {
                if (childProgramPart instanceof CourseList)
                {
                    checkCourseList((CourseList) childProgramPart, report);
                }
                else
                {
                    report.setStatus(ExportStatus.CONTENT_STRUCTURE_INVALID);
                    break;
                }
            }
        }
        else
        {
            // The structure is not handled by this export
            report.setStatus(ExportStatus.CONTENT_STRUCTURE_INVALID);
        }
    }
    
    /**
     * Check if the course list has the good data and structure to be export in Apogee
     * @param courseList the course list to check
     * @param report the Apogee export report
     */
    public void checkCourseList(CourseList courseList, ExportReport report)
    {
        // Check mandatory data for course list
        checkMandatoryDataForContent(courseList, getLSEMandatoryData(courseList), report);
        
        // Check the course list structure
        for (Course course : courseList.getCourses())
        {
            checkCourse(course, report);
        }
    }
    
    /**
     * Check if the course has the good data and structure to be export in Apogee
     * @param course the course to check
     * @param report the Apogee export report
     */
    public void checkCourse(Course course, ExportReport report)
    {
        // Check mandatory data for course
        checkMandatoryDataForContent(course, getELPMandatoryData(course), report);
        
        // Check mandatory data for course orgUnits
        checkMandatoryDataForOrgunits(course, course.getOrgUnits(), getOrgUnitMandatoryDataForELP(), report);
        
         // Check the course structure
        for (CourseList courseList : course.getCourseLists())
        {
            checkCourseList(courseList, report);
        }
    }
    
    
    /**
     * Check if the content has a value for all mandatory data
     * @param content the content to check
     * @param mandatoryData the list of mandatory data path
     * @param report the Apogee export report
     */
    public void checkMandatoryDataForContent(Content content, List<String> mandatoryData, ExportReport report)
    {
        for (String dataPath : mandatoryData)
        {
            if (VERSION_APOGEE_ATTRIBUTE_NAME.equals(dataPath))
            {
                _checkVersionApogee(content, report);
            }
            else if (content.hasValue(dataPath))
            {
                if (org.ametys.runtime.model.type.ModelItemTypeConstants.STRING_TYPE_ID.equals(content.getType(dataPath).getId()))
                {
                    _checkSimpleData(content, dataPath, report);    
                }
                else if (ModelItemTypeConstants.CONTENT_ELEMENT_TYPE_ID.equals(content.getType(dataPath).getId()))
                {
                    _checkTableRef(content, dataPath, report);
                }
            }
            else
            {
                _addMandatoryDataPathAndReport(content, dataPath, report);
            }
        }
    }

    /**
     * Check if the content has a value for the simple data
     * @param content the content to check
     * @param dataPath the data path
     * @param report the Apogee export report
     */
    protected void _checkSimpleData(Content content, String dataPath, ExportReport report)
    {
        if (content.isMultiple(dataPath))
        {
            String[] values = content.getValue(dataPath);
            if (values.length == 0 || StringUtils.isBlank(values[0]))
            {
                _addMandatoryDataPathAndReport(content, dataPath, report);
            }
        }
        else
        {
            String value = content.getValue(dataPath);
            if (StringUtils.isBlank(value))
            {
                _addMandatoryDataPathAndReport(content, dataPath, report);
            }
        }
    }

    /**
     * Check if the content has a value for the simple table ref data
     * @param content the content to check
     * @param dataPath the data path
     * @param report the Apogee export report
     */
    protected void _checkTableRef(Content content, String dataPath, ExportReport report)
    {
        // We handle only simple table ref
        if (!content.isMultiple(dataPath))
        {
            ContentValue value = content.getValue(dataPath);
            Content refContent = value.getContent();
            if (!value.hasValue(CODE_APOGEE_ATTRIBUTE_NAME))
            {
                _addMandatoryDataPathAndReport(refContent, CODE_APOGEE_ATTRIBUTE_NAME, report);
            }
        }
    }

    /**
     * Check if the version Apogee exist in the Apogee code
     * @param content the content
     * @param report the Apogee export report
     */
    protected void _checkVersionApogee(Content content, ExportReport report)
    {
        if (!content.hasValue(CODE_APOGEE_ATTRIBUTE_NAME))
        {
            report.addInvalidDataPath(content, new I18nizableText("plugin.odf-sync", "PLUGINS_ODF_SYNC_EXPORT_APOGEE_MANDATORY_VERSION"));
        }
        else
        {
            String codeApogee = content.getValue(CODE_APOGEE_ATTRIBUTE_NAME);
            if (!codeApogee.contains("-") || StringUtils.isBlank(StringUtils.substringAfterLast(codeApogee, "-")))
            {
                report.addInvalidDataPath(content, new I18nizableText("plugin.odf-sync", "PLUGINS_ODF_SYNC_EXPORT_APOGEE_MANDATORY_VERSION"));
            }
        }
    }
    
    /**
     * Check if the orgUnits has a value for all mandatory data
     * @param content the content to check
     * @param orgUnits the list of orgUnit to check
     * @param mandatoryData the list of mandatory data path
     * @param report the Apogee export report
     */
    public void checkMandatoryDataForOrgunits(Content content, List<String> orgUnits, List<String> mandatoryData, ExportReport report)
    {
        if (orgUnits.size() == 0)
        {
            // No orgUnits, course data is invalid
            _addMandatoryDataPathAndReport(content, "orgUnit", report);
        }
        else
        {
            for (String orgUnitId : orgUnits)
            {
                OrgUnit orgUnit = _resolver.resolveById(orgUnitId);
                checkMandatoryDataForContent(orgUnit, mandatoryData, report);
            }
        }
    }
    
    /**
     * Create a course list in Apogee
     * @param courseList the course list to create
     * @param parentApogee the parent in Apogee
     * @param creationService the service to create element in Apogee
     * @param report the Apogee export report
     * @throws RemoteException if an export error occurred
     */
    protected void _createCourseList(CourseList courseList, Content parentApogee, CreationSEMetierServiceInterface creationService, ExportReport report) throws RemoteException
    {
        // Create ELPs before the list
        for (Course course : courseList.getCourses()) 
        {
            _createCourse(course, courseList, creationService, report);
        }
        
        String codELP = getCodeApogee(parentApogee);
        String codLSE = getCodeApogee(courseList);

        _apogeeWS.createLSE(courseList, null, codLSE, creationService);
        
        Long nbELP = null;
        Double ectsMin = 0D;
        Double ectsMax = 0D;
        String choiceTypeAmetys = courseList.getValue("choiceType");
        if (choiceTypeAmetys.equals("CHOICE"))
        {
            nbELP = courseList.getValue("min");
            
            List<Double> ectsList = courseList.getCourses()
                .stream()
                .map(Course::getEcts)
                .sorted()
                .collect(Collectors.toList());
            
            for (int i = 0; i < nbELP; i++)
            {
                ectsMin += ectsList.get(0);
            }
            
            ectsMin = (Math.floor(ectsMin) == 0D) ? 1D : Math.floor(ectsMin);
            
            for (int i = 1; i <= nbELP; i++)
            {
                ectsMax += ectsList.get(ectsList.size() - i);
            }
            
            ectsMax = (Math.ceil(ectsMax) == 0D || ectsMin > ectsMax) ? 1D : Math.ceil(ectsMax);
        }
        
        _apogeeWS.createLinkETPELPLSE(null, null, codLSE, codELP, nbELP, ectsMin, ectsMax, creationService);
    }
    
    /**
     * Create a course in Apogee
     * @param course the course to create
     * @param parentApogee the parent in Apogee
     * @param creationService the service to create element in Apogee
     * @param report the Apogee export report
     * @throws RemoteException if an export error occurred
     */
    protected void _createCourse(Course course, Content parentApogee, CreationSEMetierServiceInterface creationService, ExportReport report) throws RemoteException
    {
        String codELP = getCodeApogee(course);
        _apogeeWS.createELP(course, null, codELP, creationService);
        
        for (CourseList courseList : course.getCourseLists())
        {
            _createCourseList(courseList, course, creationService, report);
        }
    }
    
    /**
     * Check if the container is a semester and set the exportStatus to CONTENT_STRUCTURE_INVALID if the container nature code cannot be found
     * @param containerNatureCode The container nature code
     * @param report The ExportReport that contains the exportStatus
     * @return true if the container is a semester, false if it is not, and null if the container nature code could not be retrieved
     */
    protected boolean checkContainerSemester(String containerNatureCode, ExportReport report)
    {
        if (containerNatureCode.isBlank())
        {
            //The structure is not handled by this export because the container has no type
            report.setStatus(ExportStatus.CONTENT_STRUCTURE_INVALID);
            return false;
        }
        
        return "semestre".equals(containerNatureCode);
    }
    
    /**
     * Check if the container is a year and set the exportStatus to CONTENT_STRUCTURE_INVALID if the container nature code cannot be found
     * @param containerNatureCode The container nature code
     * @param report The ExportReport that contains the exportStatus
     * @return true if the container is a year, false if it is not, and null if the container nature code could not be retrieved
     */
    protected boolean checkContainerYear(String containerNatureCode, ExportReport report)
    {
        if (containerNatureCode.isBlank())
        {
            //The structure is not handled by this export because the container has no type
            report.setStatus(ExportStatus.CONTENT_STRUCTURE_INVALID);
            return false;
        }
        
        return "annee".equals(containerNatureCode);
    }
    
    private void _addMandatoryDataPathAndReport(Content content, String dataPath, ExportReport report)
    {
        I18nizableText invalidMessage = new I18nizableText(
            "plugin.odf-sync",
            "PLUGINS_ODF_SYNC_EXPORT_APOGEE_MANDATORY_FIELD",
            Map.of("fieldName", content.getDefinition(dataPath).getLabel())
        );
        report.addInvalidDataPath(content, invalidMessage);
    }
}
