/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.apogee.ws.structure;

import java.rmi.RemoteException;
import java.util.List;

import org.ametys.cms.repository.Content;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramPart;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.odfsync.export.ExportReport;
import org.ametys.plugins.odfsync.export.ExportReport.ExportStatus;

import gouv.education.apogee.commun.client.ws.creationse.CreationSEMetierServiceInterface;

/**
 * The structure to export in Apogee the following program
 * <br>Program / SubProgram / Container (year) / Container (semester) / UE / ELP / ...
 * <br>into DIP / VDI / ETP-VET / ELP / LSE / ELP / ...
 */
public class ApogeeFullStructure extends AbstractApogeeStructure
{
    @Override
    public void checkProgram(Program program, ExportReport report)
    {
        // Check mandatory data for program
        checkMandatoryDataForContent(program, getDIPMandatoryData(program), report);
        
        // Check the program structure
        List<ProgramPart> programPartChildren = program.getProgramPartChildren();
        for (ProgramPart programPart : programPartChildren)
        {
            if (programPart instanceof SubProgram)
            {
                checkSubProgram((SubProgram) programPart, report);
            }
            else
            {
                // The structure is not handled by this export
                report.setStatus(ExportStatus.CONTENT_STRUCTURE_INVALID);
                break;
            }
        }
        
        if (programPartChildren.isEmpty())
        {
            // The structure is not handled by this export
            report.setStatus(ExportStatus.CONTENT_STRUCTURE_INVALID);
        }
    }
    
    @Override
    public void checkSubProgram(SubProgram subProgram, ExportReport report)
    {
        // Check mandatory data for subProgram
        checkMandatoryDataForContent(subProgram, getVDIMandatoryData(subProgram), report);
        
        
        // Check mandatory data for subProgram orgUnits
        checkMandatoryDataForOrgunits(subProgram, subProgram.getOrgUnits(), getOrgUnitMandatoryDataForDIP(), report);
        
        // Check the subProgram structure
        List<ProgramPart> programPartChildren = subProgram.getProgramPartChildren();
        for (ProgramPart childProgramPart : programPartChildren)
        {
            if (childProgramPart instanceof Container)
            {
                Container containerChildProgramPart = (Container) childProgramPart;
                String childContainerNatureCode = getContainerNatureCode(containerChildProgramPart);
                
                checkContainerAsYear(containerChildProgramPart, report, childContainerNatureCode);
            }
            else
            {
                // The structure is not handled by this export
                report.setStatus(ExportStatus.CONTENT_STRUCTURE_INVALID);
                break;
            }
        }
        
        if (programPartChildren.isEmpty())
        {
            // The structure is not handled by this export
            report.setStatus(ExportStatus.CONTENT_STRUCTURE_INVALID);
        }
    }
    
    @Override
    public void createProgram(Program program, ExportReport report)
    {
        try
        {
            CreationSEMetierServiceInterface creationService = _apogeeWS.getCreationService();
            String codDIP = getCodeApogee(program);
            _apogeeWS.createDIP(program, null, codDIP, creationService);
            for (ProgramPart pp : program.getProgramPartChildren())
            {
                _createSubProgram((SubProgram) pp, program, creationService, report);
            }
        }
        catch (Exception e)
        {
            report.setStatus(ExportStatus.ERROR);
            getLogger().error("An error occurred exporting the program '{}' ({}) in Apogee", program.getTitle(), program.getId(), e);
        }
    }
    
    /**
     * Create a subProgram in Apogee
     * @param subProgram the subProgram to create
     * @param programParent the program parent in Apogee
     * @param creationService the service to create element in Apogee
     * @param report the Apogee export report
     * @throws RemoteException if an export error occurred
     */
    protected void _createSubProgram(SubProgram subProgram, Content programParent, CreationSEMetierServiceInterface creationService, ExportReport report) throws RemoteException
    {
        String codDip = getCodeApogee(programParent);
        Long versionDIP = getVersionApogee(subProgram);
        
        _apogeeWS.createVDI(subProgram, programParent.getTitle() + " - " + subProgram.getTitle(), codDip, versionDIP, creationService);
        
        for (ProgramPart pp : subProgram.getProgramPartChildren())
        {
            _createContainerAsETPVET((Container) pp, programParent, subProgram, creationService, report);
        }
    }
    
    /**
     * Create a container as ETP/VET in Apogee
     * @param container the container to create
     * @param programParent the program parent in Apogee
     * @param parentSubProgram the parent subProgram
     * @param creationService the service to create element in Apogee
     * @param report the Apogee export report
     * @throws RemoteException if an export error occurred
     */
    protected void _createContainerAsETPVET(Container container, Content programParent, Content parentSubProgram, CreationSEMetierServiceInterface creationService, ExportReport report) throws RemoteException
    {
        String codDIP = getCodeApogee(programParent);
        Long versionDIP = getVersionApogee(parentSubProgram);
        String codETP = getCodeApogee(container);
        Long versionETP = getVersionApogee(container);
        
        // Create the ETP / VET from the container
        _apogeeWS.createETP(container, null, codETP, creationService);
        _apogeeWS.createVET(container, null, codETP, versionETP, creationService);
        
        // Link to the DIP / VDI (Program / SubProgram)
        _apogeeWS.createLinkDIPETP(codDIP, versionDIP, codETP, versionETP, creationService);
        
        for (ProgramPart pp : container.getProgramPartChildren())
        {
            _createContainerAsELP((Container) pp, programParent, container, creationService, report);
        }
    }
    
    /**
     * Create a container in Apogee
     * @param container the container to create
     * @param programParent the program parent in Apogee
     * @param parentYearContainer the parent year container
     * @param creationService the service to create element in Apogee
     * @param report the Apogee export report
     * @throws RemoteException if an export error occurred
     */
    protected void _createContainerAsELP(Container container, Content programParent, Content parentYearContainer, CreationSEMetierServiceInterface creationService, ExportReport report) throws RemoteException
    {
        String codETP = getCodeApogee(parentYearContainer);
        Long versionETP = getVersionApogee(parentYearContainer);
        String codELP = getCodeApogee(container);
        
        // Create the ELP from the container
        _apogeeWS.createELP(container, null, codELP, creationService);
        
        // Create a mandatory LSE with random code
        String codLSE = org.ametys.core.util.StringUtils.generateKey();
        _apogeeWS.createMandatoryLSE("LSE - " + parentYearContainer.getTitle(), codLSE, codELP, creationService);
        
        // Create the link between ETP and LSE (year container and semester container)
        _apogeeWS.createLinkETPELPLSE(codETP, versionETP, codLSE, null, null, null, null, creationService);
        
        for (ProgramPart pp : container.getProgramPartChildren())
        {
            _createCourseList((CourseList) pp, container, creationService, report);
        }
    }
}
