/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.cdmfr.extractor;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Optional;

import org.apache.commons.collections4.CollectionUtils;
import org.w3c.dom.Element;

import org.ametys.cms.repository.Content;
import org.ametys.core.util.dom.DOMUtils;
import org.ametys.odf.program.TraversableProgramPart;
import org.ametys.plugins.odfsync.cdmfr.ImportCDMFrContext;
import org.ametys.plugins.odfsync.cdmfr.components.ImportCDMFrComponent;
import org.ametys.plugins.odfsync.utils.ContentWorkflowDescription;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.ModelHelper;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelItemContainer;

/**
 * This class provides methods to extract traversable program part's values from a CMD-fr import document
 */
public class ImportTraversableProgramPartValuesExtractor extends ImportCDMFrValuesExtractor
{
    /** Tag to identify a subprogram */
    protected static final String _TAG_SUBPROGRAM = "subProgram";
    
    /** Tag to identify a container */
    protected static final String _TAG_CONTAINER = "container";
    
    /** The imported content's synchronization code */
    protected String _synchronizationCode;
    
    /**
     * Creates an import traversable program part values extractor
     * @param element the DOM element containing the XML values
     * @param factory the values extractor factory
     * @param component the import CDM-fr component
     * @param synchronizationCode the imported content's synchronization code
     * @param context the import context
     * @param models the model of the extracted values
     */
    public ImportTraversableProgramPartValuesExtractor(Element element, ImportCDMFrValuesExtractorFactory factory, ImportCDMFrComponent component, String synchronizationCode, ImportCDMFrContext context, Model... models)
    {
        this(element, factory, component, synchronizationCode, context, Arrays.asList(models));
    }
    
    /**
     * Creates an import traversable program part values extractor
     * @param element the DOM element containing the XML values
     * @param factory the values extractor factory
     * @param component the import CDM-fr component
     * @param synchronizationCode the imported content's synchronization code
     * @param context the imported content's context
     * @param modelItemContainers the model of the extracted values
     */
    public ImportTraversableProgramPartValuesExtractor(Element element, ImportCDMFrValuesExtractorFactory factory, ImportCDMFrComponent component, String synchronizationCode, ImportCDMFrContext context, Collection<? extends ModelItemContainer> modelItemContainers)
    {
        super(element, factory, component, context, modelItemContainers);
        
        _synchronizationCode = synchronizationCode;
    }
    
    @Override
    protected Optional<ModelItem> _getModelItemFromNodeName(Element parent, String nodeName, Collection< ? extends ModelItemContainer> modelItemContainers)
    {
        if (_TAG_SUBPROGRAM.equals(nodeName) || _TAG_CONTAINER.equals(nodeName) || _TAG_COURSELIST.equals(nodeName))
        {
            return Optional.of(ModelHelper.getModelItem(TraversableProgramPart.CHILD_PROGRAM_PARTS, modelItemContainers));
        }
        else
        {
            return super._getModelItemFromNodeName(parent, nodeName, modelItemContainers);
        }
    }
    
    @Override
    protected boolean _hasChildForAttribute(Element element, String attributeName)
    {
        if (TraversableProgramPart.CHILD_PROGRAM_PARTS.equals(attributeName))
        {
            return DOMUtils.hasChildElement(element, _TAG_SUBPROGRAM)
                    || DOMUtils.hasChildElement(element, _TAG_CONTAINER)
                    || DOMUtils.hasChildElement(element, _TAG_COURSELIST);
        }
        else
        {
            return super._hasChildForAttribute(element, attributeName);
        }
    }
    
    @Override
    protected <T> Object _extractElementValue(Element parent, ElementDefinition<T> definition, Optional<Object> additionalData) throws Exception
    {
        String attributePath = definition.getPath();
        if (TraversableProgramPart.CHILD_PROGRAM_PARTS.equals(attributePath))
        {
            List<Content> childProgramParts = new ArrayList<>();
            List<Element> children = DOMUtils.getChildElements(parent);
            int courseListPosition = 0;
            for (Element child : children)
            {
                String tagName = child.getNodeName();
                if (_TAG_SUBPROGRAM.equals(tagName) || _TAG_CONTAINER.equals(tagName))
                {
                    String title = DOMUtils.getChildElementByTagName(child, "title").getTextContent();
                    String subContentSyncCode = child.getAttribute("CDMid");
                    ContentWorkflowDescription workflowDescription = _TAG_SUBPROGRAM.equals(tagName) ? ContentWorkflowDescription.SUBPROGRAM_WF_DESCRIPTION : ContentWorkflowDescription.CONTAINER_WF_DESCRIPTION;
                    Content subContent = _component.importOrSynchronizeContent(child, workflowDescription, title, subContentSyncCode, _context);
                    CollectionUtils.addIgnoreNull(childProgramParts, subContent);
                }
                else if (_TAG_COURSELIST.equals(tagName))
                {
                    String title = _getAttributeOrDefault(child, "name", "Liste d'éléments pédagogiques");
                    // For courseList from another source than Ametys, there is no unique code, then a code is generated with the parent ID and the position in the parent : [parentId]-[position]
                    courseListPosition++;
                    String subContentSyncCode = _getAttributeOrDefault(child, "code", _synchronizationCode + "-" + courseListPosition);
                    Content subContent = _component.importOrSynchronizeContent(child, ContentWorkflowDescription.COURSELIST_WF_DESCRIPTION, title, subContentSyncCode, _context);
                    CollectionUtils.addIgnoreNull(childProgramParts, subContent);
                }
            }
            
            return childProgramParts.toArray(new Content[childProgramParts.size()]);
        }
        else
        {
            return super._extractElementValue(parent, definition, additionalData);
        }
    }

}
