/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.pegase.ws;

import java.io.IOException;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.util.HttpUtils;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

import fr.pcscol.pegase.odf.ApiClient;
import fr.pcscol.pegase.odf.api.EspacesExterneApi;
import fr.pcscol.pegase.odf.api.MaquettesApi;
import fr.pcscol.pegase.odf.api.MaquettesExterneApi;
import fr.pcscol.pegase.odf.api.ObjetsMaquetteApi;
import fr.pcscol.pegase.odf.api.ObjetsMaquetteExterneApi;

/**
 * Manager for Pégase API.
 * Get or create API objects and configure it.
 */
public class PegaseApiManager extends AbstractLogEnabled implements Component, Serviceable, Initializable
{
    /** Role */
    public static final String ROLE = PegaseApiManager.class.getName();

    /** The Pégase token manager */
    protected PegaseTokenManager _pegaseTokenManager;
    
    /* Pégase APIs */
    private ObjetsMaquetteExterneApi _objetsMaquetteExterneApi;
    private MaquettesExterneApi _maquettesExterneApi;
    private ObjetsMaquetteApi _objetsMaquetteApi;
    private MaquettesApi _maquettesApi;
    private EspacesExterneApi _espacesExterneApi;
    
    /* Configuration */
    private boolean _isActive;
    private String _pegaseODFUrl;

    public void service(ServiceManager manager) throws ServiceException
    {
        _pegaseTokenManager = (PegaseTokenManager) manager.lookup(PegaseTokenManager.ROLE);
    }
    
    public void initialize() throws Exception
    {
        _isActive = Config.getInstance().getValue("pegase.activate", true, false);
        if (_isActive)
        {
            /* Configuration for API */
            _pegaseODFUrl = HttpUtils.sanitize(Config.getInstance().getValue("pegase.odf.url"));
        }
    }
    
    private void _updateApiClient(ApiClient apiClient, boolean external) throws IOException
    {
        apiClient.setBearerToken(_pegaseTokenManager.getToken());
        apiClient.setServerUrl(_pegaseODFUrl);
        apiClient.useExternalApi(external);
    }
    
    /**
     * Getter for objetsMaquetteExterneApi
     * @return objetsMaquetteExterneApi
     * @throws IOException If an error occurs while retrieving the token
     */
    public ObjetsMaquetteExterneApi getObjetsMaquetteExterneApi() throws IOException
    {
        if (_objetsMaquetteExterneApi == null)
        {
            _objetsMaquetteExterneApi = new ObjetsMaquetteExterneApi();
        }
        
        _updateApiClient(_objetsMaquetteExterneApi.getApiClient(), true);
        
        return _objetsMaquetteExterneApi;
    }

    /**
     * Getter for MaquettesApi
     * @return maquettesApi
     * @throws IOException If an error occurs while retrieving the token
     */
    public MaquettesExterneApi getMaquettesExterneApi() throws IOException
    {
        if (_maquettesExterneApi == null)
        {
            _maquettesExterneApi = new MaquettesExterneApi();
        }
        
        _updateApiClient(_maquettesExterneApi.getApiClient(), true);
        
        return _maquettesExterneApi;
    }
    
    /**
     * Getter for ObjetsMaquetteApi
     * @return objetsMaquetteApi
     * @throws IOException If an error occurs while retrieving the token
     */
    public ObjetsMaquetteApi getObjetsMaquetteApi() throws IOException
    {
        if (_objetsMaquetteApi == null)
        {
            _objetsMaquetteApi = new ObjetsMaquetteApi();
        }
        
        _updateApiClient(_objetsMaquetteApi.getApiClient(), false);
        
        return _objetsMaquetteApi;
    }
    
    /**
     * Getter for ObjetsMaquetteApi
     * @return objetsMaquetteApi
     * @throws IOException If an error occurs while retrieving the token
     */
    public MaquettesApi getMaquettesApi() throws IOException
    {
        if (_maquettesApi == null)
        {
            _maquettesApi = new MaquettesApi();
        }
        
        _updateApiClient(_maquettesApi.getApiClient(), false);
        
        return _maquettesApi;
    }
    
    /**
     * Getter for EspacesExterneApi
     * @return espacesExterneApi
     * @throws IOException If an error occurs while retrieving the token
     */
    public EspacesExterneApi getEspacesExterneApi() throws IOException
    {
        if (_espacesExterneApi == null)
        {
            _espacesExterneApi = new EspacesExterneApi();
        }
        
        _updateApiClient(_espacesExterneApi.getApiClient(), true);
        
        return _espacesExterneApi;
    }
}
