/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.cdmfr.extractor;

import java.util.Arrays;
import java.util.Collection;
import java.util.Optional;

import org.w3c.dom.Element;

import org.ametys.cms.repository.Content;
import org.ametys.core.util.dom.DOMUtils;
import org.ametys.odf.coursepart.CoursePart;
import org.ametys.plugins.odfsync.cdmfr.ImportCDMFrContext;
import org.ametys.plugins.odfsync.cdmfr.components.ImportCDMFrComponent;
import org.ametys.plugins.odfsync.utils.ContentWorkflowDescription;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.ModelItemContainer;

/**
 * This class provides methods to extract course part's values from a CMD-fr import document
 */
public class ImportCoursePartValuesExtractor extends ImportCDMFrValuesExtractor
{
    /**
     * Creates an import course part values extractor
     * @param element the DOM element containing the XML values
     * @param factory the values extractor factory
     * @param component the import CDM-fr component
     * @param context the import context
     * @param models the model of the extracted values
     */
    public ImportCoursePartValuesExtractor(Element element, ImportCDMFrValuesExtractorFactory factory, ImportCDMFrComponent component, ImportCDMFrContext context, Model... models)
    {
        this(element, factory, component, context, Arrays.asList(models));
    }
    
    /**
     * Creates an import course part values extractor
     * @param element the DOM element containing the XML values
     * @param factory the values extractor factory
     * @param component the import CDM-fr component
     * @param context the imported content's context
     * @param modelItemContainers the model of the extracted values
     */
    public ImportCoursePartValuesExtractor(Element element, ImportCDMFrValuesExtractorFactory factory, ImportCDMFrComponent component, ImportCDMFrContext context, Collection<? extends ModelItemContainer> modelItemContainers)
    {
        super(element, factory, component, context, modelItemContainers);
    }
    
    @Override
    protected <T> Object _extractElementValue(Element parent, ElementDefinition<T> definition, Optional<Object> additionalData) throws Exception
    {
        if (CoursePart.COURSE_HOLDER.equals(definition.getName()))
        {
            String syncCode = DOMUtils.getChildElementByTagName(parent, CoursePart.COURSE_HOLDER).getTextContent().trim();
            Content content = _component.getContent(ContentWorkflowDescription.COURSE_WF_DESCRIPTION.getContentType(), syncCode, _context);

            if (content == null)
            {
                _context.getLogger().warn("There is no course corresponding to the CDM ID '{}'.", syncCode);
            }
            
            return content;
        }
        else
        {
            return super._extractElementValue(parent, definition, additionalData);
        }
    }
}
