/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.ui.AddTaskClientSideElement;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollectionHelper;

/**
 * This implementation test if at least one SCC is associated to the model defined by the sccModelId.
 * If yes, the first collection is used to create the button, it there are no collections, the button is not displayed.
 * Also, it set some elements to be used by the import tool (SCCSearchTool for most cases).
 */
public class GlobalSynchronizationClientSideElement extends AddTaskClientSideElement
{
    /** SCC helper */
    protected SynchronizableContentsCollectionHelper _sccHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _sccHelper = (SynchronizableContentsCollectionHelper) smanager.lookup(SynchronizableContentsCollectionHelper.ROLE);
    }
    
    @Override
    public List<Script> getScripts(boolean ignoreRights, Map<String, Object> contextParameters)
    {
        List<Script> clonedScripts = new ArrayList<>();
        
        List<Script> scripts = super.getScripts(ignoreRights, contextParameters);
        
        for (Script script : scripts)
        {
            if (script.getParameters().containsKey("modelsToSync"))
            {
                String[] modelsToSync = ((String) script.getParameters().get("modelsToSync")).split(",");
                clonedScripts.addAll(getClonedScripts(script, modelsToSync));
            }
        }
        
        return clonedScripts;
    }
    
    /**
     * Get cloned scripts from the original script.
     * @param script Original script
     * @param modelsToSync SCC models to synchronize
     * @return {@link List} of modified scripts
     */
    protected List<Script> getClonedScripts(Script script, String[] modelsToSync)
    {
        List<Script> clonedScripts = new ArrayList<>();
        
        for (String sccModelId : modelsToSync)
        {
            SynchronizableContentsCollection collection = _sccHelper.getSCCFromModelId(sccModelId);
            if (collection != null)
            {
                Script clonedScript = new Script(script);
                
                Map<String, Object> openToolParams = new HashMap<>();
                openToolParams.put("collectionId", collection.getId());
                clonedScript.getParameters().put("schedulable-parameters", openToolParams);
                
                clonedScripts.add(clonedScript);
            }
        }
        
        return clonedScripts;
    }
}
