/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.apogee.scc;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.sql.Clob;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.Constants;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.collections4.ListUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.schedule.progression.ProgressionTrackerFactory;
import org.ametys.core.util.JSONUtils;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.cdmfr.CDMFRHandler;
import org.ametys.plugins.contentio.synchronize.AbstractSimpleSynchronizableContentsCollection;
import org.ametys.plugins.contentio.synchronize.SynchronizableContentsCollection;
import org.ametys.plugins.odfsync.apogee.ApogeeDAO;
import org.ametys.plugins.odfsync.apogee.scc.impl.OrgUnitSynchronizableContentsCollection;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

import com.google.common.base.CharMatcher;

/**
 * Abstract class for Apogee synchronization
 */
public abstract class AbstractApogeeSynchronizableContentsCollection extends AbstractSimpleSynchronizableContentsCollection implements Contextualizable, ApogeeSynchronizableContentsCollection
{
    /** Name of parameter holding the data source id */
    public static final String PARAM_DATASOURCE_ID = "datasourceId";
    /** Name of parameter holding the administrative year */
    public static final String PARAM_YEAR = "year";
    /** Name of parameter holding the adding unexisting children parameter  */
    public static final String PARAM_ADD_UNEXISTING_CHILDREN = "add-unexisting-children";
    /** Name of parameter holding the link existing children parameter  */
    public static final String PARAM_ADD_EXISTING_CHILDREN = "add-existing-children";
    /** Name of parameter holding the field ID column */
    protected static final String __PARAM_ID_COLUMN = "idColumn";
    /** Name of parameter holding the fields mapping */
    protected static final String __PARAM_MAPPING = "mapping";
    /** Name of parameter into mapping holding the synchronized property */
    protected static final String __PARAM_MAPPING_SYNCHRO = "synchro";
    /** Name of parameter into mapping holding the path of metadata */
    protected static final String __PARAM_MAPPING_METADATA_REF = "metadata-ref";
    /** Name of parameter into mapping holding the remote attribute */
    protected static final String __PARAM_MAPPING_ATTRIBUTE = "attribute";
    /** Name of parameter holding the criteria */
    protected static final String __PARAM_CRITERIA = "criteria";
    /** Name of parameter into criteria holding a criterion */
    protected static final String __PARAM_CRITERIA_CRITERION = "criterion";
    /** Name of parameter into criterion holding the id */
    protected static final String __PARAM_CRITERIA_CRITERION_ID = "id";
    /** Name of parameter into criterion holding the label */
    protected static final String __PARAM_CRITERIA_CRITERION_LABEL = "label";
    /** Name of parameter into criterion holding the type */
    protected static final String __PARAM_CRITERIA_CRITERION_TYPE = "type";
    /** Name of paramter holding columns */
    protected static final String __PARAM_COLUMNS = "columns";
    /** Name of paramter into columns holding column */
    protected static final String __PARAM_COLUMNS_COLUMN = "column";
    
    /** Parameter value to add unexisting children from Ametys on the current element */
    protected Boolean _addUnexistingChildren;

    /** Parameter value to add existing children in Ametys on the current element */
    protected Boolean _addExistingChildren;

    /** Name of the Apogée column which contains the ID */
    protected String _idColumn;
    
    /** Mapping between metadata and columns */
    protected Map<String, List<String>> _mapping;
    
    /** Synchronized fields */
    protected Set<String> _syncFields;
    
    /** Synchronized fields */
    protected Set<String> _columns;
    
    /** Synchronized fields */
    protected Set<ApogeeCriterion> _criteria;
    
    /** Context */
    protected Context _context;
    
    /** The DAO for remote DB Apogee */
    protected ApogeeDAO _apogeeDAO;
    
    /** The JSON utils */
    protected JSONUtils _jsonUtils;
    
    /** The ODF helper */
    protected ODFHelper _odfHelper;
    
    /** The Apogee SCC helper */
    protected ApogeeSynchronizableContentsCollectionHelper _apogeeSCCHelper;
    
    /** The CDM-fr handler */
    protected CDMFRHandler _cdmfrHandler;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _apogeeDAO = (ApogeeDAO) manager.lookup(ApogeeDAO.ROLE);
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _apogeeSCCHelper = (ApogeeSynchronizableContentsCollectionHelper) manager.lookup(ApogeeSynchronizableContentsCollectionHelper.ROLE);
        _cdmfrHandler = (CDMFRHandler) manager.lookup(CDMFRHandler.ROLE);
    }

    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    protected void configureDataSource(Configuration configuration) throws ConfigurationException
    {
        try
        {
            org.apache.cocoon.environment.Context ctx = (org.apache.cocoon.environment.Context) _context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
            File apogeeMapping = new File(ctx.getRealPath("/WEB-INF/param/odf/apogee-mapping.xml"));
            
            try (InputStream is = apogeeMapping.isFile()
                    ? new FileInputStream(apogeeMapping)
                    : getClass().getResourceAsStream("/org/ametys/plugins/odfsync/apogee/apogee-mapping.xml"))
            {
                Configuration cfg = new DefaultConfigurationBuilder().build(is);
                Configuration child = cfg.getChild(getMappingName());
                if (child != null)
                {
                    _criteria = new LinkedHashSet<>();
                    _columns = new LinkedHashSet<>();
                    _idColumn = child.getChild(__PARAM_ID_COLUMN).getValue();
                    _mapping = new HashMap<>();
                    _syncFields = new HashSet<>();
                    String mappingAsString = child.getChild(__PARAM_MAPPING).getValue();
                    _mapping.put(getIdField(), List.of(getIdColumn()));
                    if (StringUtils.isNotEmpty(mappingAsString))
                    {
                        List<Object> mappingAsList = _jsonUtils.convertJsonToList(mappingAsString);
                        for (Object object : mappingAsList)
                        {
                            @SuppressWarnings("unchecked")
                            Map<String, Object> field = (Map<String, Object>) object;
                            
                            String metadataRef = (String) field.get(__PARAM_MAPPING_METADATA_REF);
                            
                            String[] attributes = ((String) field.get(__PARAM_MAPPING_ATTRIBUTE)).split(",");
                            _mapping.put(metadataRef, Arrays.asList(attributes));
        
                            boolean isSynchronized = field.containsKey(__PARAM_MAPPING_SYNCHRO) ? (Boolean) field.get(__PARAM_MAPPING_SYNCHRO) : false;
                            if (isSynchronized)
                            {
                                _syncFields.add(metadataRef);
                            }
                        }
                    }
    
                    Configuration[] criteria = child.getChild(__PARAM_CRITERIA).getChildren(__PARAM_CRITERIA_CRITERION);
                    for (Configuration criterion : criteria)
                    {
                        String id = criterion.getChild(__PARAM_CRITERIA_CRITERION_ID).getValue();
                        I18nizableText label = _getCriterionLabel(criterion.getChild(__PARAM_CRITERIA_CRITERION_LABEL), id);
                        String type = criterion.getChild(__PARAM_CRITERIA_CRITERION_TYPE).getValue("STRING");
                        
                        _criteria.add(new ApogeeCriterion(id, label, type));
                    }
    
                    Configuration[] columns = child.getChild(__PARAM_COLUMNS).getChildren(__PARAM_COLUMNS_COLUMN);
                    for (Configuration column : columns)
                    {
                        _columns.add(column.getValue());
                    }
                }
            }
        }
        catch (Exception e)
        {
            throw new ConfigurationException("Error while parsing apogee-mapping.xml", e);
        }
    }
    
    private I18nizableText _getCriterionLabel(Configuration configuration, String defaultValue)
    {
        if (configuration.getAttributeAsBoolean("i18n", false))
        {
            return new I18nizableText("plugin.odf-sync", configuration.getValue(defaultValue));
        }
        else
        {
            return new I18nizableText(configuration.getValue(defaultValue));
        }
    }
    
    @Override
    public List<ModifiableContent> populate(Logger logger, ContainerProgressionTracker progressionTracker)
    {
        boolean isRequestAttributeOwner = false;
        
        Request request = ContextHelper.getRequest(_context);
        
        try
        {
            if (request.getAttribute(ApogeeSynchronizableContentsCollectionHelper.HANDLED_CONTENTS) == null)
            {
                _startHandleCDMFR();
                request.setAttribute(ApogeeSynchronizableContentsCollectionHelper.HANDLED_CONTENTS, new HashSet<>());
                isRequestAttributeOwner = true;
            }
            
            return super.populate(logger, progressionTracker);
        }
        finally
        {
            if (isRequestAttributeOwner)
            {
                _endHandleCDMFR(request);
                request.removeAttribute(ApogeeSynchronizableContentsCollectionHelper.HANDLED_CONTENTS);
            }
        }
    }
    
    @Override
    protected List<ModifiableContent> _internalPopulate(Logger logger, ContainerProgressionTracker progressionTracker)
    {
        return _importOrSynchronizeContents(Map.of("isGlobalSync", true), false, logger, progressionTracker);
    }
    
    @Override
    protected Map<String, Map<String, Object>> internalSearch(Map<String, Object> searchParameters, int offset, int limit, List<Object> sort, Logger logger)
    {
        Map<String, Object> searchParams = new HashMap<>(searchParameters);
        if (offset > 0)
        {
            searchParams.put("__offset", offset);
        }
        if (limit < Integer.MAX_VALUE)
        {
            searchParams.put("__limit", offset + limit);
        }
        searchParams.put("__order", _getSort(sort));

        // We don't use session.selectMap which reorder data
        List<Map<String, Object>> requestValues = _search(searchParams, logger);

        // Transform CLOBs to String
        Set<String> clobColumns = getClobColumns();
        if (!clobColumns.isEmpty())
        {
            for (Map<String, Object> contentValues : requestValues)
            {
                String idValue = contentValues.get(getIdColumn()).toString();
                for (String clobKey : getClobColumns())
                {
                    // Get the old values for the CLOB
                    @SuppressWarnings("unchecked")
                    Optional<List<Object>> oldValues = Optional.of(clobKey)
                        .map(contentValues::get)
                        .map(obj -> (List<Object>) obj);
                    
                    if (oldValues.isPresent())
                    {
                        // Get the new values for the CLOB
                        List<Object> newValues = oldValues.get()
                            .stream()
                            .map(value -> _transformClobToString(value, idValue, logger))
                            .filter(Objects::nonNull)
                            .collect(Collectors.toList());
                        
                        // Set the transformed CLOB values
                        contentValues.put(clobKey, newValues);
                    }
                }
            }
        }
        
        // Reorganize results
        String idColumn = getIdColumn();
        Map<String, Map<String, Object>> results = new LinkedHashMap<>();
        for (Map<String, Object> contentValues : requestValues)
        {
            results.put(contentValues.get(idColumn).toString(), contentValues);
        }
        
        for (Map<String, Object> result : results.values())
        {
            result.put(SCC_UNIQUE_ID, result.get(getIdColumn()));
        }
        
        return results;
    }
    
    @Override
    protected Map<String, Map<String, List<Object>>> getRemoteValues(Map<String, Object> searchParameters, Logger logger)
    {
        Map<String, Map<String, List<Object>>> remoteValues = new HashMap<>();
        
        Map<String, Map<String, Object>> results = internalSearch(searchParameters, 0, Integer.MAX_VALUE, null, logger);
        
        if (results != null)
        {
            remoteValues = _sccHelper.organizeRemoteValuesByAttribute(results, _mapping);
        }
        
        return remoteValues;
    }
    
    @Override
    public List<String> getLanguages()
    {
        return List.of(_apogeeSCCHelper.getSynchronizationLang());
    }
    
    @Override
    public List<ModifiableContent> importContent(String idValue, Map<String, Object> additionalParameters, Logger logger) throws Exception
    {
        boolean isRequestAttributeOwner = false;
        
        Request request = ContextHelper.getRequest(_context);
        
        try
        {
            if (request.getAttribute(ApogeeSynchronizableContentsCollectionHelper.HANDLED_CONTENTS) == null)
            {
                _startHandleCDMFR();
                request.setAttribute(ApogeeSynchronizableContentsCollectionHelper.HANDLED_CONTENTS, new HashSet<>());
                isRequestAttributeOwner = true;
            }
    
            return super.importContent(idValue, additionalParameters, logger);
        }
        finally
        {
            if (isRequestAttributeOwner)
            {
                _endHandleCDMFR(request);
                request.removeAttribute(ApogeeSynchronizableContentsCollectionHelper.HANDLED_CONTENTS);
            }
        }
    }
    
    @Override
    public void synchronizeContent(ModifiableContent content, Logger logger) throws Exception
    {
        boolean isRequestAttributeOwner = false;
        
        Request request = ContextHelper.getRequest(_context);
        
        try
        {
            if (request.getAttribute(ApogeeSynchronizableContentsCollectionHelper.HANDLED_CONTENTS) == null)
            {
                _startHandleCDMFR();
                request.setAttribute(ApogeeSynchronizableContentsCollectionHelper.HANDLED_CONTENTS, new HashSet<>());
                _addContentAttributes(request, content);
                isRequestAttributeOwner = true;
            }
            
            super.synchronizeContent(content, logger);
        }
        finally
        {
            if (isRequestAttributeOwner)
            {
                _endHandleCDMFR(request);
                request.removeAttribute(ApogeeSynchronizableContentsCollectionHelper.HANDLED_CONTENTS);
                _removeContentAttributes(request);
            }
        }
    }
    
    /**
     * Start handle CDM-fr treatments
     */
    protected void _startHandleCDMFR()
    {
        _cdmfrHandler.suspendCDMFRObserver();
    }
    
    /**
     * End handle CDM-fr treatments
     * @param request the request
     */
    protected void _endHandleCDMFR(Request request)
    {
        @SuppressWarnings("unchecked")
        Set<String> handledContents = (Set<String>) request.getAttribute(ApogeeSynchronizableContentsCollectionHelper.HANDLED_CONTENTS);
        _cdmfrHandler.unsuspendCDMFRObserver(handledContents);
    }
    
    /**
     * Add attributes from content in the request.
     * @param request The request
     * @param content The content
     */
    protected void _addContentAttributes(Request request, ModifiableContent content)
    {
        request.setAttribute(ApogeeSynchronizableContentsCollectionHelper.LANG, content.getLanguage());
    }
    
    /**
     * Remove attributes of the content from the request.
     * @param request The request
     */
    protected void _removeContentAttributes(Request request)
    {
        request.removeAttribute(ApogeeSynchronizableContentsCollectionHelper.LANG);
    }
    
    @Override
    protected Map<String, Object> putIdParameter(String idValue)
    {
        Map<String, Object> parameters = new HashMap<>();
        parameters.put(getIdField(), idValue);
        return parameters;
    }
    
    /**
     * Search the contents with the search parameters. Use id parameter to search an unique content.
     * @param searchParameters Search parameters
     * @param logger The logger
     * @return A Map of mapped metadatas extract from Apogée database ordered by content unique Apogée ID
     */
    protected abstract List<Map<String, Object>> _search(Map<String, Object> searchParameters, Logger logger);
    
    /**
     * Convert the {@link BigDecimal} values retrieved from database into long values
     * @param searchResults The initial search results from database
     * @return The converted search results
     */
    protected List<Map<String, Object>> _convertBigDecimal(List<Map<String, Object>> searchResults)
    {
        List<Map<String, Object>> convertedSearchResults = new ArrayList<>();
        
        for (Map<String, Object> searchResult : searchResults)
        {
            for (String key : searchResult.keySet())
            {
                searchResult.put(key, _convertBigDecimal(getContentType(), key, searchResult.get(key)));
            }
            
            convertedSearchResults.add(searchResult);
        }
        
        return convertedSearchResults;
    }
    
    /**
     * Convert the object in parameter to a long if it's a {@link BigDecimal}, otherwise return the object itself.
     * @param contentTypeId The content type of the parent content
     * @param attributeName The metadata name
     * @param objectToConvert The object to convert if necessary
     * @return The converted object
     */
    protected Object _convertBigDecimal(String contentTypeId, String attributeName, Object objectToConvert)
    {
        if (objectToConvert instanceof BigDecimal)
        {
            ContentType contentType = _contentTypeEP.getExtension(contentTypeId);
            if (contentType.hasModelItem(attributeName))
            {
                ModelItem definition = contentType.getModelItem(attributeName);
                String typeId = definition.getType().getId();
                switch (typeId)
                {
                    case ModelItemTypeConstants.DOUBLE_TYPE_ID:
                        return ((BigDecimal) objectToConvert).doubleValue();
                    case ModelItemTypeConstants.LONG_TYPE_ID:
                        return ((BigDecimal) objectToConvert).longValue();
                    default:
                        // Do nothing
                        break;
                }
            }
            return ((BigDecimal) objectToConvert).toString();
        }
        return objectToConvert;
    }
    
    /**
     * Transform CLOB value to String value.
     * @param value The input value
     * @param idValue The identifier of the program
     * @param logger The logger
     * @return the same value, with CLOB transformed to String.
     */
    protected Object _transformClobToString(Object value, String idValue, Logger logger)
    {
        if (value instanceof Clob)
        {
            Clob clob = (Clob) value;
            try
            {
                String strValue = IOUtils.toString(clob.getCharacterStream());
                return CharMatcher.javaIsoControl().and(CharMatcher.anyOf("\r\n\t").negate()).removeFrom(strValue);
            }
            catch (SQLException | IOException e)
            {
                logger.error("Unable to get education add elements from the program '{}'.", idValue, e);
                return null;
            }
            finally
            {
                try
                {
                    clob.free();
                }
                catch (SQLException e)
                {
                    // Ignore the exception.
                }
            }
        }
        
        return value;
    }

    /**
     * Get the list of CLOB column's names.
     * @return The list of the CLOB column's names to transform to {@link String}
     */
    protected Set<String> getClobColumns()
    {
        return Set.of();
    }
    
    /**
     * Transform a {@link List} of {@link Map} to a {@link Map} of {@link List} computed by keys (lines to columns).
     * @param lines {@link List} to reorganize
     * @return {@link Map} of {@link List}
     */
    protected Map<String, List<Object>> _transformListOfMap2MapOfList(List<Map<String, Object>> lines)
    {
        return lines.stream()
            .filter(Objects::nonNull)
            .map(Map::entrySet)
            .flatMap(Collection::stream)
            .filter(e -> Objects.nonNull(e.getValue()))
            .collect(
                Collectors.toMap(
                    Entry::getKey,
                    e -> List.of(e.getValue()),
                    (l1, l2) -> ListUtils.union(l1, l2)
                )
            );
    }
    
    /**
     * Get the name of the mapping.
     * @return the mapping name
     */
    protected abstract String getMappingName();
    
    /**
     * Get the id of data source
     * @return The id of data source
     */
    protected String getDataSourceId()
    {
        return (String) getParameterValues().get(PARAM_DATASOURCE_ID);
    }
    
    /**
     * Get the administrative year
     * @return The administrative year
     */
    protected String getYear()
    {
        return (String) getParameterValues().get(PARAM_YEAR);
    }
    
    /**
     * Check if unexisting contents in Ametys should be added from data source
     * @return <code>true</code> if unexisting contents in Ametys should be added from data source, default value is <code>true</code>
     */
    protected boolean addUnexistingChildren()
    {
        // This parameter is read many times so we store it
        if (_addUnexistingChildren == null)
        {
            _addUnexistingChildren = Boolean.valueOf((String) getParameterValues().getOrDefault(PARAM_ADD_UNEXISTING_CHILDREN, Boolean.TRUE.toString()));
        }
        return _addUnexistingChildren;
    }
    
    /**
     * Check if existing contents in Ametys should be added from data source
     * @return <code>true</code> if existing contents in Ametys should be added from data source, default value is <code>false</code>
     */
    protected boolean addExistingChildren()
    {
        // This parameter is read many times so we store it
        if (_addExistingChildren == null)
        {
            _addExistingChildren = Boolean.valueOf((String) getParameterValues().getOrDefault(PARAM_ADD_EXISTING_CHILDREN, Boolean.FALSE.toString()));
        }
        return _addExistingChildren;
    }
    
    /**
     * Get the identifier column (can be a concatened column).
     * @return the column id
     */
    protected String getIdColumn()
    {
        return _idColumn;
    }
    
    @Override
    public String getIdField()
    {
        return "apogeeSyncCode";
    }

    @Override
    public Set<String> getLocalAndExternalFields(Map<String, Object> additionalParameters)
    {
        return _syncFields;
    }
    
    @Override
    protected void configureSearchModel()
    {
        for (ApogeeCriterion criterion : _criteria)
        {
            _searchModelConfiguration.addCriterion(criterion.getId(), criterion.getLabel(), criterion.getType());
        }
        for (String columnName : _columns)
        {
            _searchModelConfiguration.addColumn(columnName);
        }
    }
    
    @Override
    protected Map<String, Object> getAdditionalAttributeValues(String idValue, Content content, Map<String, Object> additionalParameters, boolean create, Logger logger)
    {
        Map<String, Object> additionalValues = super.getAdditionalAttributeValues(idValue, content, additionalParameters, create, logger);
        
        // Handle parents
        getParentFromAdditionalParameters(additionalParameters)
            .map(this::getParentAttribute)
            .ifPresent(attribute -> additionalValues.put(attribute.getKey(), attribute.getValue()));
        
        // Handle children
        String childrenAttributeName = getChildrenAttributeName();
        if (childrenAttributeName != null)
        {
            List<ModifiableContent> children = handleChildren(idValue, content, create, logger);
            additionalValues.put(childrenAttributeName, children.toArray(new ModifiableContent[children.size()]));
        }
        
        return additionalValues;
    }
    
    /**
     * Retrieves the attribute to synchronize for the given parent (as a {@link Pair} of name and value)
     * @param parent the parent content
     * @return the parent attribute
     */
    protected Pair<String, Object> getParentAttribute(ModifiableContent parent)
    {
        return null;
    }
    
    /**
     * Import and synchronize children of the given content
     * In the usual case, if we are in import mode (create to true) or if we trust the source (removalSync to true) we just synchronize structure but we don't synchronize child contents
     * @param idValue The current content synchronization code
     * @param content The current content
     * @param create <code>true</code> if the content has been newly created
     * @param logger The logger
     * @return The handled children
     */
    protected List<ModifiableContent> handleChildren(String idValue, Content content, boolean create, Logger logger)
    {
        return importOrSynchronizeChildren(idValue, content, getChildrenSCCModelId(), getChildrenAttributeName(), create, logger);
    }
    
    @Override
    protected Set<String> getNotSynchronizedRelatedContentIds(Content content, Map<String, Object> contentValues, Map<String, Object> additionalParameters, String lang, Logger logger)
    {
        Set<String> contentIds = super.getNotSynchronizedRelatedContentIds(content, contentValues, additionalParameters, lang, logger);
        
        getParentIdFromAdditionalParameters(additionalParameters)
            .ifPresent(contentIds::add);
        
        return contentIds;
    }
    
    /**
     * Retrieves the parent id, extracted from additional parameters
     * @param additionalParameters the additional parameters
     * @return the parent id
     */
    protected Optional<ModifiableContent> getParentFromAdditionalParameters(Map<String, Object> additionalParameters)
    {
        return getParentIdFromAdditionalParameters(additionalParameters)
                .map(_resolver::resolveById);
    }

    /**
     * Retrieves the parent id, extracted from additional parameters
     * @param additionalParameters the additional parameters
     * @return the parent id
     */
    protected Optional<String> getParentIdFromAdditionalParameters(Map<String, Object> additionalParameters)
    {
        return Optional.ofNullable(additionalParameters)
                .filter(params -> params.containsKey("parentId"))
                .map(params -> params.get("parentId"))
                .filter(String.class::isInstance)
                .map(String.class::cast);
    }
    
    /**
     * Get the children SCC model id. Can be null if no implementation is defined.
     * @return the children SCC model id
     */
    protected String getChildrenSCCModelId()
    {
        // Default implementation
        return null;
    }
    
    /**
     * Get the attribute name to get children
     * @return the attribute name to get children
     */
    protected abstract String getChildrenAttributeName();

    /**
     * Transform the given {@link List} of {@link Object} to a {@link String} representing the ordered fields for SQL.
     * @param sortList The sort list object to transform to the list of ordered fields compatible with SQL.
     * @return A string representing the list of ordered fields
     */
    @SuppressWarnings("unchecked")
    protected String _getSort(List<Object> sortList)
    {
        if (sortList != null)
        {
            StringBuilder sort = new StringBuilder();
            
            for (Object sortValueObj : sortList)
            {
                Map<String, Object> sortValue = (Map<String, Object>) sortValueObj;
                
                sort.append(sortValue.get("property"));
                if (sortValue.containsKey("direction"))
                {
                    sort.append(" ");
                    sort.append(sortValue.get("direction"));
                    sort.append(",");
                }
                else
                {
                    sort.append(" ASC,");
                }
            }
            
            sort.deleteCharAt(sort.length() - 1);
            
            return sort.toString();
        }
        
        return null;
    }
    
    @Override
    public int getTotalCount(Map<String, Object> searchParameters, Logger logger)
    {
        // Remove empty parameters
        Map<String, Object> searchParams = new HashMap<>();
        for (String parameterName : searchParameters.keySet())
        {
            Object parameterValue = searchParameters.get(parameterName);
            if (parameterValue != null && !parameterValue.toString().isEmpty())
            {
                searchParams.put(parameterName, parameterValue);
            }
        }
        
        searchParams.put("__count", true);
        
        List<Map<String, Object>> results = _search(searchParams, logger);
        if (results != null && !results.isEmpty())
        {
            return Integer.valueOf(results.get(0).get("COUNT(*)").toString()).intValue();
        }
        
        return 0;
    }
    
    @Override
    protected ModifiableContent _importContent(String idValue, Map<String, Object> additionalParameters, String lang, Map<String, List<Object>> remoteValues, Logger logger) throws Exception
    {
        ModifiableContent content = super._importContent(idValue, additionalParameters, lang, _transformOrgUnitAttribute(remoteValues, logger), logger);
        if (content != null)
        {
            _apogeeSCCHelper.addToHandleContents(content.getId());
        }
        return content;
    }
    
    @Override
    protected ModifiableContent _synchronizeContent(ModifiableContent content, Map<String, List<Object>> remoteValues, Logger logger) throws Exception
    {
        if (!_apogeeSCCHelper.addToHandleContents(content.getId()))
        {
            return content;
        }
        return super._synchronizeContent(content, _transformOrgUnitAttribute(remoteValues, logger), logger);
    }
    
    /**
     * Import and synchronize children of the given content
     * In the usual case, if we are in import mode (create to true) or if we trust the source (removalSync to true) we just synchronize structure but we don't synchronize child contents
     * @param idValue The parent content synchronization code
     * @param content The parent content
     * @param sccModelId SCC model ID
     * @param attributeName The name of the attribute containing children
     * @param create <code>true</code> if the content has been newly created
     * @param logger The logger
     * @return The imported or synchronized children
     */
    protected List<ModifiableContent> importOrSynchronizeChildren(String idValue, Content content, String sccModelId, String attributeName, boolean create, Logger logger)
    {
        // Get the SCC for children
        SynchronizableContentsCollection scc = _sccHelper.getSCCFromModelId(sccModelId);
        
        return create
                // Import mode
                ? _importChildren(idValue, scc, logger)
                // Synchronization mode
                : _synchronizeChildren(content, scc, attributeName, logger);
    }
    
    /**
     * Import children
     * @param idValue The parent content synchronization code
     * @param scc The SCC
     * @param logger The logger
     * @return The imported children
     */
    protected List<ModifiableContent> _importChildren(String idValue, SynchronizableContentsCollection scc, Logger logger)
    {
        // If SCC exists, search for children
        if (scc != null && scc instanceof ApogeeSynchronizableContentsCollection)
        {
            // Synchronize or import children content
            return ((ApogeeSynchronizableContentsCollection) scc).importOrSynchronizeContents(_getChildrenSearchParametersWithParent(idValue), logger);
        }
        
        return List.of();
    }

    /**
     * Synchronize children
     * @param content Parent content
     * @param scc The SCC
     * @param attributeName The name of the attribute containing children
     * @param logger The logger
     * @return <code>true</code> if there are changes
     */
    protected List<ModifiableContent> _synchronizeChildren(Content content, SynchronizableContentsCollection scc, String attributeName, Logger logger)
    {
        // First we get the existing children in Ametys
        List<ModifiableContent> ametysChildren = Stream.of(content.getValue(attributeName, false, new ContentValue[0]))
            .map(ContentValue::getContentIfExists)
            .flatMap(Optional::stream)
            .collect(Collectors.toList());

        // Get the children remote sync codes if needed
        // These remote sync codes are needed if we want to add contents in Ametys from Apogee or delete obsolete contents
        Set<String> childrenRemoteSyncCode = (removalSync() || addUnexistingChildren() || addExistingChildren())
            ? _getChildrenRemoteSyncCode(content, scc, logger)
            : null;
        
        // Can be null if we do not want to remove obsolete contents, add existing or unexisting children
        // Or if the current content is not from Apogee
        if (childrenRemoteSyncCode != null)
        {
            // Remove obsolete children if removalSync is active
            if (removalSync())
            {
                ametysChildren = ametysChildren.stream()
                    .filter(c -> !_isChildWillBeRemoved(c, scc, childrenRemoteSyncCode, logger))
                    .collect(Collectors.toList());
            }
            
            if (addExistingChildren() || addUnexistingChildren())
            {
                // Then we add missing children if needed
                for (String code : childrenRemoteSyncCode)
                {
                    ModifiableContent child = scc.getContent(_apogeeSCCHelper.getSynchronizationLang(), code, false);
                    
                    // If the child with the given sync code does not exist, import it if the parameter to
                    // add unexisting children is checked
                    if (child == null)
                    {
                        if (addUnexistingChildren())
                        {
                            ametysChildren.addAll(_importUnexistingChildren(scc, code, null, logger));
                        }
                    }
                    // If the parameter to link existing children not already in the list is checked,
                    // it adds the content to the children list if it is not already in it.
                    else if (addExistingChildren() && !ametysChildren.contains(child))
                    {
                        ametysChildren.add(child);
                    }
                }
            }
        }

        // Then we synchronize children in Ametys
        // (it won't be synchronized twice because it reads the request parameters with handled contents)
        for (ModifiableContent childContent : ametysChildren)
        {
            _apogeeSCCHelper.synchronizeContent(childContent, logger);
        }
        
        return ametysChildren;
    }

    /**
     * Get the remote sync codes
     * @param content Parent content
     * @param scc the scc
     * @param logger The logger
     * @return the remote sync codes or null if the scc is not from Apogee
     */
    protected Set<String> _getChildrenRemoteSyncCode(Content content, SynchronizableContentsCollection scc, Logger logger)
    {
        if (scc != null && scc instanceof AbstractApogeeSynchronizableContentsCollection)
        {
            String syncCode = content.getValue(getIdField());
            return ((AbstractApogeeSynchronizableContentsCollection) scc)
                        .search(_getChildrenSearchParametersWithParent(syncCode), 0, Integer.MAX_VALUE, null, logger)
                        .keySet();
        }
        
        return null;
    }
    
    /**
     * Get the children search parameters.
     * @param parentSyncCode The parent synchronization code
     * @return a {@link Map} of search parameters
     */
    protected Map<String, Object> _getChildrenSearchParametersWithParent(String parentSyncCode)
    {
        Map<String, Object> searchParameters = new HashMap<>();
        searchParameters.put("parentCode", parentSyncCode);
        return searchParameters;
    }
    
    /**
     * Import an unexisting child in Ametys
     * @param scc the scc to import the content
     * @param syncCode the sync code of the content
     * @param additionalParameters the additional params
     * @param logger the logger
     * @return the list of created children
     */
    @SuppressWarnings("unchecked")
    protected List<ModifiableContent> _importUnexistingChildren(SynchronizableContentsCollection scc, String syncCode, Map<String, List<Object>> additionalParameters, Logger logger)
    {
        try
        {
            return scc.importContent(syncCode, (Map<String, Object>) (Object) additionalParameters, logger);
        }
        catch (Exception e)
        {
            logger.error("An error occured while importing a new children content with syncCode '{}' from SCC '{}'", syncCode, scc.getId(), e);
        }
        
        return Collections.emptyList();
    }
    
    /**
     * True if the content will be removed from the structure
     * @param content the content
     * @param scc the scc
     * @param childrenRemoteSyncCode the remote sync codes
     * @param logger the logger
     * @return <code>true</code> if the content will be removed from the structure
     */
    protected boolean _isChildWillBeRemoved(ModifiableContent content, SynchronizableContentsCollection scc, Set<String> childrenRemoteSyncCode, Logger logger)
    {
        String syncCode = content.getValue(scc.getIdField());
        return !childrenRemoteSyncCode.contains(syncCode);
    }
    
    @Override
    public List<ModifiableContent> importOrSynchronizeContents(Map<String, Object> searchParams, Logger logger)
    {
        ContainerProgressionTracker containerProgressionTracker = ProgressionTrackerFactory.createContainerProgressionTracker("Import or synchronize contents", logger);
        
        return _importOrSynchronizeContents(searchParams, true, logger, containerProgressionTracker);
    }
    
    @SuppressWarnings("unchecked")
    private Map<String, List<Object>> _transformOrgUnitAttribute(Map<String, List<Object>> remoteValues, Logger logger)
    {
        // Transform orgUnit values and import content if necessary (useful for Course and SubProgram)
        SynchronizableContentsCollection scc = _sccHelper.getSCCFromModelId(OrgUnitSynchronizableContentsCollection.MODEL_ID);

        List<Object> orgUnitCodes = remoteValues.get("orgUnit");
        if (orgUnitCodes != null && !orgUnitCodes.isEmpty())
        {
            List<?> orgUnitContents = null;
            String orgUnitCode = orgUnitCodes.get(0).toString();
            
            if (scc != null)
            {
                try
                {
                    ModifiableContent orgUnitContent = scc.getContent(_apogeeSCCHelper.getSynchronizationLang(), orgUnitCode, false);
                    if (orgUnitContent == null)
                    {
                        orgUnitContents = scc.importContent(orgUnitCode, null, logger);
                    }
                    else
                    {
                        orgUnitContents = List.of(orgUnitContent);
                    }
                }
                catch (Exception e)
                {
                    logger.error("An error occured during the import of the OrgUnit identified by the synchronization code '{}'", orgUnitCode, e);
                }
            }
            
            if (orgUnitContents == null)
            {
                // Impossible link to orgUnit
                remoteValues.remove("orgUnit");
                logger.warn("Impossible to import the OrgUnit with the synchronization code '{}', check if you set the OrgUnit SCC with the following model ID: '{}'", orgUnitCode, OrgUnitSynchronizableContentsCollection.MODEL_ID);
            }
            else
            {
                remoteValues.put("orgUnit", (List<Object>) orgUnitContents);
            }
        }
        
        return remoteValues;
    }
    
    @Override
    public boolean handleRightAssignmentContext()
    {
        // Rights on ODF contents are handled by ODFRightAssignmentContext
        return false;
    }
}
