/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.cdmfr;

import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.cocoon.Constants;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.Context;
import org.apache.commons.collections.SetUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.io.comparator.LastModifiedFileComparator;
import org.apache.commons.io.comparator.NameFileComparator;
import org.apache.commons.io.comparator.SizeFileComparator;
import org.slf4j.Logger;

import org.ametys.cms.repository.ModifiableContent;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.schedule.progression.SimpleProgressionTracker;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;

import com.google.common.collect.Lists;

/**
 * Class for CDMFr import and synchronization
 */
public class CDMFrSynchronizableContentsCollection extends AbstractCDMFrSynchronizableContentsCollection implements Contextualizable
{
    /** Data source parameter : folder */
    protected static final String __PARAM_FOLDER = "folder";
    
    private static final String _CRITERIA_FILENAME = "filename";
    private static final String _CRITERIA_LAST_MODIFIED_AFTER = "lastModifiedAfter";
    private static final String _CRITERIA_LAST_MODIFIED_BEFORE = "lastModifiedBefore";
    private static final String _COLUMN_FILENAME = "filename";
    private static final String _COLUMN_LAST_MODIFIED = "lastModified";
    private static final String _COLUMN_LENGTH = "length";
    
    private static final Map<String, Comparator<File>> _NAME_TO_COMPARATOR = new HashMap<>();
    static
    {
        _NAME_TO_COMPARATOR.put(_COLUMN_FILENAME, NameFileComparator.NAME_INSENSITIVE_COMPARATOR);
        _NAME_TO_COMPARATOR.put(_COLUMN_LAST_MODIFIED, LastModifiedFileComparator.LASTMODIFIED_COMPARATOR);
        _NAME_TO_COMPARATOR.put(_COLUMN_LENGTH, SizeFileComparator.SIZE_COMPARATOR);
    }

    /** The Cocoon context */
    protected Context _cocoonContext;
    
    /** CDM-fr folder */
    protected File _cdmfrFolder;

    /** Default language configured for ODF */
    protected String _odfLang;
    
    /** List of synchronized contents (to avoid a treatment twice or more) */
    protected Set<String> _updatedContents;
    
    @Override
    public void contextualize(org.apache.avalon.framework.context.Context context) throws ContextException
    {
        _cocoonContext = (Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }

    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        super.configure(configuration);
        _updatedContents = new HashSet<>();
    }
    
    @SuppressWarnings("unchecked")
    @Override
    protected List<ModifiableContent> _internalPopulate(Logger logger, ContainerProgressionTracker progressionTracker)
    {
        _updatedContents.clear();
        
        try
        {
            _startHandleCDMFR();
    
            SimpleProgressionTracker progressionTrackerCDMfrFiles = progressionTracker.addSimpleStep("files", new I18nizableText("plugin.odf-sync", "PLUGINS_ODF_SYNC_GLOBAL_SYNCHRONIZATION_CDMFR_STEP_LABEL"));
            
            List<ModifiableContent> contents = new ArrayList<>();
            
            Map<String, Object> parameters = new HashMap<>();
            parameters.put("validateAfterImport", validateAfterImport());
            parameters.put("removalSync", removalSync());
            parameters.put("contentPrefix", getContentPrefix());
            parameters.put("collectionId", getId());
            
            File[] cdmfrFiles = _cdmfrFolder.listFiles(new CDMFrFileFilter(null, null, null));
            
            progressionTrackerCDMfrFiles.setSize(cdmfrFiles.length);

            for (File cdmfrFile : cdmfrFiles)
            {
                Map<String, Object> resultMap = _handleFile(cdmfrFile, parameters, logger);
                if (resultMap.containsKey("importedPrograms"))
                {
                    contents.addAll((List<ModifiableContent>) resultMap.remove("importedPrograms"));
                }
                parameters.putAll(resultMap);
                progressionTrackerCDMfrFiles.increment();
            }
            
            _updatedContents.addAll((Set<String>) parameters.getOrDefault("updatedContents", SetUtils.EMPTY_SET));
            _nbCreatedContents += (int) parameters.getOrDefault("nbCreatedContents", 0);
            _nbSynchronizedContents += (int) parameters.getOrDefault("nbSynchronizedContents", 0);
            _nbError += (int) parameters.getOrDefault("nbError", 0);
            
            return contents;
        }
        finally
        {
            _endHandleCDMFR(_updatedContents);
        }
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public List<ModifiableContent> importContent(String idValue, Map<String, Object> additionalParameters, Logger logger) throws Exception
    {
        _updatedContents.clear();
        
        try
        {
            _startHandleCDMFR();
            
            File cdmfrFile = new File(_cdmfrFolder, idValue);
            if (!cdmfrFile.exists())
            {
                logger.error("The file '{}' doesn't exists in the repository '{}'.", idValue, _cdmfrFolder.getAbsolutePath());
            }
            else if (!cdmfrFile.isFile())
            {
                logger.error("The element '{}' is not a file.", cdmfrFile.getAbsolutePath());
            }
            else
            {
                Map<String, Object> parameters = new HashMap<>();
                parameters.put("validateAfterImport", validateAfterImport());
                parameters.put("removalSync", removalSync());
                parameters.put("contentPrefix", getContentPrefix());
                parameters.put("collectionId", getId());
                
                Map<String, Object> resultMap = _handleFile(cdmfrFile, parameters, logger);
                
                _updatedContents.addAll((Set<String>) resultMap.getOrDefault("updatedContents", SetUtils.EMPTY_SET));
                _nbCreatedContents += (int) resultMap.getOrDefault("nbCreatedContents", 0);
                _nbSynchronizedContents += (int) resultMap.getOrDefault("nbSynchronizedContents", 0);
                _nbError += (int) resultMap.getOrDefault("nbError", 0);
                
                return (List<ModifiableContent>) resultMap.getOrDefault("importedPrograms", Collections.emptyList());
            }
        }
        finally
        {
            _endHandleCDMFR(_updatedContents);
        }
        
        return null;
    }

    /**
     * Handle the CDM-fr file to import all the programs and its dependencies containing into it.
     * @param cdmfrFile The CDM-fr file
     * @param parameters Parameters used to import the file
     * @param logger The logger
     * @return The list of imported/synchronized programs
     */
    protected Map<String, Object> _handleFile(File cdmfrFile, Map<String, Object> parameters, Logger logger)
    {
        String absolutePath = cdmfrFile.getAbsolutePath();
        
        logger.info("Processing CDM-fr file '{}'", absolutePath);
        
        try (InputStream fis = new FileInputStream(cdmfrFile))
        {
            return _importCDMFrComponent.handleInputStream(fis, parameters, this, logger);
        }
        catch (IOException e)
        {
            logger.error("An error occured while reading or closing the file {}.", absolutePath, e);
        }
        catch (ProcessingException e)
        {
            logger.error("An error occured while handling the file {}.", absolutePath, e);
        }
        catch (Exception e)
        {
            logger.error("An unknown error happens while handling the file {}.", absolutePath, e);
        }
        
        return new HashMap<>();
    }
    
    @Override
    public Map<String, Map<String, Object>> search(Map<String, Object> searchParameters, int offset, int limit, List<Object> sort, Logger logger)
    {
        File[] cdmfrFiles = internalSearch(searchParameters, logger);

        // Trier
        cdmfrFiles = _sortFiles(cdmfrFiles, sort);
        
        // Parser uniquement les fichiers entre offset et limit
        int i = 0;
        Map<String, Map<String, Object>> files = new LinkedHashMap<>();
        for (File cdmfrFile : cdmfrFiles)
        {
            if (i >= offset && i < offset + limit)
            {
                Map<String, Object> file = new HashMap<>();
                file.put(SCC_UNIQUE_ID, cdmfrFile.getName());
                file.put(_COLUMN_FILENAME, cdmfrFile.getName());
                file.put(_COLUMN_LAST_MODIFIED, DateUtils.dateToString(new Date(cdmfrFile.lastModified())));
                file.put(_COLUMN_LENGTH, cdmfrFile.length());
                files.put(cdmfrFile.getName(), file);
            }
            else if (i >= offset + limit)
            {
                break;
            }
            i++;
        }
        
        return files;
    }

    @Override
    public int getTotalCount(Map<String, Object> searchParameters, Logger logger)
    {
        return internalSearch(searchParameters, logger).length;
    }
    
    @SuppressWarnings("unchecked")
    private File[] _sortFiles(File[] files, List<Object> sortList)
    {
        if (sortList != null)
        {
            for (Object sortValueObj : Lists.reverse(sortList))
            {
                Map<String, Object> sortValue = (Map<String, Object>) sortValueObj;
                Comparator<File> comparator = _NAME_TO_COMPARATOR.get(sortValue.get("property"));
                Object direction = sortValue.get("direction");
                if (direction != null && direction.toString().equalsIgnoreCase("DESC"))
                {
                    comparator = Collections.reverseOrder(comparator);
                }
                Arrays.sort(files, comparator);
            }
        }
        
        return files;
    }
    
    /**
     * Search values and return the result without any treatment.
     * @param searchParameters Search parameters to restrict the search
     * @param logger The logger
     * @return {@link File} tab listing the available CDM-fr files corresponding to the filter.
     */
    protected File[] internalSearch(Map<String, Object> searchParameters, Logger logger)
    {
        String filename = null;
        Date lastModifiedAfter = null;
        Date lastModifiedBefore = null;
        if (searchParameters != null && !searchParameters.isEmpty())
        {
            filename = MapUtils.getString(searchParameters, _CRITERIA_FILENAME);
            lastModifiedAfter = DateUtils.parse(MapUtils.getString(searchParameters, _CRITERIA_LAST_MODIFIED_AFTER));
            lastModifiedBefore = DateUtils.parse(MapUtils.getString(searchParameters, _CRITERIA_LAST_MODIFIED_BEFORE));
        }
        FileFilter filter = new CDMFrFileFilter(filename, lastModifiedAfter, lastModifiedBefore);
        
        return _cdmfrFolder.listFiles(filter);
    }

    @Override
    protected void configureDataSource(Configuration configuration) throws ConfigurationException
    {
        configureSpecificParameters();
        
        _odfLang = Config.getInstance().getValue("odf.programs.lang");
    }
    
    /**
     * Configure the specific parameters of this implementation of CDM-fr import.
     */
    protected void configureSpecificParameters()
    {
        String cdmfrFolderPath = getParameterValues().get(__PARAM_FOLDER).toString();

        _cdmfrFolder = new File(cdmfrFolderPath);
        if (!_cdmfrFolder.isAbsolute())
        {
            // No : consider it relative to context path
            _cdmfrFolder = new File(_cocoonContext.getRealPath("/" + cdmfrFolderPath));
        }
        
        if (!_cdmfrFolder.isDirectory())
        {
            throw new RuntimeException("The path '" + cdmfrFolderPath + "' defined in the SCC '" + getLabel().getLabel() + "' (" + getId() + ") is not a directory.");
        }
        
        if (!_cdmfrFolder.canRead())
        {
            throw new RuntimeException("The folder '" + cdmfrFolderPath + "' defined in the SCC '" + getLabel().getLabel() + "' (" + getId() + ") is not readable.");
        }
    }
    
    @Override
    protected void configureSearchModel()
    {
        _searchModelConfiguration.displayMaskImported(false);
        _searchModelConfiguration.addCriterion(_CRITERIA_FILENAME, new I18nizableText("plugin.odf-sync", "PLUGINS_ODF_SYNC_IMPORT_CDMFR_CRITERIA_FILENAME"));
        _searchModelConfiguration.addCriterion(_CRITERIA_LAST_MODIFIED_AFTER, new I18nizableText("plugin.odf-sync", "PLUGINS_ODF_SYNC_IMPORT_CDMFR_CRITERIA_LASTMODIFIED_AFTER"), "DATETIME", "edition.date");
        _searchModelConfiguration.addCriterion(_CRITERIA_LAST_MODIFIED_BEFORE, new I18nizableText("plugin.odf-sync", "PLUGINS_ODF_SYNC_IMPORT_CDMFR_CRITERIA_LASTMODIFIED_BEFORE"), "DATETIME", "edition.date");
        _searchModelConfiguration.addColumn(_COLUMN_FILENAME, new I18nizableText("plugin.odf-sync", "PLUGINS_ODF_SYNC_IMPORT_CDMFR_COLUMN_FILENAME"), 350);
        _searchModelConfiguration.addColumn(_COLUMN_LAST_MODIFIED, new I18nizableText("plugin.odf-sync", "PLUGINS_ODF_SYNC_IMPORT_CDMFR_COLUMN_DATE"), 200, true, "DATETIME");
        _searchModelConfiguration.addColumn(_COLUMN_LENGTH, new I18nizableText("plugin.odf-sync", "PLUGINS_ODF_SYNC_IMPORT_CDMFR_COLUMN_SIZE"), 120, true, "DOUBLE", "Ext.util.Format.fileSize");
    }

    @Override
    public ModifiableContent getContent(String lang, String idValue, boolean forceStrictCheck)
    {
        return null;
    }
    
    @Override
    public boolean handleRightAssignmentContext()
    {
        return false;
    }

    @Override
    protected List<ModifiableContent> _getContentsToRemove(AmetysObjectIterable<ModifiableContent> contents)
    {
        return contents.stream()
                .filter(content -> !_updatedContents.contains(content.getId()))
                .collect(Collectors.toList());
    }
}
