/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.cdmfr.extractor;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Optional;

import org.apache.commons.lang3.StringUtils;
import org.apache.xpath.XPathAPI;
import org.w3c.dom.Element;

import org.ametys.cms.contenttype.ContentAttributeDefinition;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.repository.Content;
import org.ametys.core.util.dom.DOMUtils;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.orgunit.OrgUnitFactory;
import org.ametys.odf.person.PersonFactory;
import org.ametys.plugins.odfsync.cdmfr.ImportCDMFrContext;
import org.ametys.plugins.odfsync.cdmfr.components.ImportCDMFrComponent;
import org.ametys.plugins.odfsync.utils.ContentWorkflowDescription;
import org.ametys.plugins.repository.data.extractor.xml.ModelAwareXMLValuesExtractor;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.ModelItemContainer;
import org.ametys.runtime.model.type.ElementType;

/**
 * This class provides methods to extract values from a CMD-fr import document
 */
public class ImportCDMFrValuesExtractor extends ModelAwareXMLValuesExtractor
{
    /** Tag to identify a courseList */
    protected static final String _TAG_COURSELIST = "coursesReferences";
    
    /** Tag used for each value of multiples data */
    protected static final String _MULTIPLE_DATA_ITEM_TAG = "item";
    
    /** The values extractor factory */
    protected ImportCDMFrValuesExtractorFactory _factory;
    
    /** The import CDM-fr component */
    protected ImportCDMFrComponent _component;
    
    /** The import context */
    protected ImportCDMFrContext _context;
    
    /**
     * Creates an import CDM-fr values extractor
     * @param element the DOM element containing the XML values
     * @param factory the values extractor factory
     * @param component the import CDM-fr component
     * @param context the import context
     * @param models the model of the extracted values
     */
    public ImportCDMFrValuesExtractor(Element element, ImportCDMFrValuesExtractorFactory factory, ImportCDMFrComponent component, ImportCDMFrContext context, Model... models)
    {
        this(element, factory, component, context, Arrays.asList(models));
    }
    
    /**
     * Creates an import CDM-fr values extractor
     * @param element the DOM element containing the XML values
     * @param factory the values extractor factory
     * @param component the import CDM-fr component
     * @param context the imported content's context
     * @param modelItemContainers the model of the extracted values
     */
    public ImportCDMFrValuesExtractor(Element element, ImportCDMFrValuesExtractorFactory factory, ImportCDMFrComponent component, ImportCDMFrContext context, Collection<? extends ModelItemContainer> modelItemContainers)
    {
        super(element, modelItemContainers);
        
        _factory = factory;
        _component = component;
        _context = context;
    }
    
    @Override
    protected <T> Object _extractElementValue(Element parent, ElementDefinition<T> definition, Optional<Object> additionalData) throws Exception
    {
        if (definition instanceof ContentAttributeDefinition)
        {
            String contentTypeId = ((ContentAttributeDefinition) definition).getContentTypeId();
            if (PersonFactory.PERSON_CONTENT_TYPE.equals(contentTypeId))
            {
                return _extractContentValues(parent, definition, "person", ContentWorkflowDescription.PERSON_WF_DESCRIPTION);
            }
            else if (OrgUnitFactory.ORGUNIT_CONTENT_TYPE.equals(contentTypeId))
            {
                return _extractContentValues(parent, definition, "orgunit", ContentWorkflowDescription.ORGUNIT_WF_DESCRIPTION);
            }
            else
            {
                ContentType contentType = _factory.getContentTypeExtensionPoint().getExtension(contentTypeId);
                if (contentType.isReferenceTable() && _factory.getODFReferenceTableHelper().isTableReference(contentTypeId))
                {
                    return _extractTableRefValues(parent, definition, contentTypeId);
                }
            }
        }

        if (definition.isMultiple())
        {
            ElementType<T> type = definition.getType();
            String dataName = definition.getName();
            
            Element child = DOMUtils.getChildElementByTagName(parent, dataName);
            return type.valueFromXML(child, _MULTIPLE_DATA_ITEM_TAG, additionalData);
        }
        else
        {
            return super._extractElementValue(parent, definition, additionalData);
        }
    }
    
    /**
     * Extract the content values for the given attribute
     * @param parent the DOM element of the definition's parent
     * @param definition the attribute's definition
     * @param contentTagName the tag name of the contents to extract
     * @param workflowDescription the {@link ContentWorkflowDescription} of the extracted contents
     * @return the extracted contents
     * @throws Exception if an error occurs
     */
    protected Object _extractContentValues(Element parent, ElementDefinition definition, String contentTagName, ContentWorkflowDescription workflowDescription) throws Exception
    {
        Element child = DOMUtils.getChildElementByTagName(parent, definition.getName());
        
        if (definition.isMultiple())
        {
            List<Content> contents = new ArrayList<>();
            List<Element> items = DOMUtils.getChildElementsByTagName(child, _MULTIPLE_DATA_ITEM_TAG);
            for (Element item : items)
            {
                _extractContentValue(item, contentTagName, workflowDescription)
                    .ifPresent(contents::add);
            }
            
            return contents.toArray(new Content[contents.size()]);
        }
        else
        {
            return _extractContentValue(child, contentTagName, workflowDescription).orElse(null);
        }
    }
    
    /**
     * Extract the content value for the given attribute
     * @param item the DOM element of the content item
     * @param contentTagName the tag name of the contents to extract
     * @param workflowDescription the {@link ContentWorkflowDescription} of the extracted contents
     * @return the extracted content
     * @throws Exception if an error occurs
     */
    protected Optional<Content> _extractContentValue(Element item, String contentTagName, ContentWorkflowDescription workflowDescription) throws Exception
    {
        String syncCode = item.getTextContent().trim();
        Element contentElement = (Element) XPathAPI.selectSingleNode(_context.getDocument().getFirstChild(), contentTagName + "[@CDMid = '" + syncCode + "']");
        if (contentElement != null)
        {
            Element titleElement = DOMUtils.getChildElementByTagName(contentElement, "title");
            String title = titleElement != null ? titleElement.getTextContent() : syncCode;
            Content content = _component.importOrSynchronizeContent(contentElement, workflowDescription, title, syncCode, _context);
            return Optional.ofNullable(content);
        }
        else
        {
            _context.getLogger().warn("There is no {} tag corresponding to the CDM ID '{}'.", contentTagName, syncCode);
            return Optional.empty();
        }
    }
    
    /**
     * Extract the table reference values for the given attribute
     * @param parent the DOM element of the definition's parent
     * @param definition the attribute's definition
     * @param contentTypeId the id of the content type
     * @return the extracted contents
     * @throws Exception if an error occurs
     */
    protected Object _extractTableRefValues(Element parent, ElementDefinition definition, String contentTypeId) throws Exception
    {
        Element child = DOMUtils.getChildElementByTagName(parent, definition.getName());
        
        if (definition.isMultiple())
        {
            List<Content> contents = new ArrayList<>();
            List<Element> items = DOMUtils.getChildElementsByTagName(child, _MULTIPLE_DATA_ITEM_TAG);
            for (Element item : items)
            {
                _extractTableRefValue(item, contentTypeId)
                    .ifPresent(contents::add);
            }
            
            return contents.toArray(new Content[contents.size()]);
        }
        else
        {
            return _extractTableRefValue(child, contentTypeId).orElse(null);
        }
    }
    
    /**
     * Extract the content value for the given attribute
     * @param item the DOM element of the content item
     * @param contentTypeId the id of the content type
     * @return the extracted content
     * @throws Exception if an error occurs
     */
    protected Optional<Content> _extractTableRefValue(Element item, String contentTypeId) throws Exception
    {
        String syncCode = item.getTextContent().trim();
        if (StringUtils.isNotEmpty(syncCode))
        {
            Optional<Content> content = Optional.ofNullable(_factory.getODFReferenceTableHelper().getItemFromCDM(contentTypeId, syncCode))
                    .map(OdfReferenceTableEntry::getContent);
            if (content.isEmpty())
            {
                _context.getLogger().warn("There is no entry corresponding to the CDM-fr or Ametys code '{}' in the reference table '{}'.", syncCode, contentTypeId);
            }
            return content;
        }

        return Optional.empty();
    }

    /**
     * Retrieve the element's attribute with the given name, or the default value if there is no such attribute
     * @param element the element
     * @param attributeName the name of the attribute to retrieve
     * @param defaultValue the default value
     * @return the element's attribute
     */
    protected String _getAttributeOrDefault(Element element, String attributeName, String defaultValue)
    {
        String attributeValue = element.getAttribute(attributeName);
        return StringUtils.isNotEmpty(attributeValue) ? attributeValue : defaultValue;
    }
}
