/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.odfsync.pegase.ws.structure;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.regex.Pattern;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.ContentValue;
import org.ametys.cms.data.RichTextHelper;
import org.ametys.cms.repository.Content;
import org.ametys.odf.ODFHelper;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.course.CourseFactory;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.courselist.CourseList.ChoiceType;
import org.ametys.odf.enumeration.OdfReferenceTableEntry;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.Container;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.ProgramPart;
import org.ametys.odf.program.SubProgram;
import org.ametys.plugins.odfsync.export.AbstractExportStructure;
import org.ametys.plugins.odfsync.export.ExportReport;
import org.ametys.plugins.odfsync.export.ExportReport.ExportStatus;
import org.ametys.plugins.odfsync.export.ExportReport.ProblemTypes;
import org.ametys.plugins.odfsync.pegase.ws.PegaseApiManager;
import org.ametys.plugins.odfsync.pegase.ws.PegaseExportException;
import org.ametys.plugins.odfsync.pegase.ws.PegaseHelper;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.data.UnknownDataException;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;

import fr.pcscol.pegase.odf.ApiException;
import fr.pcscol.pegase.odf.api.ObjetsMaquetteApi;
import fr.pcscol.pegase.odf.model.Contexte;
import fr.pcscol.pegase.odf.model.CreerFormationRequest;
import fr.pcscol.pegase.odf.model.CreerGroupementRequest;
import fr.pcscol.pegase.odf.model.CreerObjetFormationRequest;
import fr.pcscol.pegase.odf.model.CreerObjetMaquetteRequest;
import fr.pcscol.pegase.odf.model.DescripteursAglae;
import fr.pcscol.pegase.odf.model.DescripteursEnqueteRequest;
import fr.pcscol.pegase.odf.model.DescripteursFormationRequest;
import fr.pcscol.pegase.odf.model.DescripteursFormationSyllabus;
import fr.pcscol.pegase.odf.model.DescripteursGroupementRequest;
import fr.pcscol.pegase.odf.model.DescripteursObjetFormationRequest;
import fr.pcscol.pegase.odf.model.DescripteursObjetFormationSyllabus;
import fr.pcscol.pegase.odf.model.DescripteursObjetMaquetteRequest;
import fr.pcscol.pegase.odf.model.DescripteursSiseRequest;
import fr.pcscol.pegase.odf.model.DescripteursSyllabus;
import fr.pcscol.pegase.odf.model.Enfant;
import fr.pcscol.pegase.odf.model.EnfantsStructure;
import fr.pcscol.pegase.odf.model.MaquetteStructure;
import fr.pcscol.pegase.odf.model.ObjetMaquetteDetail;
import fr.pcscol.pegase.odf.model.ObjetMaquetteStructure;
import fr.pcscol.pegase.odf.model.ObjetMaquetteSummary;
import fr.pcscol.pegase.odf.model.Pageable;
import fr.pcscol.pegase.odf.model.PagedObjetMaquetteSummaries;
import fr.pcscol.pegase.odf.model.PlageDeChoix;
import fr.pcscol.pegase.odf.model.TypeObjetMaquette;

/**
 * The structure to export the program in Pegase
 */
public class PegaseProgramStructure extends AbstractExportStructure implements Component, Initializable
{
    /** Role */
    public static final String ROLE = PegaseProgramStructure.class.getName();
    
    /* Constants */
    /** The attribute name for the Pégase code */
    private static final String __CODE_PEGASE_ATTRIBUTE_NAME = "pegaseCode";
    /** The pattern of a Pégase code */
    private static final Pattern __CODE_PATTERN = Pattern.compile("^[A-Z0-9\\-]{3,30}$");
    /** The attribute name for the Pégase Sync code (UUID) */
    private static final String __PEGASE_SYNC_CODE = "pegaseSyncCode";
    
    /** The ects attribute path */
    private static final String __ECTS_DATA_PATH = "ects";
    
    /** The type ID for a subProgram in pegase */
    private static final String __PARCOURS_TYPE_ID = "PARCOURS-TYPE";
    /** The type ID for a year in pegase */
    private static final String __ANNEE_TYPE_ID = "ANNEE";
    /** The type ID for a semester in pegase */
    private static final String __SEMESTRE_TYPE_ID = "SEMESTRE";
    
    /** The mandatory attributes by content type */
    private static final Map<String, Set<String>> __MANDATORY_ATTRIBUTES_BY_CONTENT_TYPE = new HashMap<>();
    static
    {
        __MANDATORY_ATTRIBUTES_BY_CONTENT_TYPE.put(
                ProgramFactory.PROGRAM_CONTENT_TYPE,
                Set.of(
                        AbstractProgram.EDUCATION_KIND
                )
        );
        __MANDATORY_ATTRIBUTES_BY_CONTENT_TYPE.put(
                CourseFactory.COURSE_CONTENT_TYPE,
                Set.of(Course.COURSE_TYPE)
        );
    }

    /* Components */
    private PegaseApiManager _pegaseApiManager;
    private ODFHelper _odfHelper;
    private RichTextHelper _richTextHelper;
    private PegaseHelper _pegaseHelper;

    /* Pégase configuration */
    private boolean _isActive;
    private String _structureCode;
    private boolean _trustAmetys;

    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        
        _odfHelper = (ODFHelper) manager.lookup(ODFHelper.ROLE);
        _pegaseApiManager = (PegaseApiManager) manager.lookup(PegaseApiManager.ROLE);
        _richTextHelper = (RichTextHelper) manager.lookup(RichTextHelper.ROLE);
        _pegaseHelper = (PegaseHelper) manager.lookup(PegaseHelper.ROLE);
    }
    
    public void initialize() throws Exception
    {
        _isActive = Config.getInstance().getValue("pegase.activate", true, false);
        if (_isActive)
        {
            _structureCode = Config.getInstance().getValue("pegase.structure.code");
            _trustAmetys = Config.getInstance().getValue("pegase.trust", true, false);
        }
    }
    
    private String _getEtag(ObjetsMaquetteApi objetsMaquetteApi)
    {
        List<String> etags = objetsMaquetteApi.getApiClient().getResponseHeaders().get("ETag");
        
        // Defaults to version 1
        String etag = "1";
        if (etags != null && !etags.isEmpty())
        {
            etag = etags.get(0);
        }
        
        return etag;
    }
    
    private Pageable _getPageable(int page, int taille, List<String> tri)
    {
        Pageable pageable = new Pageable();
        pageable.setPage(page);
        pageable.setTaille(taille);
        pageable.setTri(tri);
        
        return pageable;
    }
    
    /**
     * Checks if the program has all the required fields
     * @param program the program to check
     * @param report the Pegase export report
     */
    @Override
    public void checkProgram(Program program, ExportReport report)
    {
        if (!_isActive)
        {
            throw new UnsupportedOperationException("Pégase is not active in the configuration, you cannot check the program for the export.");
        }

        try
        {
            _checkProgramItem(program, report);
        }
        catch (PegaseExportException e)
        {
            report.setStatus(ExportStatus.ERROR);
            getLogger().error("An error occured while checking the program", e);
        }
    }

    /**
     * Checks if the program item has all the required fields
     * @param programItem the program item to check
     * @param report the Pegase export report
     * @throws PegaseExportException If an error occurs while retrieving the Pégase espace
     */
    private void _checkProgramItem(ProgramItem programItem, ExportReport report) throws PegaseExportException
    {
        // Check program item attributes
        _checkAttributes((Content) programItem, report);
        
        // Check children
        for (ProgramItem child : _odfHelper.getChildProgramItems(programItem))
        {
            _checkProgramItem(child, report);
        }
    }
    
    private void _checkAttributes(Content content, ExportReport report) throws PegaseExportException
    {
        // If it is a Program, check the codes and the coherence with Pégase
        if (content instanceof Program)
        {
            _checkPegaseCodesForProgram(content, report);
        }
        // If it is a Program, only check the codes
        else if (content instanceof ProgramItem)
        {
            _checkPegaseCodes(content, report);
        }

        // Check the other mandatory attributes by type
        String contentType = content.getTypes()[0];
        Set<String> mandatoryAttributes = __MANDATORY_ATTRIBUTES_BY_CONTENT_TYPE.getOrDefault(contentType, Set.of());
        for (String attribute : mandatoryAttributes)
        {
            if (!content.hasValue(attribute))
            {
                _addMandatoryDataPathAndReport(content, attribute, report);
            }
        }
    }
    
    private void _checkPegaseCodesForProgram(Content content, ExportReport report) throws PegaseExportException
    {
        UUID syncCode = null;
        
        // Check if the Pegase sync code is present and valid
        boolean hasValidSyncCode = false;
        if (content.hasValue(__PEGASE_SYNC_CODE))
        {
            syncCode = _getUuidIfValid(content.getValue(__PEGASE_SYNC_CODE));
            
            hasValidSyncCode = syncCode != null;
            if (!hasValidSyncCode)
            {
                report.addInvalidDataPath(content, new I18nizableText("plugin.odf-sync", "PLUGINS_ODF_SYNC_EXPORT_PEGASE_INVALID_SYNC_CODE"));
            }
        }
        
        // Check if the Pegase code is present
        boolean hasValidPegaseCode = false;
        if (content.hasValue(__CODE_PEGASE_ATTRIBUTE_NAME))
        {
            hasValidPegaseCode = _isValidPegaseCode(content.getValue(__CODE_PEGASE_ATTRIBUTE_NAME));
            if (!hasValidPegaseCode)
            {
                report.addInvalidDataPath(content, new I18nizableText("plugin.odf-sync", "PLUGINS_ODF_SYNC_EXPORT_PEGASE_INVALID_CODE"));
            }
        }
        
        // If both codes are present and valid, check their coherence with each other and with Pégase
        if (hasValidSyncCode && hasValidPegaseCode)
        {
            _checkProgramCoherenceWithTwoCodes(content.getValue(__CODE_PEGASE_ATTRIBUTE_NAME), syncCode, report);
        }
        // If only the sync codes is present and valid, check its coherence with Pégase
        else if (hasValidSyncCode && !hasValidPegaseCode)
        {
            _getProgramAndcheckCoherenceForUUID(syncCode, report);
        }
        // If only the Pégase codes is present and valid, check its coherence with Pégase
        else if (!hasValidSyncCode && hasValidPegaseCode)
        {
            _checkProgramCoherenceByCode(content.getValue(__CODE_PEGASE_ATTRIBUTE_NAME), report);
        }
        // If neither of the codes is present, report the problem
        else
        {
            _addMandatoryPegaseCodeAndReport(content, report);
        }
    }
    
    /**
     * Check if an object or a non editable program for the UUID and if the pegaseCode requested is coherent
     * @param report The report
     * @param pegaseCode The code wanted
     * @param pegaseSyncCode The UUID of the Pégase object requested
     * @throws PegaseExportException If an error occurs while retriving the espaceId
     */
    private void _checkProgramCoherenceWithTwoCodes(String pegaseCode, UUID pegaseSyncCode, ExportReport report) throws PegaseExportException
    {
        ObjetMaquetteSummary programFound = _getProgramAndcheckCoherenceForUUID(pegaseSyncCode, report);
        
        // If a program was found and was coherent with the UUID, check that the pegaseCode requested is coherent with the PegaseCode of the program found
        if (programFound != null
            && !ExportStatus.NON_EDITABLE_PROGRAM_ALREADY_EXISTS.equals(report.getStatus())
            && !pegaseCode.equals(programFound.getCode()))
        {
            report.setStatus(ExportStatus.NON_EDITABLE_PROGRAM_ALREADY_EXISTS);
        }
    }
    
    /**
     * Check if an object or a non editable program already exists for the UUID and retrives it
     * @param report The report
     * @param syncCode The UUID of the Pégase object requested
     * @throws PegaseExportException If an error occurs while retriving the espaceId
     */
    private ObjetMaquetteSummary _getProgramAndcheckCoherenceForUUID(UUID syncCode, ExportReport report) throws PegaseExportException
    {
        try
        {
            ObjetsMaquetteApi objetsMaquetteApi = _pegaseApiManager.getObjetsMaquetteApi();
            
            // Get the objects with the same ID (syncCode) as the one requested
            PagedObjetMaquetteSummaries pagedObjetMaquetteSummaries = objetsMaquetteApi.rechercherObjetMaquette(_structureCode, _getPageable(0, 10, List.of()), null, _pegaseHelper.getEspaceId(_structureCode).toString(), List.of(), null, null, List.of(syncCode), null, null, null, null);
            
            // Check that it can be edited
            if (pagedObjetMaquetteSummaries.getTotalElements() == 1)
            {
                ObjetMaquetteSummary objetMaquetteSummary = pagedObjetMaquetteSummaries.getItems().get(0);

                // If the object is not a Program or it is already validated, then it is not editable
                if (!TypeObjetMaquette.FORMATION.equals(objetMaquetteSummary.getTypeObjetMaquette())
                    || objetMaquetteSummary.getValideInAnyContexte())
                {
                    report.setStatus(ExportStatus.NON_EDITABLE_PROGRAM_ALREADY_EXISTS);
                }
                
                return objetMaquetteSummary;
            }
            // Should not be possible, there can only be one object with the ID
            else
            {
                report.setStatus(ExportStatus.NON_EDITABLE_PROGRAM_ALREADY_EXISTS);
                return null;
            }
        }
        catch (IOException | ApiException e)
        {
            report.setStatus(ExportStatus.ERROR);
            getLogger().warn("Une erreur est survenue lors la recherche de l'élément de code de synchronization Pégase '{}' pour vérifier son existance préalable dans Pégase", syncCode, e);
        }
        
        return null;
    }
    
    /**
     * Check if an object or a non editable program already exists for the PegaseCode wanted
     * @param report The report
     * @param pegaseCode The code wanted
     * @throws PegaseExportException If an error occurred while retrieving the Pégase espace
     */
    private void _checkProgramCoherenceByCode(String pegaseCode, ExportReport report) throws PegaseExportException
    {
        try
        {
            ObjetsMaquetteApi objetsMaquetteApi = _pegaseApiManager.getObjetsMaquetteApi();
            
            // Get the objects with the same code as the one requested
            PagedObjetMaquetteSummaries pagedObjetMaquetteSummaries = objetsMaquetteApi.rechercherObjetMaquette(_structureCode, _getPageable(0, 10, List.of()), pegaseCode, _pegaseHelper.getEspaceId(_structureCode).toString(), List.of(), null, null, List.of(), null, null, null, null);
            
            Long numberOfResults = pagedObjetMaquetteSummaries.getTotalElements();
            // If there is one result, check if it is coherent
            if (numberOfResults == 1)
            {
                ObjetMaquetteSummary objetMaquetteSummary = pagedObjetMaquetteSummaries.getItems().get(0);
                
                // If the object found is not a program or if it is validated, then it is not editable
                if (!TypeObjetMaquette.FORMATION.equals(objetMaquetteSummary.getTypeObjetMaquette())
                    || objetMaquetteSummary.getValideInAnyContexte())
                {
                    report.setStatus(ExportStatus.NON_EDITABLE_PROGRAM_ALREADY_EXISTS);
                }
            }
            // If there are more than one result found, it is not editable
            else if (numberOfResults > 1)
            {
                report.setStatus(ExportStatus.NON_EDITABLE_PROGRAM_ALREADY_EXISTS);
            }
            // If there are no results, the program is exportable to Pégase (since no sync code was requested)
        }
        catch (IOException | ApiException e)
        {
            report.setStatus(ExportStatus.ERROR);
            getLogger().warn("Une erreur est survenue lors la recherche de l'élément de code de synchronization Pégase '{}' pour vérifier son existance préalable dans Pégase", pegaseCode, e);
        }
    }
    
    /**
     * Check if there is at least one of the two Pegase codes
     * @param report The report
     * @param content The content
     */
    private void _checkPegaseCodes(Content content, ExportReport report)
    {
        // If there is no pegase sync code or if it is invalid, check the pegaseCode
        if (!content.hasValue(__PEGASE_SYNC_CODE) || !(_getUuidIfValid(content.getValue(__PEGASE_SYNC_CODE)) != null))
        {
            // If there is no pegase code or it is invalid, report the missing fields
            if (!content.hasValue(__CODE_PEGASE_ATTRIBUTE_NAME) || !_isValidPegaseCode(content.getValue(__CODE_PEGASE_ATTRIBUTE_NAME)))
            {
                _addMandatoryPegaseCodeAndReport(content, report);
            }
        }
    }

    private UUID _getUuidIfValid(String uuidToCheck)
    {
        try
        {
            return UUID.fromString(uuidToCheck);
        }
        catch (Exception e)
        {
            return null;
        }
    }
    
    private boolean _isValidPegaseCode(String attribute)
    {
        return __CODE_PATTERN.matcher(attribute).matches();
    }
    
    private void _addMandatoryDataPathAndReport(Content content, String dataPath, ExportReport report)
    {
        I18nizableText invalidMessage = new I18nizableText(
            "plugin.odf-sync",
            "PLUGINS_ODF_SYNC_EXPORT_PEGASE_MANDATORY_FIELD",
            Map.of("fieldName", content.getDefinition(dataPath).getLabel())
        );
        report.addInvalidDataPath(content, invalidMessage);
    }
    
    private void _addMandatoryPegaseCodeAndReport(Content content, ExportReport report)
    {
        I18nizableText invalidMessage = new I18nizableText(
            "plugin.odf-sync",
            "PLUGINS_ODF_SYNC_EXPORT_PEGASE_MANDATORY_FIELDS",
            Map.of("fieldName", content.getDefinition(__PEGASE_SYNC_CODE).getLabel(),
                   "secondFieldName", content.getDefinition(__CODE_PEGASE_ATTRIBUTE_NAME).getLabel())
        );
        report.addInvalidDataPath(content, invalidMessage);
    }
    
    private Set<ProgramItem> _getAllChildrenProgramItems(ProgramItem programItem)
    {
        return _getAllChildrenProgramItems(programItem, new HashSet<>());
    }

    
    private Set<ProgramItem> _getAllChildrenProgramItems(ProgramItem programItem, Set<ProgramItem> allChildrenProgramItems)
    {
        allChildrenProgramItems.add(programItem);
        
        for (ProgramItem child : _odfHelper.getChildProgramItems(programItem))
        {
            if (!allChildrenProgramItems.contains(child))
            {
                allChildrenProgramItems.addAll(_getAllChildrenProgramItems(child, allChildrenProgramItems));
            }
        }
        
        return allChildrenProgramItems;
    }
    
    private ObjetMaquetteAndEtagIfExists _getPegaseProgramIfAlreadyExists(Program program, ExportReport report) throws PegaseExportException
    {
        return _getPegaseObjetMaquetteDetailIfAlreadyExists(program, TypeObjetMaquette.FORMATION, report);
    }
    
    private ObjetMaquetteAndEtagIfExists _getPegaseObjetFormationIfAlreadyExists(Content content, ExportReport report) throws PegaseExportException
    {
        return _getPegaseObjetMaquetteDetailIfAlreadyExists(content, TypeObjetMaquette.OBJET_FORMATION, report);
    }
    
    private ObjetMaquetteAndEtagIfExists _getPegaseGroupIfAlreadyExists(Content content, ExportReport report) throws PegaseExportException
    {
        return _getPegaseObjetMaquetteDetailIfAlreadyExists(content, TypeObjetMaquette.GROUPEMENT, report);
    }
    
    private ObjetMaquetteAndEtagIfExists _getPegaseObjetMaquetteDetailIfAlreadyExists(Content content, TypeObjetMaquette type, ExportReport report) throws PegaseExportException
    {
        try
        {
            // If there is a sync code, it has been verified to be valid, try to get the Pégase object
            if (content.hasValue(__PEGASE_SYNC_CODE))
            {
                String id = content.getValue(__PEGASE_SYNC_CODE);
                try
                {
                    ObjetsMaquetteApi objetsMaquetteApi = _pegaseApiManager.getObjetsMaquetteApi();
                    ObjetMaquetteDetail objetMaquetteFound = objetsMaquetteApi.lireObjetMaquette(_structureCode, UUID.fromString(id));
                    
                    // If the object found is not in the right type, throw an exception
                    if (!type.equals(objetMaquetteFound.getTypeObjetMaquette()))
                    {
                        throw new PegaseExportException("Erreur lors de l'export de " + content.getTitle() + ". Un élément avec l'identifiant Pégase existe déjà et n'est pas mutualisable ou modifiable");
                    }
                    
                    return new ObjetMaquetteAndEtagIfExists(objetMaquetteFound, _getEtag(objetsMaquetteApi));
                }
                catch (ApiException e)
                {
                    throw new PegaseExportException("Un code de synchronisation ('" + id + "') Pégase a été entré pour l'élément " + content.getTitle() + " mais aucun élément ne correspond à ce code dans Pégase.", e);
                }
            }
            else if (content.hasValue(__CODE_PEGASE_ATTRIBUTE_NAME))
            {
                String pegaseCode = content.getValue(__CODE_PEGASE_ATTRIBUTE_NAME);
                
                return _getObjetMaquetteDetailFromPaged(content, pegaseCode, type, report);
            }
        }
        catch (ApiException | IOException e)
        {
            throw new PegaseExportException("Une erreur est survenue en essayant de récupérer un élément déjà existant pour l'élément '" + content.getTitle() + "'.", e);
        }
        
        return new ObjetMaquetteAndEtagIfExists(null, null);
    }
    
    private ObjetMaquetteAndEtagIfExists _getObjetMaquetteDetailFromPaged(Content content, String pegaseCode, TypeObjetMaquette type, ExportReport report) throws ApiException, IOException, PegaseExportException
    {
        PagedObjetMaquetteSummaries pagedObjetsMaquetteFound = _pegaseApiManager.getObjetsMaquetteApi().rechercherObjetMaquette(_structureCode, _getPageable(0, Integer.MAX_VALUE, List.of()), pegaseCode, _pegaseHelper.getEspaceId(_structureCode).toString(), List.of(), null, null, List.of(), null, false, false, false);
        
        ObjetMaquetteAndEtagIfExists objetMaquetteFound = _lookForObjetMaquetteInPage(content, pagedObjetsMaquetteFound, pegaseCode, type, report);
        
        if (objetMaquetteFound != null)
        {
            return objetMaquetteFound;
        }
        
        for (int page = 0; page < pagedObjetsMaquetteFound.getTotalPages(); page++)
        {
            objetMaquetteFound = _lookForObjetMaquetteInPage(content, page, Integer.MAX_VALUE, pegaseCode, type, report);

            if (objetMaquetteFound != null)
            {
                return objetMaquetteFound;
            }
        }
        
        return new ObjetMaquetteAndEtagIfExists(null, null);
    }
    
    private ObjetMaquetteAndEtagIfExists _lookForObjetMaquetteInPage(Content content, int page, int taille, String pegaseCode, TypeObjetMaquette type, ExportReport report) throws ApiException, IOException, PegaseExportException
    {
        PagedObjetMaquetteSummaries pagedObjetsMaquetteFound = _pegaseApiManager.getObjetsMaquetteApi().rechercherObjetMaquette(_structureCode, _getPageable(page, taille, List.of()), pegaseCode, _pegaseHelper.getEspaceId(_structureCode).toString(), List.of(), null, null, List.of(), null, false, false, false);
        
        return _lookForObjetMaquetteInPage(content, pagedObjetsMaquetteFound, pegaseCode, type, report);
    }
    
    private ObjetMaquetteAndEtagIfExists _lookForObjetMaquetteInPage(Content content, PagedObjetMaquetteSummaries pagedObjetsMaquetteFound, String pegaseCode, TypeObjetMaquette type, ExportReport report) throws ApiException, IOException, UnknownDataException, AmetysRepositoryException, PegaseExportException
    {
        List<ObjetMaquetteSummary> items = pagedObjetsMaquetteFound.getItems();
        if (items != null)
        {
            for (ObjetMaquetteSummary objetMaquette : items)
            {
                if (pegaseCode.equals(objetMaquette.getCode()))
                {
                    if (objetMaquette.getTypeObjetMaquette().equals(type) && !objetMaquette.getValideInAnyContexte())
                    {
                        ObjetsMaquetteApi objetsMaquetteApi = _pegaseApiManager.getObjetsMaquetteApi();
                        return new ObjetMaquetteAndEtagIfExists(objetsMaquetteApi.lireObjetMaquette(_structureCode, objetMaquette.getId()), _getEtag(objetsMaquetteApi));
                    }
                    else
                    {
                        report.updateExportReport(ExportStatus.ERROR, ProblemTypes.ELEMENT_ALREADY_EXIST);
                        throw new PegaseExportException("Erreur lors de l'export de " + content.getTitle() + ". Un élément avec l'identifiant ou le code Pégase existe déjà et n'est pas mutualisable ou modifiable");
                    }
                }
            }
        }
        
        return new ObjetMaquetteAndEtagIfExists(null, null);
    }
    
    private void _initCreerObjetMaquetteRequest(CreerObjetMaquetteRequest creerObjetMaquetteRequest, TypeObjetMaquette typeObjetMaquette, Boolean mutualise, Content content) throws PegaseExportException
    {
        creerObjetMaquetteRequest.setCode(content.getValue(__CODE_PEGASE_ATTRIBUTE_NAME));
        creerObjetMaquetteRequest.setTypeObjetMaquette(typeObjetMaquette);
        creerObjetMaquetteRequest.setEspaceId(_pegaseHelper.getEspaceId(_structureCode));
        creerObjetMaquetteRequest.setMutualise(mutualise);
    }
    
    private Enfant _createPegaseChild(UUID pegaseId, boolean mandatory)
    {
        Enfant pegaseChild = new Enfant();
        
        pegaseChild.setId(pegaseId.toString());
        pegaseChild.setObligatoire(mandatory);
        
        return pegaseChild;
    }
    
    /**
     * Create a program in Pegase
     * @param program the program to export
     * @param report the Pegase export report
     */
    @Override
    public void createProgram(Program program, ExportReport report)
    {
        if (!_isActive)
        {
            throw new UnsupportedOperationException("Pégase is not active in the configuration, you cannot import or synchronize a program in Pégase.");
        }

        int nbTotal = _getAllChildrenProgramItems(program).size();

        report.setNbTotal(nbTotal);

        try
        {
            // Create the Program in Pegase
            ObjetMaquetteDetail pegaseProgram = _createOrUpdateProgram(program, report);
            report.addElementExported(program);

            // Get the children of the Ametys program
            List<ProgramPart> children = program.getProgramPartChildren();
            
            // The map of the children to link to the program and their own children
            Map<String, PegaseChildWithChildren> childrenToLink = new HashMap<>();

            for (ProgramPart child : children)
            {
                // Create the pegase instance of the child
                PegaseChildWithChildren enfant = _createChild((Content) child, false, report);

                // Add the Enfant (created from the Pegase Id of the child) to the list of "Enfant" to link, if it is not already attached
                if (enfant != null)
                {
                    childrenToLink.put(enfant.pegaseChild().getId(), enfant);
                }
            }

            _detachAndAttachAllChildren(program, pegaseProgram.getId(), childrenToLink, report);
        }
        catch (Exception e)
        {
            report.updateStatus(ExportStatus.ERROR);
            getLogger().error("Une erreur est survenue lors de l'export de la formation '{}' ({}) dans Pégase", program.getTitle(), program.getId(), e);
        }
    }
    
    private ObjetMaquetteDetail _createOrUpdateProgram(Program program, ExportReport report) throws PegaseExportException
    {
        try
        {
            ObjetMaquetteAndEtagIfExists objetMaquetteAndEtagIfExists = _getPegaseProgramIfAlreadyExists(program, report);
            
            ObjetMaquetteDetail objetMaquetteDetailFound = objetMaquetteAndEtagIfExists.objetMaquetteDetail();
            
            // Program does not already exists, need to create it
            if (objetMaquetteDetailFound == null)
            {
                return _createProgram(program);
            }
            // Program already exists, need to update it
            else
            {
                // Get the previous etag (number of version) required to update the program
                String etag = objetMaquetteAndEtagIfExists.etag();
                return _updateProgram(program, objetMaquetteDetailFound, etag);
            }
        }
        catch (Exception e)
        {
            report.updateExportReport(ExportStatus.ERROR, ProblemTypes.API_ERROR);

            throw new PegaseExportException("Une erreur est survenue lors de la création ou modification de la formation dans Pégase", e);
        }
    }
    
    private ObjetMaquetteDetail _createProgram(Program program) throws ApiException, IOException, PegaseExportException
    {
        ObjetsMaquetteApi objetsMaquetteApi = _pegaseApiManager.getObjetsMaquetteApi();
        
        CreerFormationRequest creerFormationRequest = new CreerFormationRequest();
        _initCreerObjetMaquetteRequest(creerFormationRequest, TypeObjetMaquette.FORMATION, false, program);
        
        // Update descripteurs formation
        creerFormationRequest.setDescripteursObjetMaquette(_getDescripteursFormationRequest(program));

        ObjetMaquetteDetail programCreated = objetsMaquetteApi.creerObjetMaquette(_structureCode, creerFormationRequest);
        
        String etag = _getEtag(objetsMaquetteApi);
        
        return _updateDescripteurProgram(program, programCreated.getId(), etag, objetsMaquetteApi);
    }
    
    private ObjetMaquetteDetail _updateProgram(Program program, ObjetMaquetteDetail objetMaquetteDetailFound, String etag) throws IOException, ApiException, PegaseExportException
    {
        ObjetsMaquetteApi objetsMaquetteApi = _pegaseApiManager.getObjetsMaquetteApi();

        // Update descripteurs formation
        DescripteursFormationRequest descripteursFormationRequest = _getDescripteursFormationRequest(program);
        
        // If no updates were necessary, return the former objetMaquetteDetail
        UUID pegaseProgramId = objetMaquetteDetailFound.getId();
        objetsMaquetteApi.modifierDescripteursObjetMaquette(_structureCode, pegaseProgramId, etag, descripteursFormationRequest);
        
        return _updateDescripteurProgram(program, pegaseProgramId, _getEtag(objetsMaquetteApi), objetsMaquetteApi);
    }
    
    private ObjetMaquetteDetail _updateDescripteurProgram(AbstractProgram program, UUID pegaseId, String etag, ObjetsMaquetteApi objetsMaquetteApi) throws ApiException, IOException, PegaseExportException
    {
        _updateDescripteursEnqueteFormation(program, pegaseId, etag, objetsMaquetteApi);
        
        return _updateDescripteursSyllabus(program, pegaseId, TypeObjetMaquette.FORMATION, new DescripteursFormationSyllabus());
    }
    
    private DescripteursFormationRequest _getDescripteursFormationRequest(Program program)
    {
        DescripteursFormationRequest descripteursFormationRequest = new DescripteursFormationRequest();
        
        String programTitle = program.getTitle();
        
        descripteursFormationRequest.setLibelle(StringUtils.truncate(programTitle, 50));
        descripteursFormationRequest.setLibelleLong(StringUtils.truncate(programTitle, 150));
        descripteursFormationRequest.setEcts(_getEctsFromContent(program));
        
        // Education king -> Type
        descripteursFormationRequest.setType(_pegaseHelper.getPegaseCodeForField(program, AbstractProgram.EDUCATION_KIND));
        
        return descripteursFormationRequest;
    }
    
    private ObjetMaquetteDetail _updateDescripteursSyllabus(Content content, UUID pegaseProgramId, TypeObjetMaquette type, DescripteursSyllabus descripteursSyllabus) throws ApiException, IOException, PegaseExportException
    {
        ObjetsMaquetteApi objetsMaquetteApi = _pegaseApiManager.getObjetsMaquetteApi();
        
        String description = null;
        if (content.hasValue("description"))
        {
            description = _richTextHelper.richTextToString(content.getValue("description"));
        }
        else if (content.hasValue("presentation"))
        {
            description = _richTextHelper.richTextToString(content.getValue("presentation"));
        }
        
        if (description != null)
        {
            descripteursSyllabus.setDescription(StringUtils.truncate(description, 2000)); // la description doit faire moins de 2000 caractères
        }
        
        if (content.hasValue("objectives"))
        {
            descripteursSyllabus.setObjectif(StringUtils.truncate(_richTextHelper.richTextToString(content.getValue("objectives")), 2000)); // l'objectif doit faire moins de 2000 caractères
        }
        
        if (content.hasValue("neededPrerequisite"))
        {
            descripteursSyllabus.setPrerequisPedagogique(StringUtils.truncate(_richTextHelper.richTextToString(content.getValue("neededPrerequisite")), 2000)); // les prérequis doit faire moins de 2000 caractères
        }
        
        // Retrieve the created object to get its current Etag
        ObjetMaquetteAndEtagIfExists objetMaquetteExistingAndEtag = _getPegaseObjetMaquetteDetailIfAlreadyExists(content, type, new ExportReport(content));
        
        return objetsMaquetteApi.modifierDescripteursSyllabusObjetMaquette(_structureCode, pegaseProgramId, objetMaquetteExistingAndEtag.etag(), descripteursSyllabus);
    }
    
    private ObjetMaquetteDetail _updateDescripteursEnqueteFormation(AbstractProgram program, UUID pegaseProgramId, String etag, ObjetsMaquetteApi objetsMaquetteApi) throws ApiException
    {
        DescripteursSiseRequest descripteursSiseRequest = new DescripteursSiseRequest();
        
        // Degree -> typeDiplome
        descripteursSiseRequest.setTypeDiplome(_pegaseHelper.getPegaseCodeForField(program, AbstractProgram.DEGREE));
        // EducationLevel -> niveauFormation
        descripteursSiseRequest.setNiveauDiplomeSise(_pegaseHelper.getPegaseCodeForField(program, AbstractProgram.LEVEL));
        // RncpLevel -> niveauDiplome
        descripteursSiseRequest.setNiveauDiplome(_pegaseHelper.getPegaseCodeForField(program, AbstractProgram.RNCP_LEVEL));
        // Domain
        descripteursSiseRequest.setDomaineFormation(_pegaseHelper.getPegaseCodeForFirstValue(program, AbstractProgram.DOMAIN));
        // Mention
        descripteursSiseRequest.setMention(_pegaseHelper.getPegaseCodeForField(program, AbstractProgram.MENTION));
        // ProgramFields -> champFormation
        descripteursSiseRequest.setChampFormation(_pegaseHelper.getPegaseCodeForField(program, AbstractProgram.PROGRAM_FIELD));
        
        // Update descripteurs enquete
        DescripteursEnqueteRequest descripteursEnqueteRequest = new DescripteursEnqueteRequest();
        descripteursEnqueteRequest.setDescripteursSise(descripteursSiseRequest);
        
        DescripteursAglae descripteursAglae = new DescripteursAglae();
        descripteursAglae.setHabilitePourBoursesAglae(false);
        descripteursEnqueteRequest.setDescripteursAglae(descripteursAglae);
        
        return objetsMaquetteApi.modifierDescripteursEnqueteObjetMaquette(_structureCode, pegaseProgramId, etag, descripteursEnqueteRequest);
    }
    
    private BigDecimal _getEctsFromContent(Content content)
    {
        if (content.hasValue(__ECTS_DATA_PATH))
        {
            Object value = content.getValue(__ECTS_DATA_PATH);
            if (value instanceof Double ects)
            {
                return BigDecimal.valueOf(ects);
            }
            else if (value instanceof ContentValue contentValue)
            {
                try
                {
                    BigDecimal ects = contentValue.getContentIfExists()
                            .map(c -> c.<String>getValue(OdfReferenceTableEntry.CODE))
                            .map(Double::parseDouble)
                            .map(BigDecimal::valueOf)
                            .orElse(null);
                    return ects;
                }
                catch (NumberFormatException e)
                {
                    getLogger().debug("The ects value '{}' is not a double, therefore, it is not compatible with Pégase ects and will not be exported", contentValue);
                }
                
            }
        }
        
        return null;
    }
    
    private  PegaseChildWithChildren _createChild(Content content, boolean parentMutualise, ExportReport report)
    {
        boolean success = true;
        try
        {
            // Conteneur : semestre, année -> OF
            if (content instanceof Container container)
            {
                return _createOFFromContainer(container, parentMutualise, report);
            }
            // Parcours -> OF
            if (content instanceof SubProgram subProgram)
            {
                return _createObjetFormationAndChildren(subProgram, __PARCOURS_TYPE_ID, parentMutualise, report);
            }
            // ELP -> OF
            if (content instanceof Course course)
            {
                return _createObjetFormationAndChildren(course, _pegaseHelper.getPegaseCodeForFirstValue(course, Course.COURSE_TYPE), parentMutualise, report);
            }
            // liste d'ELP -> Groupement
            if (content instanceof CourseList)
            {
                return _createGroupFromCourseList((CourseList) content, parentMutualise, report);
            }
    
            return null;
        }
        catch (Exception ex)
        {
            success = false;
            getLogger().error("Erreur lors de l'export de l'élément '{}'", content.getTitle(), ex);
            return null;
        }
        finally
        {
            if (success)
            {
                report.addElementExported(content);
            }
        }
    }
    
    /**
     * Creates a Pégase objet formation from a container
     * @param container The container
     * @param parentMutualise If the element's parent is mutualised
     * @param report The report
     * @return An {@link Enfant} object, Pégase child
     * @throws PegaseExportException If an error occurs
     * @throws IOException If an API error occurs
     * @throws ApiException If an error occurs
     */
    private PegaseChildWithChildren _createOFFromContainer(Container container, boolean parentMutualise, ExportReport report) throws PegaseExportException, IOException, ApiException
    {
        String type = null;

        String natureCode = getContainerNatureCode(container);
        boolean isYear = "annee".equals(natureCode);
        boolean isSemester = "semestre".equals(natureCode);

        // If it is a container of nature : semester of year, export it as an OF of type semester or year
        if (isYear || isSemester)
        {
            type = isYear ? __ANNEE_TYPE_ID : __SEMESTRE_TYPE_ID;

            return _createObjetFormationAndChildren(container, type, parentMutualise, report);
        }
        
        // Else, ignore it
        return null;
    }
    
    /**
     * Creates a Pégase objet formation from a content
     * @param content The content
     * @param parentMutualise If the element's parent is mutualised
     * @param report The report
     * @return An {@link Enfant} object, Pégase child
     * @throws PegaseExportException If an error occurs
     * @throws IOException If an API error occurs
     * @throws ApiException If an error occurs
     */
    private PegaseChildWithChildren _createObjetFormationAndChildren(Content content, String type, boolean parentMutualise, ExportReport report) throws IOException, PegaseExportException, ApiException
    {
        // Create or update the object
        ObjetMaquetteDetail objetMaquetteDetailCreated = _createOrUpdateObjetFormation(content, type, parentMutualise, report);
        
        _updateDescripteursSyllabus(content, objetMaquetteDetailCreated.getId(), TypeObjetMaquette.OBJET_FORMATION, new DescripteursObjetFormationSyllabus());
        
        // Create the children if there are any
        ChildrenToAttachForObjectRequest childrenToAttach = _createChildrenIfAny(content, objetMaquetteDetailCreated, report);

        UUID objetMaquetteDetailCreatedId = objetMaquetteDetailCreated.getId();
        
        // Create and return an Enfant object from the child created
        return new PegaseChildWithChildren(_createPegaseChild(objetMaquetteDetailCreatedId, true), childrenToAttach);

    }
    
    private ObjetMaquetteDetail _createOrUpdateObjetFormation(Content content, String type, boolean parentMutualise, ExportReport report) throws PegaseExportException
    {
        try
        {
            // Check if the object already exists
            ObjetMaquetteAndEtagIfExists objetMaquetteAndEtagIfExists = _getPegaseObjetFormationIfAlreadyExists(content, report);
            
            ObjetMaquetteDetail objetMaquetteDetail = objetMaquetteAndEtagIfExists.objetMaquetteDetail();
            // If it already exists, update the object
            if (objetMaquetteDetail != null)
            {
                return _updateObjetFormation(objetMaquetteAndEtagIfExists, content, type, parentMutualise, report);
            }
            // If it did not already exist, create the object
            else
            {
                return _createObjetFormation(content, type, parentMutualise, report);
            }
        }
        catch (Exception e)
        {
            report.updateExportReport(ExportStatus.ERROR, ProblemTypes.API_ERROR);

            throw new PegaseExportException("Une erreur est survenue lors de la récupération de l'objet Pégase déjà existant", e);
        }
    }
    
    private PegaseChildWithChildren _createGroupFromCourseList(CourseList courseList, boolean parentMutualise, ExportReport report) throws PegaseExportException
    {
        ChoiceType type = courseList.getType();
        boolean mandatory = !ChoiceType.OPTIONAL.equals(type);
        PlageDeChoix plageDeChoix = ChoiceType.CHOICE.equals(type) ? _buildPlageDeChoix(courseList) : null;
        
        ObjetMaquetteDetail objetMaquetteDetailCreated = _createOrUpdateGroup(courseList, parentMutualise, plageDeChoix, report);

        ChildrenToAttachForObjectRequest childrenToAttach = _createChildrenIfAny(courseList, objetMaquetteDetailCreated, report);

        UUID objetMaquetteDetailCreatedId = objetMaquetteDetailCreated.getId();
        
        return new PegaseChildWithChildren(_createPegaseChild(objetMaquetteDetailCreatedId, mandatory), childrenToAttach);
    }
    
    private ObjetMaquetteDetail _createOrUpdateGroup(CourseList courseList, boolean parentMutualise, PlageDeChoix plageDeChoix, ExportReport report) throws PegaseExportException
    {
        try
        {
            // Check if the object already exists
            ObjetMaquetteAndEtagIfExists objetMaquetteAndEtagIfExists = _getPegaseGroupIfAlreadyExists(courseList, report);
            
            ObjetMaquetteDetail objetMaquetteDetail = objetMaquetteAndEtagIfExists.objetMaquetteDetail();
            // If it already exists, update the object
            if (objetMaquetteDetail != null)
            {
                return _updateGroup(objetMaquetteAndEtagIfExists, courseList, parentMutualise, plageDeChoix, report);
            }
            // If it did not already exist, create the object
            else
            {
                return _createGroup(courseList, parentMutualise, plageDeChoix, report);
            }
        }
        catch (Exception e)
        {
            report.updateExportReport(ExportStatus.ERROR, ProblemTypes.API_ERROR);

            throw new PegaseExportException("Une erreur est survenue lors de la récupération de l'objet Pégase déjà existant", e);
        }
    }
    
    private ObjetMaquetteDetail _updateGroup(ObjetMaquetteAndEtagIfExists objetMaquetteAndEtagIfExists, Content content, boolean parentMutualise, PlageDeChoix plageDeChoix, ExportReport report) throws PegaseExportException, IOException
    {
        DescripteursGroupementRequest descripteursGroupementRequest = new DescripteursGroupementRequest();
        
        String label = content.getTitle();
        descripteursGroupementRequest.setLibelle(StringUtils.truncate(label, 50));
        descripteursGroupementRequest.setLibelleLong(StringUtils.truncate(label, 150));
        descripteursGroupementRequest.setPlageDeChoix(plageDeChoix);
        
        return _updateObjetMaquette(objetMaquetteAndEtagIfExists, content, parentMutualise, descripteursGroupementRequest, report);
    }
    
    private ObjetMaquetteDetail _createGroup(CourseList courseList, boolean parentMutualise, PlageDeChoix plageDeChoix, ExportReport report) throws PegaseExportException
    {
        DescripteursGroupementRequest descripteursGroupementRequest = new DescripteursGroupementRequest();
        if (plageDeChoix != null)
        {
            descripteursGroupementRequest.setPlageDeChoix(plageDeChoix);
        }
        
        CreerGroupementRequest creerGroupementRequest = new CreerGroupementRequest();

        return _createObjetMaquette(courseList, TypeObjetMaquette.GROUPEMENT, parentMutualise, descripteursGroupementRequest, creerGroupementRequest, report);
    }
    
    private ChildrenToAttachForObjectRequest _createChildrenIfAny(Content content, ObjetMaquetteDetail objetMaquetteDetailCreated, ExportReport report) throws UnknownDataException, AmetysRepositoryException
    {
        Map<String, PegaseChildWithChildren> children = new HashMap<>();
        if (content instanceof ProgramItem programItem)
        {
            // Create the children
            children = _createAllChildren(_odfHelper.getChildProgramItems(programItem), objetMaquetteDetailCreated.getMutualise(), report);
        }
        
        return new ChildrenToAttachForObjectRequest(content, objetMaquetteDetailCreated.getId(), children);
    }
    
    private Map<String, PegaseChildWithChildren> _createAllChildren(List<ProgramItem> programItem, boolean parentMutualise, ExportReport report)
    {
        Map<String, PegaseChildWithChildren> children = new HashMap<>();

        for (ProgramItem childContent : programItem)
        {
            PegaseChildWithChildren child = _createChild((Content) childContent, parentMutualise, report);

            // If the child was created, add it to the list of children to attach
            if (child != null)
            {
                children.put(child.pegaseChild().getId(), child);
            }
        }

        return children;
    }
    
    private void _detachAndAttachAllChildren(Content content, UUID pegaseParentID, Map<String, PegaseChildWithChildren> children, ExportReport report) throws PegaseExportException
    {
        try
        {
            // The children that are not going to be related to the program anymore
            Map<String, PegaseChildWithChildren> childrenToCheck = Map.copyOf(children);
            
            _detachChildren(content, pegaseParentID, children, report);
            
            // For each child that is not directly related to the program anymore, attach its own children recursively
            for (PegaseChildWithChildren childToAttach : childrenToCheck.values())
            {
                if (!children.containsValue(childToAttach))
                {
                    ChildrenToAttachForObjectRequest childrenToAttach = childToAttach.childrenToAttachRequest();
                    _attachAllChildren(childrenToAttach.parentContent(), childrenToAttach.parentId(), childrenToAttach.children(), report);
                }
            }
            
            _attachAllChildren(content, pegaseParentID, children, report);
        }
        catch (Exception e)
        {
            throw new PegaseExportException("Une erreur est survenue lors de l'attachement des enfants de la formation '" + content.getTitle() + "'");
        }
    }
    
    private void _detachChildren(Content content, UUID pegaseParentObjectId, Map<String, PegaseChildWithChildren> children, ExportReport report) throws UnknownDataException, AmetysRepositoryException, PegaseExportException
    {
        try
        {
            // Detach children recursively
            for (PegaseChildWithChildren childToAttach : children.values())
            {
                ChildrenToAttachForObjectRequest childrenToAttach = childToAttach.childrenToAttachRequest();
                _detachChildren(childrenToAttach.parentContent(), childrenToAttach.parentId(), childrenToAttach.children(), report);
            }
            
            MaquetteStructure structureMaquetteParent = _pegaseApiManager.getMaquettesApi().lireStructureMaquette(_structureCode, pegaseParentObjectId);
            
            ObjetMaquetteStructure parentRacine = structureMaquetteParent.getRacine();
            List<EnfantsStructure> childrenAlreadyAttached = parentRacine.getEnfants();
            
            // If there are children already attached and the option trustAmetys is checked, remove every child already attached from the list of children to attach
            if (childrenAlreadyAttached != null)
            {
                for (EnfantsStructure child : childrenAlreadyAttached)
                {
                    UUID childId = child.getObjetMaquette().getId();
                    String stringChildId = childId.toString();
                    
                    // Detach children in Pégase in order to reattach them correctly
                    if (children.keySet().contains(stringChildId))
                    {
                        _pegaseApiManager.getObjetsMaquetteApi().retirerEnfant(_structureCode, pegaseParentObjectId, childId);
                    }
                    // If trust ametys is selected, detach children attached in Pégase which are not attached in Ametys
                    else if (_trustAmetys)
                    {
                        try
                        {
                            _pegaseApiManager.getObjetsMaquetteApi().retirerEnfant(_structureCode, pegaseParentObjectId, childId);
                        }
                        catch (ApiException e)
                        {
                            report.updateExportReport(ExportStatus.ERROR, ProblemTypes.API_ERROR, content);
                            getLogger().warn("L'enfant de code Pégase '{}' de l'élément '{}' n'a pas pu être détaché dans Pégase", child.getId(), content.getTitle(), e);
                        }
                    }
                }
            }
        }
        catch (ApiException e)
        {
            report.updateExportReport(ExportStatus.WARN, ProblemTypes.API_ERROR, content);
 
            getLogger().warn("Les liens avec les enfants de l'élément '{}' n'ont pas pu être traités car les enfants depuis Pégase n'ont pas pu être récupérés.", content.getTitle(), e);
        }
        catch (Exception e)
        {
            throw new PegaseExportException("Erreur lors de l'attachement des enfants de l'élément " + content.getTitle() + ".", e);
        }
    }
    
    private void _attachAllChildren(Content content, UUID pegaseParentId, Map<String, PegaseChildWithChildren> children, ExportReport report) throws UnknownDataException, AmetysRepositoryException, PegaseExportException, IOException
    {
        // If there are children that need to be attached
        if (!children.isEmpty())
        {
            // For each child, attach its own children recursively
            for (PegaseChildWithChildren childToAttach : children.values())
            {
                ChildrenToAttachForObjectRequest childrenToAttach = childToAttach.childrenToAttachRequest();
                _attachAllChildren(childrenToAttach.parentContent(), childrenToAttach.parentId(), childrenToAttach.children(), report);
            }
            
            for (PegaseChildWithChildren child : children.values())
            {
                Enfant pegaseChildToAttach = child.pegaseChild();
                try
                {
                    _changeMutualiseIfNecessary(UUID.fromString(pegaseChildToAttach.getId()), pegaseParentId);
                
                    // Try to attach them
                    _pegaseApiManager.getObjetsMaquetteApi().ajouterEnfant(_structureCode, pegaseParentId, pegaseChildToAttach);
                }
                catch (ApiException ex)
                {
                    report.updateExportReport(ExportStatus.WARN, ProblemTypes.LINKS_MISSING, content);
                    
                    getLogger().warn("Une erreur est survenue lors de l'attachement de l'enfant ({}) à l'élément ({}) dans Pégase", pegaseChildToAttach.getId(), content.getTitle(), ex);
                }
            }
        }
    }
    
    private ObjetMaquetteAndEtagIfExists _changeMutualiseIfNecessary(UUID childId, UUID parentID) throws IOException, ApiException
    {
        ObjetsMaquetteApi objetsMaquetteApi = _pegaseApiManager.getObjetsMaquetteApi();
        ObjetMaquetteDetail childObjetMaquetteDetail = objetsMaquetteApi.lireObjetMaquette(_structureCode, childId);
        String etag = _getEtag(objetsMaquetteApi);
        ObjetMaquetteAndEtagIfExists objetMaquetteAndEtagIfExists = new ObjetMaquetteAndEtagIfExists(childObjetMaquetteDetail, etag);
        
        List<Contexte> contextes = childObjetMaquetteDetail.getContextes();
        
        // If the object has no parents (orphan), it does not need to become mutualised, and if it has multiple parents, it is already mutualised
        // If the object has exactly one parent
        if (!childObjetMaquetteDetail.getMutualise() && contextes.size() == 1)
        {
            List<UUID> chemins = contextes.get(0).getChemin();
            int numberOfChemins = chemins.size();
            
            // If the direct parent is not the program we are attaching it to, set mutualise to true
            // If there is only one element in chemins, then it is the child itself
            if (numberOfChemins > 1 && !parentID.equals(chemins.get(numberOfChemins - 1)))
            {
                return _makeHierarchyMutualised(objetMaquetteAndEtagIfExists);
            }
        }
        
        return objetMaquetteAndEtagIfExists;
    }
    
    private ObjetMaquetteAndEtagIfExists _makeHierarchyMutualised(ObjetMaquetteAndEtagIfExists objetMaquetteAndEtagIfExists) throws ApiException, IOException
    {
        ObjetMaquetteDetail objetMaquetteDetail = objetMaquetteAndEtagIfExists.objetMaquetteDetail();
        
        ObjetsMaquetteApi objetsMaquetteApi = null;

        // If there are any, make the children mutualise
        if (objetMaquetteDetail.getEnfants().size() != 0)
        {
            MaquetteStructure structureMaquetteParent = _pegaseApiManager.getMaquettesApi().lireStructureMaquette(_structureCode, objetMaquetteDetail.getId());
            
            objetsMaquetteApi = _pegaseApiManager.getObjetsMaquetteApi();
            ObjetMaquetteStructure racine = structureMaquetteParent.getRacine();
            
            List<EnfantsStructure> children = racine.getEnfants();
            
            for (EnfantsStructure child : children)
            {
                // Need to retrieve the objet maquette again to get the Etag
                ObjetMaquetteDetail childObjetMaquette = objetsMaquetteApi.lireObjetMaquette(_structureCode, child.getObjetMaquette().getId());
                // Make the child hierarchy mutualise recusively
                _makeHierarchyMutualised(new ObjetMaquetteAndEtagIfExists(childObjetMaquette, _getEtag(objetsMaquetteApi)));
            }
        }
        else
        {
            // Retrieve the API again to reset the "external" attribute to false
            objetsMaquetteApi = _pegaseApiManager.getObjetsMaquetteApi();
        }
        
        objetMaquetteDetail = objetsMaquetteApi.rendreMutualise(_structureCode, objetMaquetteDetail.getId(), objetMaquetteAndEtagIfExists.etag());
        
        return new ObjetMaquetteAndEtagIfExists(objetMaquetteDetail, _getEtag(objetsMaquetteApi));
    }
    
    private ObjetMaquetteDetail _updateObjetFormation(ObjetMaquetteAndEtagIfExists objetMaquetteAndEtagIfExists, Content content, String type, boolean parentMutualise, ExportReport report) throws PegaseExportException, IOException
    {
        DescripteursObjetFormationRequest descripteursObjetFormationRequest = new DescripteursObjetFormationRequest();
        
        String label = content.getTitle();
        descripteursObjetFormationRequest.setLibelle(StringUtils.truncate(label, 50));
        descripteursObjetFormationRequest.setLibelleLong(StringUtils.truncate(label, 150));
        
        // Set the formation type (ANNEE, SEMESTRE, UE...)
        descripteursObjetFormationRequest.setType(type);
        
        descripteursObjetFormationRequest.setEcts(_getEctsFromContent(content));
        
        return _updateObjetMaquette(objetMaquetteAndEtagIfExists, content, parentMutualise, descripteursObjetFormationRequest, report);
    }
    
    private ObjetMaquetteDetail _updateObjetMaquette(ObjetMaquetteAndEtagIfExists objetMaquetteAndEtagIfExists, Content content, boolean parentMutualise, DescripteursObjetMaquetteRequest descObjetMaquette, ExportReport report) throws PegaseExportException, IOException
    {
        try
        {
            ObjetMaquetteAndEtagIfExists objetMaquetteAndEtagIfExistsMutualised = objetMaquetteAndEtagIfExists;
            // If the parent is mutualized, make the object mutualized too, and retrieve the new Etag and object from the update
            if (parentMutualise)
            {
                objetMaquetteAndEtagIfExistsMutualised = _makeHierarchyMutualised(objetMaquetteAndEtagIfExists);
            }
            
            ObjetMaquetteDetail objetMaquetteDetailFound = objetMaquetteAndEtagIfExistsMutualised.objetMaquetteDetail();
            ObjetsMaquetteApi objetsMaquetteApi = _pegaseApiManager.getObjetsMaquetteApi();
            
            ObjetMaquetteDetail objetMaquetteDetail = objetsMaquetteApi.modifierDescripteursObjetMaquette(_structureCode, objetMaquetteDetailFound.getId(), objetMaquetteAndEtagIfExistsMutualised.etag(), descObjetMaquette);
            
            return objetMaquetteDetail;
        }
        catch (ApiException e)
        {
            report.updateExportReport(ExportStatus.ERROR, ProblemTypes.ELEMENT_NOT_EXPORTED);

            throw new PegaseExportException("L'élément " + content.getTitle() + " n'a pas pu être exportée car sa modification dans Pégase a posé un problème", e);
        }
    }
    
    private ObjetMaquetteDetail _createObjetFormation(Content content, String type, boolean parentMutualise, ExportReport report) throws PegaseExportException
    {
        DescripteursObjetFormationRequest descripteursObjetFormationRequest = new DescripteursObjetFormationRequest();
        descripteursObjetFormationRequest.setType(type);
        
        descripteursObjetFormationRequest.setEcts(_getEctsFromContent(content));
        
        CreerObjetFormationRequest creerObjetFormationRequest = new CreerObjetFormationRequest();
        
        return _createObjetMaquette(content, TypeObjetMaquette.OBJET_FORMATION, parentMutualise, descripteursObjetFormationRequest, creerObjetFormationRequest, report);
    }
    
    private ObjetMaquetteDetail _createObjetMaquette(Content content, TypeObjetMaquette typeObjetMaquette, boolean parentMutualise, DescripteursObjetMaquetteRequest descripteursObjetMaquetteRequest, CreerObjetMaquetteRequest creerObjetMaquetteRequest, ExportReport report) throws PegaseExportException
    {
        try
        {
            ObjetsMaquetteApi objetsMaquetteApi = _pegaseApiManager.getObjetsMaquetteApi();
            
            _initCreerObjetMaquetteRequest(creerObjetMaquetteRequest, typeObjetMaquette, parentMutualise, content);

            String label = content.getTitle();
            descripteursObjetMaquetteRequest.setLibelle(StringUtils.truncate(label, 50));
            descripteursObjetMaquetteRequest.setLibelleLong(StringUtils.truncate(label, 150));
            
            creerObjetMaquetteRequest.setDescripteursObjetMaquette(descripteursObjetMaquetteRequest);

            ObjetMaquetteDetail objetMaquetteDetail = objetsMaquetteApi.creerObjetMaquette(_structureCode, creerObjetMaquetteRequest);
            
            return objetMaquetteDetail;
        }
        catch (IOException e)
        {
            report.updateExportReport(ExportStatus.ERROR, ProblemTypes.API_ERROR);

            throw new PegaseExportException("Une erreur est survenue lors de la création de l'objet Pégase", e);
        }
        catch (ApiException e)
        {
            report.updateExportReport(ExportStatus.ERROR, ProblemTypes.ELEMENT_NOT_EXPORTED);

            throw new PegaseExportException("L'élément " + content.getTitle() + " n'a pas pu être exporté dû à un problème rencontré avec Pégase", e);
        }
    }
    
    private PlageDeChoix _buildPlageDeChoix(CourseList courseList)
    {
        PlageDeChoix plageDeChoix = new PlageDeChoix();
        
        Long min = courseList.getValue(CourseList.MIN_COURSES, false, 0L);
        plageDeChoix.setMin(min.intValue());
        
        Long max = courseList.getValue(CourseList.MAX_COURSES, false, min);
        plageDeChoix.setMax(max.intValue());
        
        return plageDeChoix;
    }
    
    /**
     * If the objet exists, store the objetMaquette detail and its Etag
     * @param objetMaquetteDetail The objet maquette detail
     * @param etag The etag of the object (its version)
     */
    protected record ObjetMaquetteAndEtagIfExists (ObjetMaquetteDetail objetMaquetteDetail, String etag) { /* empty */ }
    
    /**
     * Store the parameters necessary to attach all children of an object
     * @param parentContent The content of the object
     * @param parentId The pegase ID of the object
     * @param children The children of the object
     */
    protected record ChildrenToAttachForObjectRequest (Content parentContent, UUID parentId, Map<String, PegaseChildWithChildren> children) { /* empty */ }
    
    /**
     * Store the Pegase child and its children to attach request
     * @param pegaseChild The pegase child
     * @param childrenToAttachRequest The children to attach request parameters as a {@link ChildrenToAttachForObjectRequest} object
     */
    protected record PegaseChildWithChildren (Enfant pegaseChild, ChildrenToAttachForObjectRequest childrenToAttachRequest) { /* empty */ }
}
