/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfsync.pegase.ws;

import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Base64.Decoder;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpEntity;
import org.apache.http.NameValuePair;
import org.apache.http.StatusLine;
import org.apache.http.client.HttpResponseException;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.message.BasicNameValuePair;

import org.ametys.core.util.HttpUtils;
import org.ametys.core.util.JSONUtils;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Manager to request Pégase token when needed.
 */
public class PegaseTokenManager extends AbstractLogEnabled implements Component, Serviceable, Initializable
{
    /** Role */
    public static final String ROLE = PegaseTokenManager.class.getName();

    /** The JSON utils */
    protected JSONUtils _jsonUtils;
    
    /* Token */
    private String _token;
    private Instant _expirationDate;
    
    /* Configuration */
    private boolean _isActive;
    private String _username;
    private String _password;
    private String _authUrl;

    public void service(ServiceManager manager) throws ServiceException
    {
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
    }
    
    public void initialize() throws Exception
    {
        _isActive = Config.getInstance().getValue("pegase.activate", true, false);
        if (_isActive)
        {
            /* Authentication configuration */
            _authUrl = HttpUtils.sanitize(Config.getInstance().getValue("pegase.auth.url"));
            _username = Config.getInstance().getValue("pegase.api.username");
            _password = Config.getInstance().getValue("pegase.api.password");
        }
    }
    /**
     * Get the token to log to Pégase API
     * @throws IOException if an error occurs
     * @return a valid token
     */
    public synchronized String getToken() throws IOException
    {
        if (!_isActive)
        {
            throw new UnsupportedOperationException("Pégase is not active in the configuration, you cannot request a token.");
        }
        
        if (_expirationDate == null || Instant.now().isAfter(_expirationDate))
        {
            CloseableHttpClient httpClient = _getHttpClient();
    
            List<NameValuePair> urlParams = new ArrayList<>();
            urlParams.add(new BasicNameValuePair("username", _username));
            urlParams.add(new BasicNameValuePair("password", _password));
            urlParams.add(new BasicNameValuePair("token", "true"));
            
            // Prepare a request object
            HttpPost postRequest = new HttpPost(_authUrl);
            
            // HTTP parameters
            postRequest.setEntity(new UrlEncodedFormEntity(urlParams, "UTF-8"));
            postRequest.setHeader("Content-Type", "application/x-www-form-urlencoded");
            
            // Execute the request
            _token = _executeHttpRequest(httpClient, postRequest);
            _expirationDate = _extractExpirationDate();
        }
        
        return _token;
    }
    
    /**
     * Extract the expiration date from the token.
     * @return the expiration dat of the current token
     */
    protected Instant _extractExpirationDate()
    {
        String[] splitToken = _token.split("\\.");
        
        if (splitToken.length < 2)
        {
            getLogger().error("Invalid token format, cannot get the expiration date. The token will be reset at each API call.");
            return null;
        }
        
        String playload = splitToken[1];
        Decoder decoder = Base64.getUrlDecoder();
        byte[] decodedToken = decoder.decode(playload);
        
        String decodedTokenString = new String(decodedToken, StandardCharsets.UTF_8);
        Map<String, Object> map = _jsonUtils.convertJsonToMap(decodedTokenString);
        Long expirationDateInSeconds = Long.parseLong(map.get("exp").toString());
        
        return Instant.ofEpochSecond(expirationDateInSeconds);
    }
    
    private CloseableHttpClient _getHttpClient()
    {
        RequestConfig requestConfig = RequestConfig.custom().build();
        return HttpClientBuilder.create()
                .setDefaultRequestConfig(requestConfig)
                .useSystemProperties()
                .build();
    }
    
    private String _executeHttpRequest(CloseableHttpClient httpClient, HttpUriRequest httpRequest) throws IOException
    {
        httpRequest.setHeader("accept", "application/json");
        
        // Execute the request
        try (CloseableHttpResponse httpResponse = httpClient.execute(httpRequest))
        {
            StatusLine statusLine = httpResponse.getStatusLine();
            if (statusLine.getStatusCode() / 100 != 2)
            {
                throw new HttpResponseException(statusLine.getStatusCode(), statusLine.getReasonPhrase());
            }
            
            HttpEntity entity = httpResponse.getEntity();
            if (entity == null)
            {
                throw new IOException("The response entity is empty.");
            }
            
            try (InputStream is = entity.getContent())
            {
                return IOUtils.toString(is, StandardCharsets.UTF_8);
            }
        }
    }
}
