/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.observation;

import java.util.Map;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.odf.orgunit.OrgUnit;
import org.ametys.odf.program.Program;
import org.ametys.plugins.odfweb.repository.FirstLevelPageFactory;
import org.ametys.plugins.odfweb.repository.ODFPageCache;
import org.ametys.plugins.odfweb.repository.OdfPageHandler;
import org.ametys.plugins.odfweb.repository.OdfPageResolver;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.provider.RequestAttributeWorkspaceSelector;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.VirtualFactoryExpression;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.WebConstants;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.PageQueryHelper;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.repository.sitemap.Sitemap;

/**
 * Clear ODF caches when page ODF root page has changed
 *
 */
public class ClearODFPageCacheObserver extends AbstractLogEnabled implements Observer, Serviceable, Contextualizable
{
    /** The ODF root page handler. */
    protected OdfPageHandler _odfPageHandler;
    
    /** The ODF page cache */
    protected ODFPageCache _odfPageCache;
    
    /** Site manager */
    protected SiteManager _siteManager;
    
    /** Odf page resolver */
    protected OdfPageResolver _odfPageResolver;
    
    /** Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** Avalon context */
    protected Context _context;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _odfPageHandler = (OdfPageHandler) manager.lookup(OdfPageHandler.ROLE);
        _odfPageCache = (ODFPageCache) manager.lookup(ODFPageCache.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _odfPageResolver = (OdfPageResolver) manager.lookup(OdfPageResolver.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }

    @Override
    public boolean supports(Event event)
    {
        Content content = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
        return event.getId().equals(org.ametys.web.ObservationConstants.EVENT_PAGE_CHANGED) 
            || event.getId().equals(org.ametys.web.ObservationConstants.EVENT_SITE_UPDATED)
            || event.getId().equals(ObservationConstants.EVENT_CONTENT_MODIFIED) && content instanceof OrgUnit/* for handling moved orgUnit */
            || event.getId().equals(ObservationConstants.EVENT_CONTENT_VALIDATED) && (content instanceof OrgUnit || content instanceof Program);
    }

    @Override
    public int getPriority(Event event)
    {
        // Will be processed AFTER live synchronization observers and BEFORE page indexation observer
        return MAX_PRIORITY + 1100;
    }

    @Override
    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Content targetContent = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
        Page targetPage = (Page) event.getArguments().get(org.ametys.web.ObservationConstants.ARGS_PAGE);
        Site targetSite = (Site) event.getArguments().get(org.ametys.web.ObservationConstants.ARGS_SITE);
        if (targetPage != null)
        {
            _clearCache(targetPage);
        }
        else if (targetContent instanceof Program)
        {
            _clearCache((Program) targetContent);
        }
        else if (targetContent instanceof OrgUnit)
        {
            _clearCache((OrgUnit) targetContent);
        }
        
        else if (targetSite != null)
        {
            _clearCache(targetSite);
        }
    }
    
    /**
     * Clear the cache when necessary in case of a page 
     * @param page The page
     */
    protected void _clearCache(Page page)
    {
        if (_odfPageHandler.isODFRootPage(page))
        {
            // Clear ODF page cache
            _odfPageCache.clearCache(page);
        }
    }
    
    /**
     * Clear the cache when necessary in case of a program 
     * @param program The program
     */
    protected void _clearCache(Program program)
    {
        AmetysObjectIterable<Page> rootPages = _getOdfRootPages();
        
        for (Page rootPage : rootPages)
        {
            _odfPageCache.clearCache(rootPage);
        }
    }
    
    private AmetysObjectIterable<Page> _getOdfRootPages()
    {
        Expression expression = new VirtualFactoryExpression(FirstLevelPageFactory.class.getName());
        String query = PageQueryHelper.getPageXPathQuery(null, null, null, expression, null);
        return _resolver.query(query);
    }
    
    /**
     * Clear the cache when necessary in case of an orgUnit 
     * @param orgUnit The orgUnit
     */
    protected void _clearCache(OrgUnit orgUnit)
    {
        Request request = ContextHelper.getRequest(_context);

        // Retrieve current workspace
        String currentWsp = RequestAttributeWorkspaceSelector.getForcedWorkspace(request);

        try
        {
            // Use live workspace
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, WebConstants.LIVE_WORKSPACE);
            for (Site site : _siteManager.getSites())
            {
                if (_odfPageHandler.hasOdfRootPage(site))
                {
                    try (AmetysObjectIterable<Sitemap> sitemaps = site.getSitemaps())
                    {
                        for (Sitemap sitemap : sitemaps)
                        {
                            for (Page page : _odfPageHandler.getOdfRootPages(sitemap.getSiteName(), sitemap.getSitemapName()))
                            {
                                // Clear ODF page cache
                                _odfPageCache.clearCache(page);
                            }
                        }
                    }
                }
            }
        }
        finally
        {
            // Restore context
            RequestAttributeWorkspaceSelector.setForcedWorkspace(request, currentWsp);
        }
    }
    
    /**
     * Clear the cache when necessary in case of a site 
     * @param site The site
     */
    protected void _clearCache(Site site)
    {
        String siteName = site.getName();
        
        for (Sitemap sitemap : site.getSitemaps())
        {
            for (Page odfRootPage : _odfPageHandler.getOdfRootPages(siteName, sitemap.getSitemapName()))
            {
                _odfPageCache.clearCache(odfRootPage);
            }
        }
    }
}
