/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.program;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.content.indexing.solr.SolrFieldNames;
import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.search.SearchResults;
import org.ametys.cms.search.query.AndQuery;
import org.ametys.cms.search.query.ConstantNilScoreQuery;
import org.ametys.cms.search.query.ContentTypeQuery;
import org.ametys.cms.search.query.DocumentTypeQuery;
import org.ametys.cms.search.query.JoinQuery;
import org.ametys.cms.search.query.MatchAllQuery;
import org.ametys.cms.search.query.MaxScoreOrQuery;
import org.ametys.cms.search.query.OrQuery;
import org.ametys.cms.search.query.Query;
import org.ametys.cms.search.query.StringQuery;
import org.ametys.cms.search.solr.SearcherFactory.Searcher;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.odf.program.ProgramFactory;
import org.ametys.odf.program.ProgramPart;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.program.SubProgramFactory;
import org.ametys.odf.program.TraversableProgramPart;
import org.ametys.odf.skill.ProgramSkillsProperty;
import org.ametys.plugins.odfweb.repository.OdfPageResolver;
import org.ametys.plugins.odfweb.repository.ProgramPage;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.runtime.model.View;
import org.ametys.web.frontoffice.SearchGenerator;
import org.ametys.web.indexing.solr.SolrWebFieldNames;
import org.ametys.web.repository.page.Page;
import org.ametys.web.search.query.PageContentQuery;

/**
 * ODF search results
 */
public class FrontODFSearch extends SearchGenerator
{
    /**
     * Enumeration for display subprogram mode
     */
    public enum DisplaySubprogramMode
    {
        /** Display no subprogram */
        NONE,
        /** Display all subprograms */
        ALL,
        /** Display all subprograms with highlighting those which match the search criteria*/
        ALL_WITH_HIGHLIGHT,
        /** Display matching subprograms only */
        MATCHING_SEARCH_ONLY
    }

    /** The matching subprograms */
    protected List<String> _matchingSubProgramIds;
    
    private DisplaySubprogramMode _displaySubprogramMode;
    private OdfPageResolver _odfPageResolver;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _odfPageResolver = (OdfPageResolver) smanager.lookup(OdfPageResolver.ROLE);
    }
    
    @Override
    protected Collection<String> getContentTypes(Request request)
    {
        return Arrays.asList(parameters.getParameter("contentType", ProgramFactory.PROGRAM_CONTENT_TYPE));
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        _displaySubprogramMode = null;
        _matchingSubProgramIds = new ArrayList<>();
        
        super.generate();
    }
    
    @Override
    protected List<Query> getContentQueries(Request request, Collection<String> siteNames, String language)
    {
        List<Query> contentQueries = super.getContentQueries(request, siteNames, language);
        
        // Add catalog query
        String catalog = parameters.getParameter("catalog", request.getParameter("catalog"));
        contentQueries.add(new ConstantNilScoreQuery(new StringQuery(ProgramItem.CATALOG, catalog)));
        
        // Add query on acquired skill's id if present
        String skillId = request.getParameter("skillId");
        if (StringUtils.isNotBlank(skillId))
        {
            contentQueries.add(new StringQuery(ProgramSkillsProperty.PROGRAM_SKILLS_PROPERTY_NAME, skillId));
        }
        
        return contentQueries;
    }
    
    @Override
    protected Query getQuery(Request request, Collection<String> siteNames, String language) throws IllegalArgumentException
    {
        List<Query> finalQueries = new ArrayList<>();
        
        List<Query> wordingQueries = getWordingQueries(request, siteNames, language);
        Query wordingQuery = wordingQueries.isEmpty() ? new MatchAllQuery() : new AndQuery(wordingQueries);
        
        // Query to execute on pages
        List<Query> pagesQueries = new ArrayList<>();  
        pagesQueries.addAll(getPageQueries(request, siteNames, language)); // add specific queries to pages
        
        if (!pagesQueries.isEmpty())
        {
            pagesQueries.add(wordingQuery); // add wording queries
            finalQueries.add(new AndQuery(pagesQueries));
        }
        
        // Query to execute on joined contents
        List<Query> contentQueries = new ArrayList<>(wordingQueries); // add wording queries
        contentQueries.addAll(getContentQueries(request, siteNames, language)); // add specific queries to contents
        Query contentQuery = new AndQuery(contentQueries);
        
        List<Query> contentOrResourcesQueries = new ArrayList<>();
        contentOrResourcesQueries.add(contentQuery);
        contentOrResourcesQueries.addAll(getContentResourcesOrAttachmentQueries(wordingQuery)); // add queries on join content's resources
        
        Query programContentQuery = new OrQuery(contentOrResourcesQueries);
        Query programPageQuery = new PageContentQuery(programContentQuery);
        
        finalQueries.add(programPageQuery);

        Query subProgramPageQuery = null;
        if (_searchOnSubPrograms())
        {
            subProgramPageQuery = getSubProgramPageQuery(programContentQuery); // add query on joined subprograms
            finalQueries.add(subProgramPageQuery);
        }
        
        return finalQueries.isEmpty() ? new MatchAllQuery() : finalQueries.size() == 1 ? finalQueries.get(0) : new MaxScoreOrQuery(finalQueries);
    }
    
    @Override
    protected SearchResults<AmetysObject> search(Request request, Collection<String> siteNames, String language, int pageIndex, int start, int maxResults, boolean saxResults)
            throws Exception
    {
        _matchingSubProgramIds = new ArrayList<>();
        if (saxResults)
        {
            DisplaySubprogramMode displaySubProgramMode = getDisplaySubProgramMode();
            if (displaySubProgramMode.equals(DisplaySubprogramMode.ALL_WITH_HIGHLIGHT) || displaySubProgramMode.equals(DisplaySubprogramMode.MATCHING_SEARCH_ONLY))
            {
                _matchingSubProgramIds = getSubProgramsMatchingSearch(request, siteNames, language);
            }
        }
        
        return super.search(request, siteNames, language, pageIndex, start, maxResults, saxResults);
    }
    
    /**
     * Get the ids of subprograms matching the current search
     * @param request The request
     * @param siteNames The site names
     * @param language The languages
     * @return the ids of matching subprograms
     * @throws Exception if failed to execute search
     */
    protected List<String> getSubProgramsMatchingSearch(Request request, Collection<String> siteNames, String language) throws Exception
    {
        List<Query> wordingQueries = getWordingQueries(request, siteNames, language);
        Query wordingQuery = wordingQueries.isEmpty() ? new MatchAllQuery() : new AndQuery(wordingQueries);
        
        // Query to execute on joined contents
        List<Query> contentQueries = new ArrayList<>(wordingQueries); // add wording queries
        contentQueries.addAll(getContentQueries(request, siteNames, language)); // add specific queries to contents
        contentQueries.add(new ContentTypeQuery(SubProgramFactory.SUBPROGRAM_CONTENT_TYPE));
        Query contentQuery = new AndQuery(contentQueries);
        
        List<Query> contentOrResourcesQueries = new ArrayList<>();
        contentOrResourcesQueries.add(contentQuery);
        contentOrResourcesQueries.addAll(getContentResourcesOrAttachmentQueries(wordingQuery)); // add queries on join content's resources
        
        Searcher searcher = _searcherFactory.create()
                .withQuery(new OrQuery(contentOrResourcesQueries))
                .addFilterQuery(new DocumentTypeQuery(SolrFieldNames.TYPE_CONTENT))
                .withLimits(0, Integer.MAX_VALUE)
                .setCheckRights(_checkRights());
        
        AmetysObjectIterable<AmetysObject> subPrograms = searcher.search();
        return subPrograms.stream().map(ao -> ao.getId()).collect(Collectors.toList());
    }
    
    /**
     * Get the page query to execute for subprogram's pages
     * @param contentQuery the initial content query
     * @return the page query for subprogram
     */
    protected Query getSubProgramPageQuery(Query contentQuery)
    {
        Query subProgramTypeQuery = new ConstantNilScoreQuery(new ContentTypeQuery(SubProgramFactory.SUBPROGRAM_CONTENT_TYPE));
        Query subProgramContentQuery = new AndQuery(subProgramTypeQuery, contentQuery);
        return new SubProgramPageContentQuery(subProgramContentQuery);
    }
    
    @Override
    protected void saxAdditionalInfosOnPageHit(Page page) throws SAXException
    {
        super.saxAdditionalInfosOnPageHit(page);
        
        DisplaySubprogramMode displaySubProgramMode = getDisplaySubProgramMode();
        
        if (displaySubProgramMode != DisplaySubprogramMode.NONE && page instanceof ProgramPage)
        {
            ContentType contentType = _cTypeExtPt.getExtension(SubProgramFactory.SUBPROGRAM_CONTENT_TYPE);
            View view = Optional.ofNullable(contentType).map(cType -> cType.getView("index")).orElse(null);
            
            if (page instanceof ProgramPage)
            {
                String programPath = page.getPathInSitemap();
                AbstractProgram<ProgramFactory> program = ((ProgramPage) page).getProgram();
                for (ProgramPart childProgramPart : program.getProgramPartChildren())
                {
                    if (childProgramPart instanceof SubProgram)
                    {
                        SubProgram subProgram = (SubProgram) childProgramPart;
                        
                        boolean matchSearch = _matchingSubProgramIds.contains(subProgram.getId());
                        if (!_displaySubprogramMode.equals(DisplaySubprogramMode.MATCHING_SEARCH_ONLY) || matchSearch)
                        {
                            AttributesImpl attrs = new AttributesImpl();
                            Page subProgramPage = _odfPageResolver.getSubProgramPage(subProgram, program, page.getSiteName());
                            if (subProgramPage != null)
                            {
                                attrs.addCDATAAttribute("path", StringUtils.substringAfterLast(subProgramPage.getPathInSitemap(), programPath));
                            }
                            else
                            {
                                getLogger().warn("The subprogram '" + subProgram.getId() + "' was returned from the search but its virtual page could not be resolved");
                            }
                            attrs.addCDATAAttribute("title", subProgram.getTitle());
                            if (_displaySubprogramMode == DisplaySubprogramMode.ALL_WITH_HIGHLIGHT)
                            {
                                attrs.addCDATAAttribute("highlight", String.valueOf(matchSearch));
                            }
                            XMLUtils.startElement(contentHandler, "subprogram", attrs);

                            try
                            {
                                subProgram.dataToSAX(contentHandler, view);
                            }
                            catch (Exception e)
                            {
                                getLogger().error("An error occurred during saxing subprogram '" + subProgram.getId() + "' metadata", e);
                            }
                            
                            XMLUtils.endElement(contentHandler, "subprogram");
                        }
                    }
                }
            }
        }
    }
    
    /**
     * Get the display mode for subprograms
     * @return the display mode
     */
    protected DisplaySubprogramMode getDisplaySubProgramMode()
    {
        if (_displaySubprogramMode == null)
        {
            String displaySubprogramsParam = parameters.getParameter("display-subprograms", "none");
            _displaySubprogramMode = DisplaySubprogramMode.valueOf(displaySubprogramsParam.toUpperCase());
        }
        return _displaySubprogramMode;
        
    }
    /**
     * Determines the search should be executed on subprograms
     * @return true to execute search also on subprograms
     */
    protected boolean _searchOnSubPrograms()
    {
        return parameters.getParameterAsBoolean("subprogram-search", false);
    }
    
    @Override
    protected void addContentTypeQuery(Collection<Query> queries, Request request)
    {
        Collection<String> cTypes = new ArrayList<>(getContentTypes(request));
        queries.add(new PageContentQuery(new ContentTypeQuery(cTypes)));
    }
    
    class SubProgramPageContentQuery extends JoinQuery
    {
        public SubProgramPageContentQuery(Query subQuery)
        {
            super(subQuery, SolrWebFieldNames.CONTENT_IDS, TraversableProgramPart.CHILD_PROGRAM_PARTS);
        }
    }
}
