/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.repository;

import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.lang3.StringUtils;

import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.courselist.CourseList;
import org.ametys.odf.program.Program;
import org.ametys.plugins.odfweb.repository.ProgramPage.AbstractTreeIterator;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.CollectionIterable;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.data.holder.ModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.impl.DefaultModelLessDataHolder;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.MemoryRepositoryData;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.virtual.VirtualPageConfiguration;

import com.google.common.collect.Iterables;

/**
 * Page representing a course.
 */
public class CoursePage extends AbstractProgramItemPage<CoursePageFactory>
{
    private Course _course;
    private String _path;
    private Page _parentPage;
    private Program _parentProgram;

    /**
     * Constructor.
     * @param factory The factory
     * @param root the odf root page.
     * @param course the course.
     * @param parentProgram the parent program
     * @param path path from the parent {@link ProgramPage}
     * @param parentPage the parent {@link Page} or null if not yet computed.
     * @param configuration The Course virtual page's configuration
     */
    public CoursePage(Page root, VirtualPageConfiguration configuration, CoursePageFactory factory, Course course, Program parentProgram, String path, Page parentPage)
    {
        super(root, configuration, factory.getScheme(), factory);
        
        _course = course;
        _path = path;
        _parentPage = parentPage;
        _parentProgram = parentProgram;
    }
    
    /**
     * Returns the associated {@link Course}.
     * @return the associated {@link Course}.
     */
    public Course getCourse()
    {
        return _course;
    }
    
    @Override
    protected ProgramItem getProgramItem()
    {
        return getCourse();
    }

    @Override
    public int getDepth() throws AmetysRepositoryException
    {
        int levelDepth = 0;
        if (StringUtils.isNotBlank(_factory.getODFPageHandler().getLevel1Metadata(_root)))
        {
            levelDepth++;
            if (StringUtils.isNotBlank(_factory.getODFPageHandler().getLevel2Metadata(_root)))
            {
                levelDepth++;
            }
        }
        
        return _root.getDepth() + levelDepth + _path.split("/").length;
    }

    @Override
    public String getTitle() throws AmetysRepositoryException
    {
        return _course.getTitle();
    }

    @Override
    public String getLongTitle() throws AmetysRepositoryException
    {
        return _course.getTitle();
    }
 
    @Override
    public AmetysObjectIterable<? extends Page> getChildrenPages() throws AmetysRepositoryException
    {
        Collection<CoursePage> children = _transformChildrenPages(_traverseCourseLists()).toList();
        return new CollectionIterable<>(children);
    }

    private CoursePage _createCoursePage(Course course)
    {
        return _factory.createCoursePage(_root, course, _parentProgram, _path + '/' + getName(), this);
    }

    @Override
    public String getPathInSitemap() throws AmetysRepositoryException
    {
        String path = _computePath(_root.getPathInSitemap());
        return path == null ? null : path + "/" + getName();
    }
    
    private Program _getParentProgram()
    {
        return _parentProgram;
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public <A extends AmetysObject> A getChild(String path) throws AmetysRepositoryException, UnknownAmetysObjectException
    {
        if (path.isEmpty())
        {
            throw new AmetysRepositoryException("path must be non empty");
        }
        
        List<String> headQueuePath = Arrays.asList(StringUtils.split(path, "/", 2));
        String name = headQueuePath.get(0);
        String queuePath = Iterables.get(headQueuePath, 1, null);
        
        return (A) _findChildPage(name)
            .map(cp -> _factory.getODFPageHandler().addRedirectIfNeeded(cp, name))
            .map(cp -> _factory.getODFPageHandler().exploreQueuePath(cp, queuePath))
            .orElseThrow(() -> new UnknownAmetysObjectException("Unknown child page '" + path + "' for page " + getId()));
    }
    
    private Optional<CoursePage> _findChildPage(String name)
    {
        return _transformChildrenPages(_traverseCourseLists().filter(child -> _filterByName(child, name))).findFirst();
    }
    
    private boolean _filterByName(Course course, String pageName)
    {
        // If last part is equals to the course code, the page matches
        return course.getCode().equals(pageName.substring(pageName.lastIndexOf("-") + 1));
    }

    @Override
    public boolean hasChild(String name) throws AmetysRepositoryException
    {
        return _findChildPage(name).isPresent();
    }

    @Override
    public String getId() throws AmetysRepositoryException
    {
        // E.g: course://licence-lea-anglais-allemand-H6YOLDDJ/parcours-1-f7usj1ss?rootId=xxx&courseId=xxx&programId=xxxx
        return "course://" + _path + "?rootId=" + _root.getId() + "&courseId=" + _course.getId() + "&programId=" + _parentProgram.getId();
    }
    @Override
    public String getName() throws AmetysRepositoryException
    {
        // E.g: langue-anglaise-1-H6YP1P98
        return _factory.getODFPageHandler().getPageName(_course);
    }

    @SuppressWarnings("unchecked")
    @Override
    public Page getParent() throws AmetysRepositoryException
    {
        if (_parentPage == null)
        {
            String childPath = _computePath(null);
            if (childPath != null)
            {
                _parentPage = childPath.isEmpty() ? _root : _root.getChild(childPath);
            }
        }
        
        return _parentPage;
    }
    
    @Override
    public String getParentPath() throws AmetysRepositoryException
    {
        return _computePath(_root.getPath());
    }

    public ModelLessDataHolder getDataHolder()
    {
        RepositoryData repositoryData = new MemoryRepositoryData(getName());
        return new DefaultModelLessDataHolder(_factory.getPageDataTypeEP(), repositoryData);
    }

    private Stream<Course> _traverseCourseLists()
    {
        CourseListTraverser traverser = new CourseListTraverser(_course.getCourseLists());
        return traverser.stream()
                .filter(Course.class::isInstance)
                .map(Course.class::cast);
    }
    
    static class CourseListTraverser extends AbstractTreeIterator<ProgramItem>
    {
        public CourseListTraverser(Collection<? extends ProgramItem> programPartChildren)
        {
            super(programPartChildren);
        }
        
        @Override
        protected Iterator<ProgramItem> provideChildIterator(ProgramItem parent)
        {
            if (parent instanceof CourseList courseList)
            {
                return new CourseListTraverser(courseList.getCourses());
            }
            
            return null;
        }
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public Course getContent()
    {
        Course course = getCourse();
        course.setContextPath(getPathInSitemap());
        
        if (!_factory.isIndexing())
        {
            // computing educational paths is actually very expensive, and only useful during rendering
            // we are very conservative here and only disable that computing for specific indexing cases
            // (we could have chosen to only enable it when rendering, but we don't want to forget specific cases)
            setCurrentEducationalPaths(course);
        }
        
        return course;
    }
    
    private String _computePath(String rootPath)
    {
        String levelsPath = _factory.getODFPageHandler().computeLevelsPath(_root, _getParentProgram());
        
        // The current program has no valid attributes for the levels selected in the ODF root
        if (levelsPath == null)
        {
            throw new UnknownAmetysObjectException("Page of program " + _getParentProgram().getId() + " does not have a valid level path");
        }
        
        return Stream.of(rootPath, levelsPath, _path)
            .filter(StringUtils::isNotEmpty)
            .collect(Collectors.joining("/"));
    }
    
    private Stream<CoursePage> _transformChildrenPages(Stream<Course> children)
    {
        return children
            .map(this::_createCoursePage)
            .filter(Objects::nonNull)
            // Test if the child page is in existing virtual pages
            .filter(page -> {
                try
                {
                    page.getPathInSitemap();
                    return true;
                }
                catch (UnknownAmetysObjectException e)
                {
                    return false;
                }
            });
    }
}
