/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.service.search;

import java.time.ZonedDateTime;
import java.util.List;
import java.util.Locale;
import java.util.Optional;
import java.util.function.Predicate;

import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.slf4j.Logger;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.core.util.DateUtils;
import org.ametys.odf.ProgramItem;
import org.ametys.odf.course.Course;
import org.ametys.odf.data.EducationalPath;
import org.ametys.odf.program.AbstractProgram;
import org.ametys.plugins.odfweb.repository.CoursePage;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.web.frontoffice.search.metamodel.ReturnableSaxer;
import org.ametys.web.frontoffice.search.metamodel.impl.PageReturnable;
import org.ametys.web.frontoffice.search.metamodel.impl.PageSaxer;
import org.ametys.web.frontoffice.search.requesttime.SearchComponentArguments;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;

/**
 * {@link ReturnableSaxer} for {@link CourseReturnable}
 * <br>
 * <br>Note for developpers: This Saxer extends PageSaxer to have access to its protected method #saxContent()
 */
public class CourseSaxer extends PageSaxer
{
    private CourseReturnable _courseReturnable;

    /**
     * Constructor
     * @param pageReturnable The course returnable (needed for superclass)
     * @param courseReturnable The associated returnable on courses
     */
    public CourseSaxer(PageReturnable pageReturnable, CourseReturnable courseReturnable)
    {
        super(pageReturnable);
        _courseReturnable = courseReturnable;
    }
    
    @Override
    public boolean canSax(AmetysObject hit, Logger logger, SearchComponentArguments args)
    {
        return hit instanceof Course;
    }
    
    @Override
    public void sax(ContentHandler contentHandler, AmetysObject hit, Logger logger, SearchComponentArguments args) throws SAXException
    {
        Course course = (Course) hit;
        _saxCourseContent(contentHandler, course, logger, args);
        
        // Get the first course page found
        Optional<CoursePage> coursePage = _resolveCoursePage(course, args.currentSite());
        if (coursePage.isPresent())
        {
            Page page = coursePage.get();
            XMLUtils.createElement(contentHandler, "uri", page.getSitemap().getName() + "/" + page.getPathInSitemap());
        }
        
        _saxCoursePages(contentHandler, course);
    }
    
    private Optional<CoursePage> _resolveCoursePage(Course course, Site currentSite)
    {
        return Optional.of(_courseReturnable._odfPageResolver)
                .map(res -> res.getCoursePage(course, (AbstractProgram) null, currentSite.getName()));
    }
    
    private void _saxCourseContent(ContentHandler contentHandler, Course course, Logger logger, SearchComponentArguments args) throws SAXException
    {
        XMLUtils.createElement(contentHandler, "title", course.getTitle());
        ZonedDateTime lastValidationDate = course.getLastValidationDate();
        if (lastValidationDate != null)
        {
            XMLUtils.createElement(contentHandler, "lastValidation", DateUtils.zonedDateTimeToString(lastValidationDate));
        }
        Locale locale = new Locale(args.currentPage().getSitemapName());
        saxContent(course, "index", locale, contentHandler, logger);
    }
    
    private void _saxCoursePages(ContentHandler contentHandler, Course course) throws SAXException
    {
        // ignore paths that is not part of a Program
        List<EducationalPath> educationalPaths = _courseReturnable._odfHelper.getEducationalPaths(course, true, true);
        
        // Filter on program items that is part of ODF pages
        Predicate<ProgramItem> filter = p -> p instanceof AbstractProgram || p instanceof Course;
        
        for (EducationalPath path : educationalPaths)
        {
            String uri = _courseReturnable._odfHelper.getEducationalPathAsString(path, ProgramItem::getId, ";", filter);
            
            XMLUtils.startElement(contentHandler, "coursePage");
            XMLUtils.createElement(contentHandler, "uri", uri);
            
            String resolvedUri = _courseReturnable._odfURIResolver.resolve(uri, false, false, false);
            XMLUtils.createElement(contentHandler, "resolvedUri", resolvedUri);
            
            // Sax course path with title
            List<ProgramItem> programItems = path.resolveProgramItems(_courseReturnable._resolver).filter(filter).toList();
            _saxCoursePath(contentHandler, programItems);
            
            XMLUtils.endElement(contentHandler, "coursePage");
        }
    }
    
    private void _saxCoursePath(ContentHandler contentHandler, List<ProgramItem> programItems) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "path");
        for (ProgramItem item : programItems)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", ((Content) item).getId());
            XMLUtils.createElement(contentHandler, "element", attrs, ((Content) item).getTitle());
        }
        XMLUtils.endElement(contentHandler, "path");
    }
}

