/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.odfweb.service.search;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.repository.Content;
import org.ametys.cms.search.SearchResult;
import org.ametys.cms.search.SearchResultsIterable;
import org.ametys.cms.search.SearchResultsIterator;
import org.ametys.cms.search.advanced.AbstractTreeNode;
import org.ametys.cms.search.advanced.AdvancedQueryBuilder;
import org.ametys.cms.search.content.ContentSearcherFactory;
import org.ametys.cms.search.query.Query;
import org.ametys.odf.program.Program;
import org.ametys.odf.program.SubProgram;
import org.ametys.odf.program.SubProgramFactory;
import org.ametys.plugins.odfweb.service.search.ProgramReturnable.DisplaySubprogramMode;
import org.ametys.web.frontoffice.search.instance.SearchServiceInstance;
import org.ametys.web.frontoffice.search.instance.model.FOSearchCriterion;
import org.ametys.web.frontoffice.search.instance.model.RightCheckingMode;
import org.ametys.web.frontoffice.search.metamodel.AdditionalParameterValueMap;
import org.ametys.web.frontoffice.search.metamodel.FacetDefinition;
import org.ametys.web.frontoffice.search.metamodel.Returnable;
import org.ametys.web.frontoffice.search.metamodel.SearchCriterionDefinition;
import org.ametys.web.frontoffice.search.metamodel.Searchable;
import org.ametys.web.frontoffice.search.requesttime.SearchComponent;
import org.ametys.web.frontoffice.search.requesttime.SearchComponentArguments;
import org.ametys.web.frontoffice.search.requesttime.impl.SearchComponentHelper;

/**
 * {@link SearchComponent} for getting subprograms matching search
 */
public class MatchingSubprogramSearchComponent implements SearchComponent, Serviceable
{
    private static final String __MATCHING_SUBPROGRAM_IDS_ATTR_NAME = MatchingSubprogramSearchComponent.class.getName() + "$matchingSubProgramIds";
    
    /** The content searcher factory */
    protected ContentSearcherFactory _contentSearcherfactory;

    /** The builder of advanced queries */
    protected AdvancedQueryBuilder _advancedQueryBuilder;
    
    /** The search component helper */
    protected SearchComponentHelper _searchComponentHelper;
    
    /** The ODF search helper */
    protected ODFSearchHelper _odfSearchHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _contentSearcherfactory = (ContentSearcherFactory) manager.lookup(ContentSearcherFactory.ROLE);
        _advancedQueryBuilder = (AdvancedQueryBuilder) manager.lookup(AdvancedQueryBuilder.ROLE);
        _searchComponentHelper = (SearchComponentHelper) manager.lookup(SearchComponentHelper.ROLE);
        _odfSearchHelper = (ODFSearchHelper) manager.lookup(ODFSearchHelper.ROLE);
    }

    @Override
    public int priority()
    {
        return SEARCH_PRIORITY - 9000;
    }

    @Override
    public boolean supports(SearchComponentArguments args)
    {
        SearchServiceInstance serviceInstance = args.serviceInstance();
        return  args.launchSearch() 
                && args.serviceInstance().getCriterionTree().isPresent()
                && _containsProgramReturnable(serviceInstance)
                && _needSubprogramDiscrimination(serviceInstance);
    }
    
    private boolean _containsProgramReturnable(SearchServiceInstance serviceInstance)
    {
        return serviceInstance.getReturnables()
                .stream()
                .map(Returnable::getId)
                .anyMatch(ProgramReturnable.ROLE::equals);
    }
    
    private boolean _needSubprogramDiscrimination(SearchServiceInstance serviceInstance)
    {
        AdditionalParameterValueMap additionalParameterValues = serviceInstance.getAdditionalParameterValues();
        DisplaySubprogramMode mode = ProgramReturnable.getDisplaySubprogramMode(additionalParameterValues);
        return mode == DisplaySubprogramMode.ALL_WITH_HIGHLIGHT || mode == DisplaySubprogramMode.MATCHING_SEARCH_ONLY;
    }
    
    @Override
    public void execute(SearchComponentArguments args) throws Exception
    {
        SearchServiceInstance serviceInstance = args.serviceInstance();

        List<String> matchingSubProgramIds = retrieveSubProgramsMatchingSearch(
                                                    serviceInstance, 
                                                    args.userInputs().criteria(),
                                                    args.serviceInstance().getFacets(),
                                                    args.userInputs().facets(),
                                                    args.currentLang());
        _storeInReqMatchingSubProgramIds(args.request(), matchingSubProgramIds);
    }
    
    /**
     * Retrieves, thanks to a (independent but close to the main one) Solr search, the {@link SubProgram}s that match the same query than the {@link Program}s, for further special display.
     * @param serviceInstance The service instance
     * @param userCriteria The user criteria
     * @param serviceFacets the service facets
     * @param userFacets the user facets
     * @param lang The current lang
     * @return the {@link SubProgram}s that match the same query than the {@link Program}s
     * @throws Exception If an exception occured during the Solr search
     */
    protected List<String> retrieveSubProgramsMatchingSearch(SearchServiceInstance serviceInstance, Map<String, Object> userCriteria, Collection<FacetDefinition> serviceFacets, Map<String, List<String>> userFacets, String lang) throws Exception
    {
        AbstractTreeNode<FOSearchCriterion> criterionTree = serviceInstance.getCriterionTree().get();
        
        Collection<Returnable> returnables = serviceInstance.getReturnables();
        Collection<Searchable> searchables = serviceInstance.getSearchables();
        AdditionalParameterValueMap additionalParameters = serviceInstance.getAdditionalParameterValues();
        Map<String, Object> contextualParameters = new HashMap<>();
        Query criterionTreeQuery = buildQuery(criterionTree, userCriteria, returnables, searchables, additionalParameters, lang, contextualParameters);
        
        List<String> facets = serviceFacets.stream()
            .filter(this::_mustKeepFacet)
            .map(FacetDefinition::getId)
            .map(id -> _odfSearchHelper.getPathFromSearchFieldId(id))
            .collect(Collectors.toList());

        Map<String, List<String>> computedUserFacets = userFacets.entrySet()
            .stream()
            .collect(Collectors.toMap(e -> _odfSearchHelper.getPathFromSearchFieldId(e.getKey()), Map.Entry::getValue));

        boolean checkRights;
        RightCheckingMode rightCheckingMode = serviceInstance.getRightCheckingMode();
        switch (rightCheckingMode)
        {
            case EXACT:
            case FAST:
                checkRights = true;
                break;
            case NONE:
                checkRights = false;
                break;
            default:
                throw new IllegalStateException("Unhandled right checking mode: " + rightCheckingMode);
        }

        SearchResultsIterable<SearchResult<Content>> results = _contentSearcherfactory.create(SubProgramFactory.SUBPROGRAM_CONTENT_TYPE)
                    .withLimits(0, Integer.MAX_VALUE)
                    .setCheckRights(checkRights)
                    .withFacets(facets)
                    // TODO possible optimization: here we don't care about the order of returned subprograms, so avoid computation of score (sort=_docid_ asc)
                    .searchWithFacets(criterionTreeQuery, computedUserFacets)
                    .getResults();
        
        List<String> contentIds = new ArrayList<>();
        SearchResultsIterator<SearchResult<Content>> resultsIterator = results.iterator();
        while (resultsIterator.hasNext())
        {
            SearchResult<Content> content = resultsIterator.next();
            String id = content.getObject().getId();
            contentIds.add(id);
        }
        
        return contentIds;
    }
    
    /**
     * Builds the query of the criterion tree
     * @param criterionTree The criterion tree of the service instance
     * @param userCriteria The user input criteria
     * @param returnables The returnables of the service instance
     * @param searchables The searchables of the service instance
     * @param additionalParameters The values of additional parameters of the service instance
     * @param currentLang The current lang
     * @param contextualParameters the search contextual parameters. 
     * @return The query of the criterion tree
     */
    protected Query buildQuery(
            AbstractTreeNode<FOSearchCriterion> criterionTree, 
            Map<String, Object> userCriteria, 
            Collection<Returnable> returnables, 
            Collection<Searchable> searchables, 
            AdditionalParameterValueMap additionalParameters, 
            String currentLang, 
            Map<String, Object> contextualParameters)
    {
        // Sanitize additional params => we do not want ProgramSearchable to generate a Join on subprograms as we are already searching subprograms here !
        AdditionalParameterValueMap modifiedAdditionalParameters = new NoSubprogramSearchAdditionalParameterValueMap(additionalParameters);
        
        Function<FOSearchCriterion, Query> queryMapper = crit -> singleCriterionToQuery(crit, userCriteria, returnables, searchables, modifiedAdditionalParameters, currentLang, contextualParameters);
        return _advancedQueryBuilder.build(criterionTree, queryMapper);
    }
    
    /**
     * Builds the query of the single criterion
     * @param searchCriterion The criterion
     * @param userCriteria The user input criteria
     * @param returnables The returnables of the service instance
     * @param searchables The searchables of the service instance
     * @param additionalParameters The values of additional parameters of the service instance
     * @param currentLang The current lang
     * @param contextualParameters the search contextual parameters. 
     * @return The query of the single criterion
     */
    protected Query singleCriterionToQuery(
            FOSearchCriterion searchCriterion, 
            Map<String, Object> userCriteria, 
            Collection<Returnable> returnables, 
            Collection<Searchable> searchables, 
            AdditionalParameterValueMap additionalParameters, 
            String currentLang,
            Map<String, Object> contextualParameters)
    {
        // Delegate to the function defined in CriterionTreeSearchComponent
        Predicate<FOSearchCriterion> subProgramFilterCriterionPredicate = crit -> _mustKeepCriterion(crit);
        return _searchComponentHelper.singleCriterionToQuery(searchCriterion, userCriteria, returnables, searchables, additionalParameters, currentLang, subProgramFilterCriterionPredicate, contextualParameters);
    }
    
    private boolean _mustKeepCriterion(FOSearchCriterion criterion)
    {
        SearchCriterionDefinition criterionDefinition = criterion.getCriterionDefinition();
        
        // We want to keep only wording criteria or common criteria between programs and subprograms: the Query is meant to return subprograms and not programs
        return _odfSearchHelper.isCriterionOnBothProgramAndSubProgram(criterionDefinition);
    }
    
    private boolean _mustKeepFacet(FacetDefinition facet)
    {
        // We want to keep common facet between programs and subprograms: the Query is meant to return subprograms and not programs
        return _odfSearchHelper.isFacetOnBothProgramAndSubProgram(facet);
    }
    
    private static void _storeInReqMatchingSubProgramIds(Request request, List<String> subPrograms)
    {
        request.setAttribute(__MATCHING_SUBPROGRAM_IDS_ATTR_NAME, subPrograms);
    }
    
    @SuppressWarnings("unchecked")
    static List<String> _getMatchingSubProgramIds(Request request)
    {
        List<String> ids = (List<String>) request.getAttribute(__MATCHING_SUBPROGRAM_IDS_ATTR_NAME);
        return ids == null ? Collections.emptyList() : ids;
    }
    
    // AdditionalParameterValueMap, which will always answer false to ProgramSearchable.PARAMETER_SEARCH_ON_SUBPROGRAMS
    private static class NoSubprogramSearchAdditionalParameterValueMap extends AdditionalParameterValueMap
    {
        private AdditionalParameterValueMap _ref;
        NoSubprogramSearchAdditionalParameterValueMap(AdditionalParameterValueMap ref)
        {
            super(Collections.emptyMap(), Collections.emptySet());
            _ref = ref;
        }
        
        @SuppressWarnings("unchecked")
        @Override
        public <T> T getValue(String parameterId) throws ClassCastException
        {
            return ProgramSearchable.PARAMETER_SEARCH_ON_SUBPROGRAMS.equals(parameterId)
                    ? (T) Boolean.FALSE
                    : _ref.getValue(parameterId);
        }
    }
}
